# Schedwi
# Copyright (C) 2013 Herve Quatremain
#
# This file is part of Schedwi.
#
# Schedwi is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 3 of the License, or
# (at your option) any later version.
#
# Schedwi is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

"""Popup window to force the status of a job or jobset."""

from muntjac.api import (Window, OptionGroup, HorizontalLayout, Button,
                         Alignment)
from muntjac.ui.button import IClickListener
from muntjac.ui.window import Notification

from simple_queries_job import sql_get_job
import status_utils


class ForceWindow(Window, IClickListener):

    """Select the status to force for a job/jobset."""

    _bt_captions = [_('Cancel'), _('OK')]

    def __init__(self, refresh_obj, job_id, application_window,
                 sql_session, workload):
        """Create and display the status selection window.


        @raise sqlalchemy.orm.exc.NoResultFound:
                the given job ID is not in the database.
        """
        super(ForceWindow, self).__init__()
        self._refresh_obj = refresh_obj
        self._application_window = application_window
        self._sql_session = sql_session
        self._workload = workload
        self._job_id = job_id

        job = sql_get_job(self._sql_session, self._job_id, self._workload)
        self.setCaption(_('Force status of %s (ID: %d)') %
                        (job.name.encode('utf-8'), job_id))

        # VerticalLayout as content by default
        v = self.getContent()
        v.setSizeFull()
        v.setMargin(True)
        v.setSpacing(True)

        status_order = [status_utils.COMPLETED, status_utils.FAILED,
                        status_utils.WAITING, status_utils.RUNNING]
        status_name = [status_utils.status2string(s) for s in status_order]
        state_select = OptionGroup(_('New status'), status_name)
        state_select.setNullSelectionAllowed(False)
        state_select.select(status_name[0])
        self._state_select = state_select
        self._status_name = status_name
        self._status_order = status_order
        v.addComponent(state_select)

        # Button box
        h_bt = HorizontalLayout()
        h_bt.setMargin(False)
        h_bt.setSpacing(True)

        for caption in self._bt_captions:
            b = Button(_(caption))
            b.addListener(self, IClickListener)
            h_bt.addComponent(b)

        v.addComponent(h_bt)
        v.setComponentAlignment(h_bt, Alignment.BOTTOM_RIGHT)

        application_window.addWindow(self)
        self.center()

    def buttonClick(self, event):
        # First button is Cancel
        if event.getButton().getCaption() != _(self._bt_captions[0]):
            st = self._status_order[self._status_name.index(
                self._state_select.getValue())]
            try:
                status_utils.set_status(self._sql_session, self._job_id,
                                        self._workload, st)
            except:
                self._application_window.showNotification(
                    _("Cannot get the edited job/jobset details"),
                    _("<br/>Maybe someone else just removed it."),
                    Notification.TYPE_ERROR_MESSAGE)
            self._refresh_obj.repaint()

        # Close the window
        self._application_window.removeWindow(self)
