# Schedwi
# Copyright (C) 2012, 2013 Herve Quatremain
#
# This file is part of Schedwi.
#
# Schedwi is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 3 of the License, or
# (at your option) any later version.
#
# Schedwi is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

"""Job list table UI."""

import time
from sqlalchemy import desc

from muntjac.ui.embedded import Embedded
from muntjac.api import (Table, VerticalLayout, HorizontalLayout, Label,
                         NativeButton, Alignment, ProgressIndicator)
from muntjac.data.util.indexed_container import IndexedContainer
from muntjac.data.property import IValueChangeListener
from muntjac.ui.table import IHeaderClickListener
from muntjac.ui.button import IClickListener
from muntjac.ui.window import Notification
from muntjac.terminal.theme_resource import ThemeResource
from muntjac.event.item_click_event import IItemClickListener

from tables.job_main import job_main
from tables.job_main_s import job_main_s
from tables.job_host import job_host
from tables.job_host_s import job_host_s
import sql_hierarchy
import path_cal
from web.cutcopypastemenu import (CutCopyPasteMenuJobTable, DeleteShortcut,
                                  CutCopyPasteMenuListener)
from web.autorefresh import AutoRefresh
import status_utils
import host_utils
import cluster_utils
from simple_queries_stat import sql_get_stat_ratio_completion
from simple_queries_ack import (sql_get_acknowledge_request_by_job_id,
                                sql_update_acknowledge_request)


class JobTable(VerticalLayout, IValueChangeListener, IHeaderClickListener,
               CutCopyPasteMenuListener):

    """List of job and jobset UI."""

    _ROOT_JOBSET_ID = 1
    _ICON_JOB = ThemeResource('icons/job.png')
    _ICON_JOBSET = ThemeResource('icons/jobset.png')
    _ICON_ENABLED = ThemeResource('icons/check.png')
    _ICON_STATUS = {
        status_utils.WAITING: ThemeResource(
            status_utils.status2icon(status_utils.WAITING)),
        status_utils.RUNNING: ThemeResource(
            status_utils.status2icon(status_utils.RUNNING)),
        status_utils.COMPLETED: ThemeResource(
            status_utils.status2icon(status_utils.COMPLETED)),
        status_utils.FAILED: ThemeResource(
            status_utils.status2icon(status_utils.FAILED))
        }

    def __init__(self, sql_session, workload=None):
        """Create the UI.

        @param sql_session:
                    SQLAlchemy session.
        @param workload:
                    workload to consider.
        """
        super(JobTable, self).__init__()
        self._sql_session = sql_session
        self._workload = workload

        self._jobset_ids = None
        self._selected = None
        self._sortCol = None
        self._sortOrd = True  # Ascending
        self._listeners = list()
        self._repaint_listeners = list()
        self._select_listeners = list()

        # Toolbar
        btbox = HorizontalLayout()
        btbox.addStyleName("toolbar")
        btbox.setSpacing(False)
        self.addComponent(btbox)

        b = NativeButton()
        b.setIcon(ThemeResource('icons/edit.png'))
        b.setDescription(_("Edit the selected job or jobset"))
        b.setStyleName('toolbutton')
        b.setEnabled(False)
        b.addListener(EditBtHandler(self), IClickListener)
        btbox.addComponent(b)
        self._btedit = b

        b = Label()  # Separator
        b.setIcon(ThemeResource('icons/sep.png'))
        btbox.addComponent(b)
        btbox.setComponentAlignment(b, Alignment.MIDDLE_CENTER)

        if workload is None:
            b = NativeButton()
            b.setIcon(ThemeResource('icons/jobset-new.png'))
            b.setDescription(_("Create a new Jobset"))
            b.setStyleName('toolbutton')
            b.setEnabled(False)
            b.addListener(NewJobsetBtHandler(self), IClickListener)
            btbox.addComponent(b)
            self._btaddjobset = b

            b = NativeButton()
            b.setIcon(ThemeResource('icons/job-new.png'))
            b.setDescription(_("Create a new Job"))
            b.setStyleName('toolbutton')
            b.setEnabled(False)
            b.addListener(NewJobBtHandler(self), IClickListener)
            btbox.addComponent(b)
            self._btaddjob = b

            b = NativeButton()
            b.setIcon(ThemeResource('icons/delete.png'))
            b.setDescription(_("Delete the selected job or jobset"))
            b.setStyleName('toolbutton')
            b.setEnabled(False)
            b.addListener(DelBtHandler(self), IClickListener)
            btbox.addComponent(b)
            self._btdel = b

            b = Label()  # Separator
            b.setIcon(ThemeResource('icons/sep.png'))
            btbox.addComponent(b)
            btbox.setComponentAlignment(b, Alignment.MIDDLE_CENTER)

            b = NativeButton()
            b.setIcon(ThemeResource('icons/cut.png'))
            b.setDescription(_("Cut the selected job or jobset"))
            b.setStyleName('toolbutton')
            b.setEnabled(False)
            b.addListener(CutBtHandler(self), IClickListener)
            btbox.addComponent(b)
            self._btcut = b

            b = NativeButton()
            b.setIcon(ThemeResource('icons/copy.png'))
            b.setDescription(_("Mark the selected job or jobset for copy"))
            b.setStyleName('toolbutton')
            b.setEnabled(False)
            b.addListener(CopyBtHandler(self), IClickListener)
            btbox.addComponent(b)
            self._btcopy = b

            b = NativeButton()
            b.setIcon(ThemeResource('icons/paste.png'))
            b.setDescription(
                _("Paste the previously copied or cut job or jobset"))
            b.setStyleName('toolbutton')
            b.setEnabled(False)
            b.addListener(PasteBtHandler(self), IClickListener)
            btbox.addComponent(b)
            self._btpaste = b
        else:
            b = NativeButton()
            b.setIcon(ThemeResource('icons/start.png'))
            b.setDescription(_("Force Start"))
            b.setStyleName('toolbutton')
            b.setEnabled(False)
            b.addListener(ForceStartBtHandler(self), IClickListener)
            btbox.addComponent(b)
            self._btstart = b

            b = NativeButton()
            b.setIcon(ThemeResource('icons/stop.png'))
            b.setDescription(_("Stop"))
            b.setStyleName('toolbutton')
            b.setEnabled(False)
            b.addListener(StopBtHandler(self), IClickListener)
            btbox.addComponent(b)
            self._btstop = b

            b = NativeButton()
            b.setIcon(ThemeResource('icons/trigger.png'))
            b.setDescription(_("Trigger manual start"))
            b.setStyleName('toolbutton')
            b.setEnabled(False)
            b.addListener(TriggerBtHandler(self), IClickListener)
            btbox.addComponent(b)
            self._bttrigger = b

            b = NativeButton()
            b.setIcon(ThemeResource('icons/force.png'))
            b.setDescription(_("Force the status"))
            b.setStyleName('toolbutton')
            b.setEnabled(False)
            b.addListener(ForceBtHandler(self), IClickListener)
            btbox.addComponent(b)
            self._btforce = b

        # Job/jobset table
        self._table = Table()
        self._table.setSizeFull()
        self._table.setSelectable(True)
        self._table.setMultiSelect(False)
        self._table.setImmediate(True)
        self._table.setColumnReorderingAllowed(False)
        self._table.setColumnCollapsingAllowed(False)
        self._table.addStyleName("jobtable")
        self.addComponent(self._table)
        self.setExpandRatio(self._table, 1.0)

        # Container
        self._table.setContainerDataSource(self._new_container())

        # Columns
        if workload is None:
            self._table.setVisibleColumns(['type',
                                           'name',
                                           'enabled',
                                           'host',
                                           'calendar',
                                           'start'])
            self._table.setColumnHeaders([_('Type'),
                                          _('Name'),
                                          _('Enabled'),
                                          _('Host/Cluster'),
                                          _('Calendar'),
                                          _('Start Time')])
        else:
            self._table.setVisibleColumns(['type',
                                           'status',
                                           'name',
                                           'progress',
                                           'enabled',
                                           'host',
                                           'calendar',
                                           'start',
                                           'message'])
            self._table.setColumnHeaders([_('Type'),
                                          _("Status"),
                                          _('Name'),
                                          _("Progress"),
                                          _('Enabled'),
                                          _('Host/Cluster'),
                                          _('Calendar'),
                                          _('Start Time'),
                                          _("Message")])
            self._table.setColumnWidth('status', 45)
            self._table.setColumnAlignment('status', Table.ALIGN_CENTER)
        self._table.setColumnWidth('type', 30)
        self._table.setColumnAlignment('type', Table.ALIGN_CENTER)
        self._table.setColumnWidth('enabled', 45)
        self._table.setColumnAlignment('enabled', Table.ALIGN_CENTER)

        # Callbacks
        self._table.addListener(self, IValueChangeListener)
        self._table.addListener(self, IHeaderClickListener)
        self._table.addListener(DoubleClick(self), IItemClickListener)
        self._popup_menu = CutCopyPasteMenuJobTable(self)
        self._table.addActionHandler(self._popup_menu)
        if workload is None:
            # Keyboard shortcuts
            self._table.addShortcutListener(DeleteShortcut(self))

    def _new_container(self):
        """Create and return a new container data source.

        @return:    the new empty container data source.
        @rtype:
            L{muntjac.data.util.indexed_container.IndexedContainer}
        """
        jContainer = IndexedContainer()
        jContainer.addContainerProperty('type', Embedded, None)
        jContainer.addContainerProperty('name', str, None)
        jContainer.addContainerProperty('enabled', Embedded, None)
        jContainer.addContainerProperty('host', str, None)
        jContainer.addContainerProperty('calendar', str, None)
        jContainer.addContainerProperty('start', str, None)
        # type_job: 0 --> jobset  and   1 --> job
        jContainer.addContainerProperty('type_job', int, None)
        if self._workload is not None:
            jContainer.addContainerProperty('status', Embedded, None)
            jContainer.addContainerProperty('message', str, None)
            jContainer.addContainerProperty('progress', ProgressIndicator,
                                            None)
        return jContainer

    def add_listener(self, obj):
        """Add a new listener.

        @param obj:
                the listener object (must implement L{JobTableListener},
                L{JobTableRepaint} or L{JobTableSelect})
        """
        if isinstance(obj, JobTableListener):
            self._listeners.append(obj)
        elif isinstance(obj, JobTableRepaint):
            self._repaint_listeners.append(obj)
        elif isinstance(obj, JobTableSelect):
            self._select_listeners.append(obj)

    def fire_listeners(self, selected_job_id, jobset_ids):
        """Call the listerners previously added with L{add_listener}.

        @param selected_job_id:
                    container/database ID of the selected job or jobset.
        @param jobset_ids:
                    the list of all the jobset IDs from the root jobset up to
                    the parent jobset of the selected object.
        """
        map(lambda obj: obj.refresh(selected_job_id, jobset_ids),
            self._listeners)

    def get_datasource(self):
        """Return the container data source used by the job table.

        @return:    the container data source.
        @rtype:
            L{muntjac.data.util.indexed_container.IndexedContainer}
        """
        return self._table.getContainerDataSource()

    def toolbar_set_state(self):
        """Set the state of the toolbar buttons."""
        if self._selected is not None:
            self._btedit.setEnabled(True)
            if self._workload is None:
                self._btdel.setEnabled(True)
                self._btcut.setEnabled(True)
                self._btcopy.setEnabled(True)
            else:
                session = self._sql_session.open_session()
                s = status_utils.get_status(session, self._selected,
                                            self._workload)
                if s.status != status_utils.RUNNING:
                    self._btstart.setEnabled(True)
                    self._btstop.setEnabled(False)
                else:
                    self._btstart.setEnabled(False)
                    self._btstop.setEnabled(True)
                self._btforce.setEnabled(True)
                if s.status == status_utils.WAITING:
                    try:
                        ack = sql_get_acknowledge_request_by_job_id(
                            session, self._selected, self._workload)
                    except:
                        self._bttrigger.setEnabled(False)
                    else:
                        if ack.status == 0:
                            self._bttrigger.setEnabled(True)
                        else:
                            self._bttrigger.setEnabled(False)
                else:
                    self._bttrigger.setEnabled(False)
                self._sql_session.close_session(session)
        else:
            self._btedit.setEnabled(False)
            if self._workload is None:
                self._btdel.setEnabled(False)
                self._btcut.setEnabled(False)
                self._btcopy.setEnabled(False)
            else:
                self._btstart.setEnabled(False)
                self._btstop.setEnabled(False)
                self._bttrigger.setEnabled(False)
                self._btforce.setEnabled(False)
        if self._workload is None:
            if self._popup_menu.is_cut_or_copy():
                self._btpaste.setEnabled(True)
            else:
                self._btpaste.setEnabled(False)
            self._btaddjobset.setEnabled(True)
            self._btaddjob.setEnabled(True)

    def headerClick(self, event):
        """Callback when the hearder of a column is clicked."""
        AutoRefresh.reset()
        col = event.getPropertyId()
        if self._sortCol == col:
            self._sortOrd = not self._sortOrd
        else:
            self._sortCol = col
        self.display_jobset_content(self._jobset_ids)

    def valueChange(self, event):
        """Callback when a jobset is selected or unselected."""
        AutoRefresh.reset()
        selected_job_id = event.getProperty().getValue()
        if selected_job_id is not None:
            self._selected = selected_job_id
        else:
            self._selected = None
        self.fire_listeners(selected_job_id, self._jobset_ids)
        self.toolbar_set_state()

    def get_selected(self):
        """Return the container/database ID of the selected job or jobset."""
        return self._selected

    def display_jobset_content(self, jobset_ids):
        """Refresh (reload) the view.

        @param jobset_ids:
                    the list of all the jobset IDs from the root jobset up to
                    the jobset for which the content must be displayed.
        """
        if jobset_ids is None:
            return
        self._jobset_ids = list(jobset_ids)
        jobset_id = jobset_ids[-1]

        # Get all the children of the jobset to display
        session = self._sql_session.open_session()
        if self._workload is None:
            query = session.query(job_main)
            query = query.filter(job_main.parent == jobset_id)
            if not self._sortCol or self._sortCol == 'type':
                if self._sortOrd:
                    query = query.order_by(job_main.type)
                else:
                    query = query.order_by(desc(job_main.type))
            elif self._sortCol == 'enabled':
                if self._sortOrd:
                    query = query.order_by(job_main.enabled)
                else:
                    query = query.order_by(desc(job_main.enabled))
            if self._sortOrd:
                query = query.order_by(job_main.name)
            else:
                query = query.order_by(desc(job_main.name))
        else:
            query = session.query(job_main_s)
            query = query.filter(job_main_s.workload_date == self._workload)
            query = query.filter(job_main_s.parent == jobset_id)
            if not self._sortCol or self._sortCol == 'type':
                if self._sortOrd:
                    query = query.order_by(job_main_s.type)
                else:
                    query = query.order_by(desc(job_main_s.type))
            elif self._sortCol == 'enabled':
                if self._sortOrd:
                    query = query.order_by(job_main_s.enabled)
                else:
                    query = query.order_by(desc(job_main_s.enabled))
            if self._sortOrd:
                query = query.order_by(job_main_s.name)
            else:
                query = query.order_by(desc(job_main_s.name))

        # Create and fill a new container
        c = self._new_container()
        for job in query.all():
            jobset_ids.append(job.id)
            item = c.addItem(job.id)
            item.getItemProperty('name').setValue(job.name.encode('utf-8'))
            item.getItemProperty('type_job').setValue(job.type)
            # Type: 0 --> jobset   1 --> Job
            if job.type == 0:
                e = Embedded(source=self._ICON_JOBSET)
                item.getItemProperty('type').setValue(e)
            else:
                e = Embedded(source=self._ICON_JOB)
                item.getItemProperty('type').setValue(e)
            # Enabled
            if job.enabled:
                e = Embedded(source=self._ICON_ENABLED)
                item.getItemProperty('enabled').setValue(e)
            # Host/Cluster
            # If the job is running, then display the host on which it is
            # running rather than the cluster/host defined for this job.
            host = None
            if self._workload is not None:
                st = status_utils.get_status(session, job.id, self._workload)
                if st.status == status_utils.RUNNING and st.host_id:
                    try:
                        host = host_utils.get_host_by_id(session, st.host_id)
                    except:
                        host = None
            if host is None:
                in_object, obj = sql_hierarchy.get_cluster_or_host_by_id(
                    session, jobset_ids, self._workload)
                if isinstance(obj, job_host) or isinstance(obj, job_host_s):
                    host = host_utils.get_host_by_id(session, obj.host_id)
                else:
                    cluster = cluster_utils.get_cluster_by_id(session,
                                                              obj.cluster_id)
                    item.getItemProperty('host').setValue(
                        _('%s (cluster)') % cluster.name.encode('utf-8'))
            if host is not None:
                s = '%s (SSL)' if host.sslenable else '%s'
                item.getItemProperty('host').setValue(s % str(host))
            jobset_ids.pop()
            # Calendar
            if job.cal_id == 0:
                # Retrieve the calendar from a parent jobset (inherited)
                in_object, obj = sql_hierarchy.get_calendar_id(session,
                                                               jobset_ids,
                                                               self._workload)
                if obj is not None:
                    s = path_cal.PathCal(session, id=obj.cal_id,
                                         workload=self._workload)
                    item.getItemProperty('calendar').setValue(
                        unicode(s).encode('utf-8'))
                else:
                    item.getItemProperty('calendar').setValue(_('Everyday'))
            else:
                s = path_cal.PathCal(session, id=job.cal_id,
                                     workload=self._workload)
                item.getItemProperty('calendar').setValue(
                    unicode(s).encode('utf-8'))
            # Start time
            hour_str = None
            if self._workload is not None:
                status = status_utils.get_status(session, job.id,
                                                 self._workload)
                if status.status != status_utils.WAITING:
                    if status.error_msg:
                        item.getItemProperty('message').setValue(
                            status.error_msg.encode('utf-8'))
                    # Display the status change time
                    st = time.localtime(status.start_time)
                    hour_str = "%02dh%02d" % (st.tm_hour, st.tm_min)
                else:
                    item.getItemProperty('message').setValue(
                        status.get_waiting_message().encode('utf-8'))
                e = Embedded(source=self._ICON_STATUS[status.status])
                e.setDescription(status_utils.status2string(status.status))
                item.getItemProperty('status').setValue(e)
            if hour_str is None:
                if job.start_time < 0:
                    # Retrieve the start time from a parent jobset (inherited)
                    in_object, obj = sql_hierarchy.get_start_time_id(
                        session, jobset_ids, self._workload)
                    if obj is not None:
                        start_time = obj.start_time
                        hour_str = "%02dh%02d" % (start_time / 100,
                                                  start_time % 100)
                    else:
                        hour_str = '00h00'
                else:
                    hour_str = "%02dh%02d" % (job.start_time / 100,
                                              job.start_time % 100)
            item.getItemProperty('start').setValue(hour_str)
            # Progress status
            if self._workload is not None:
                r = sql_get_stat_ratio_completion(session, job.id,
                                                  self._workload)
                if r is not None:
                    p = ProgressIndicator(r)
                    p.setWidth('100%')
                    p.setPollingInterval(1000000000)  # No polling
                    item.getItemProperty('progress').setValue(p)
        self._sql_session.close_session(session)

        # Only ask the container to sort `text' columns.  The other columns
        # have already been sorted in the SQL select request above.
        if self._sortCol not in ['type', 'enabled']:
            c.sort([self._sortCol], [self._sortOrd])

        selection = self._selected
        self._table.setContainerDataSource(c)
        if selection in c.getItemIds():
            self._table.select(selection)
        if self._workload is None:
            self._table.setVisibleColumns(['type', 'name',
                                           'enabled', 'host',
                                           'calendar', 'start'])
        else:
            self._table.setVisibleColumns(['type', 'status', 'name',
                                           'progress',
                                           'enabled', 'host',
                                           'calendar', 'start', 'message'])
        self.toolbar_set_state()

    def get_jobset_ids(self):
        return list(self._jobset_ids)

    def get_parent_id(self):
        """From the L{web.cutcopypastemenu.CutCopyPasteMenuListener} class.

        @return:    the database ID of the parent jobset.
        """
        return self._jobset_ids[-1]

    def get_jobset_hierarchy(self, item_id):
        """From the L{web.cutcopypastemenu.CutCopyPasteMenuListener} class.

        @return:    the hierarchy of database ID of the parent jobsets.
        """
        return list(self._jobset_ids)

    def get_name(self, item_id):
        """From the L{web.cutcopypastemenu.CutCopyPasteMenuListener} class.

        @return:    the name of the provided item ID.
        """
        c = self.get_datasource()
        return c.getItem(item_id).getItemProperty('name').getValue()

    def repaint(self):
        """From the L{web.cutcopypastemenu.CutCopyPasteMenuListener} class.
        Refresh the view.
        """
        map(lambda obj: obj.repaint(),
            self._repaint_listeners)

    def repaint_tools(self):
        """From the L{web.cutcopypastemenu.CutCopyPasteMenuListener} class.
        Repaint the toolbar.
        """
        self.toolbar_set_state()

    def is_jobset(self, item_id):
        """From the L{web.cutcopypastemenu.CutCopyPasteMenuListener} class.

        @return:    True if the item is a jobset else False.
        """
        c = self.get_datasource()
        if c.getItem(item_id).getItemProperty('type_job').getValue():
            return False
        return True


class JobTableListener(object):

    """Interface for listening for a new job selection."""

    def refresh(self, selected_job_id, jobset_hierarchy):
        """Called when a job has been selected.

        @param selected_job_id:
                    database ID of the selected job or jobset. None means
                    nothing is selected.
        @param jobset_hierarchy:
                    the jobset hierarchy.  This is a list of all the parent
                    jobsets up to the root jobset.  The first item of this
                    list is the root jobset and the last is the parent
                    of the currently selected job/jobset.
        @type jobset_hierarchy: list
        """
        raise NotImplementedError


class JobTableRepaint(object):

    """Interface for repainting (refreshing) the view."""

    def repaint(self):
        """Called when a repaint request has been sent.  This is used to
        also refresh the jobset tree that may have been changed.
        """
        raise NotImplementedError


class JobTableSelect(object):

    """Interface for selectecting/displaying an other jobset."""

    def select(self, jobset_hierarchy):
        """Called when an other jobset must be displayed.

        @param jobset_hierarchy:
                    the jobset hierarchy.  This is a list of all the parent
                    jobsets up to the root jobset.  The first item of this
                    list is the root jobset and the last is the jobset to
                    display.
        @type jobset_hierarchy: list
        """
        raise NotImplementedError


class EditBtHandler(IClickListener):

    """Callback for the Edit button."""

    def __init__(self, job_table_obj):
        """Initialize the callback.

        @param job_table_obj:
                    the associated L{JobTable} object.
        @type job_table_obj:    L{JobTable}
        """
        super(EditBtHandler, self).__init__()
        self._c = job_table_obj

    def buttonClick(self, event):
        AutoRefresh.reset()
        self._c._popup_menu.do_edit(self._c.get_selected())


class NewJobsetBtHandler(IClickListener):

    """Callback for the new Jobset button."""

    def __init__(self, job_table_obj):
        """Initialize the callback.

        @param job_table_obj:
                    the associated L{JobTable} object.
        @type job_table_obj:    L{JobTable}
        """
        super(NewJobsetBtHandler, self).__init__()
        self._c = job_table_obj

    def buttonClick(self, event):
        AutoRefresh.reset()
        self._c._popup_menu.do_new_jobset(self._c.get_parent_id())


class NewJobBtHandler(IClickListener):

    """Callback for the new Job button."""

    def __init__(self, job_table_obj):
        """Initialize the callback.

        @param job_table_obj:
                    the associated L{JobTable} object.
        @type job_table_obj:    L{JobTable}
        """
        super(NewJobBtHandler, self).__init__()
        self._c = job_table_obj

    def buttonClick(self, event):
        AutoRefresh.reset()
        self._c._popup_menu.do_new_job(self._c.get_parent_id())


class DelBtHandler(IClickListener):

    """Callback for the Delete button."""

    def __init__(self, job_table_obj):
        """Initialize the callback.

        @param job_table_obj:
                    the associated L{JobTable} object.
        @type job_table_obj:    L{JobTable}
        """
        super(DelBtHandler, self).__init__()
        self._c = job_table_obj

    def buttonClick(self, event):
        AutoRefresh.reset()
        self._c._popup_menu.do_delete(self._c.get_selected())


class CutBtHandler(IClickListener):

    """Callback for the Cut button."""

    def __init__(self, job_table_obj):
        """Initialize the callback.

        @param job_table_obj:
                    the associated L{JobTable} object.
        @type job_table_obj:    L{JobTable}
        """
        super(CutBtHandler, self).__init__()
        self._c = job_table_obj

    def buttonClick(self, event):
        AutoRefresh.reset()
        self._c._popup_menu.do_cut(self._c.get_selected())


class CopyBtHandler(IClickListener):

    """Callback for the Copy button."""

    def __init__(self, job_table_obj):
        """Initialize the callback.

        @param job_table_obj:
                    the associated L{JobTable} object.
        @type job_table_obj:    L{JobTable}
        """
        super(CopyBtHandler, self).__init__()
        self._c = job_table_obj

    def buttonClick(self, event):
        AutoRefresh.reset()
        self._c._popup_menu.do_copy(self._c.get_selected())


class PasteBtHandler(IClickListener):

    """Callback for the Paste button."""

    def __init__(self, job_table_obj):
        """Initialize the callback.

        @param job_table_obj:
                    the associated L{JobTable} object.
        @type job_table_obj:    L{JobTable}
        """
        super(PasteBtHandler, self).__init__()
        self._c = job_table_obj

    def buttonClick(self, event):
        AutoRefresh.reset()
        self._c._popup_menu.do_paste(self._c.get_parent_id())


class ForceStartBtHandler(IClickListener):

    """Callback for the Force Start button."""

    def __init__(self, job_table_obj):
        """Initialize the callback.

        @param job_table_obj:
                    the associated L{JobTable} object.
        @type job_table_obj:    L{JobTable}
        """
        super(ForceStartBtHandler, self).__init__()
        self._c = job_table_obj

    def buttonClick(self, event):
        AutoRefresh.reset()
        self._c._popup_menu.do_start(self._c.get_selected())


class StopBtHandler(IClickListener):

    """Callback for the Stop button."""

    def __init__(self, job_table_obj):
        """Initialize the callback.

        @param job_table_obj:
                    the associated L{JobTable} object.
        @type job_table_obj:    L{JobTable}
        """
        super(StopBtHandler, self).__init__()
        self._c = job_table_obj

    def buttonClick(self, event):
        AutoRefresh.reset()
        self._c._popup_menu.do_stop(self._c.get_selected())


class TriggerBtHandler(IClickListener):

    """Callback for the Trigger start manual job button."""

    def __init__(self, job_table_obj):
        """Initialize the callback.

        @param job_table_obj:
                    the associated L{JobTable} object.
        @type job_table_obj:    L{JobTable}
        """
        super(TriggerBtHandler, self).__init__()
        self._c = job_table_obj

    def buttonClick(self, event):
        AutoRefresh.reset()
        try:
            sql_update_acknowledge_request(self._c._sql_session,
                                           self._c.get_selected(),
                                           self._c._workload)
        except:
            self._c.getWindow().showNotification(
                _("Internal error."),
                _("<br/>Job/jobset (%d_%d) not found in the database.") %
                (self._c.workload_date, self._c.get_selected()),
                Notification.TYPE_ERROR_MESSAGE)
        else:
            n = Notification(
                _("Acknowledgement taken into account."),
                _("<br/>This may take up to a minute to be taken into \
                   account."),
                Notification.TYPE_WARNING_MESSAGE)
            n.setDelayMsec(5000)
            self._c.getWindow().showNotification(n)


class ForceBtHandler(IClickListener):

    """Callback for the Force status button."""

    def __init__(self, job_table_obj):
        """Initialize the callback.

        @param job_table_obj:
                    the associated L{JobTable} object.
        @type job_table_obj:    L{JobTable}
        """
        super(ForceBtHandler, self).__init__()
        self._c = job_table_obj

    def buttonClick(self, event):
        AutoRefresh.reset()
        self._c._popup_menu.do_force(self._c.get_selected())


class DoubleClick(IItemClickListener):

    """Double-click callback."""

    def __init__(self, job_table_obj):
        """Initialize the callback.

        @param job_table_obj:
                    the associated L{JobTable} object.
        @type job_table_obj:    L{JobTable}
        """
        super(DoubleClick, self).__init__()
        self._c = job_table_obj

    def itemClick(self, event):
        if event.isDoubleClick():
            item_id = event.getItemId()
            # Edit for jobs and display for jobsets
            if self._c.is_jobset(item_id):
                p = self._c.get_jobset_ids()
                p.append(item_id)
                map(lambda obj: obj.select(p), self._c._select_listeners)
            else:
                self._c._popup_menu.do_edit(item_id,
                                            event.getClientX(),
                                            event.getClientY())
