# Schedwi
# Copyright (C) 2012, 2013 Herve Quatremain
#
# This file is part of Schedwi.
#
# Schedwi is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 3 of the License, or
# (at your option) any later version.
#
# Schedwi is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

"""Widgets for the job/jobset parameters."""

from muntjac.api import CheckBox, TextField, HorizontalLayout, Label
from muntjac.data.validators.integer_validator import IntegerValidator
from muntjac.data.validators.regexp_validator import RegexpValidator
from muntjac.ui.button import IClickListener

import sql_hierarchy
from tables.job_host import job_host
from tables.job_host_s import job_host_s
from tables.job_cluster import job_cluster
from tables.job_cluster_s import job_cluster_s
from tables.job_max_duration import job_max_duration
from tables.job_max_duration_s import job_max_duration_s
from tables.job_retries import job_retries
from tables.job_retries_s import job_retries_s
from tables.job_retries_interval import job_retries_interval
from tables.job_retries_interval_s import job_retries_interval_s
from tables.job_start_limit import job_start_limit
from tables.job_start_limit_s import job_start_limit_s
from tables.job_success_return_code import job_success_return_code
from tables.job_success_return_code_s import job_success_return_code_s
from tables.job_command import job_command
from tables.job_command_s import job_command_s
from tables.job_username import job_username
from tables.job_username_s import job_username_s
from tables.job_file_out import job_file_out
from tables.job_file_out_s import job_file_out_s
from tables.job_file_err import job_file_err
from tables.job_file_err_s import job_file_err_s
from tables.job_control_group import job_control_group
from tables.job_control_group_s import job_control_group_s
from tables.job_loadenv import job_loadenv
from tables.job_loadenv_s import job_loadenv_s
from tables.job_detach import job_detach
from tables.job_detach_s import job_detach_s
from tables.job_manual import job_manual
from tables.job_manual_s import job_manual_s
from tables.job_manual_command import job_manual_command
from tables.job_manual_command_s import job_manual_command_s
from web.autorefresh import AutoRefresh
import web.selecthostwidget


class JobWidget(IClickListener):

    """Job/jobset parameter widget."""

    def __init__(self, jobset_ids, sql_session, workload=None, job_id=None):
        """Build the widget.

        @param jobset_ids:
                    IDs (list) of the parent jobsets.
        @param sql_session:
                    SQLAlchemy session (it can be an opened session)
        @param workload:
                    workload to consider.
        @param job_id:
                    ID of the edited job/jobset.
        """
        super(JobWidget, self).__init__()
        self._jobset_ids = jobset_ids
        self._sql_session = sql_session
        self._workload = workload
        self._job_id = job_id

        self._widgets = list()
        self._inherited = CheckBox(_("Inherited"))
        self._inherited.setDescription(_("Use the default value"))
        self._inherited.setImmediate(True)
        self._inherited.addListener(self, IClickListener)

    def _addInheritedWidget(self, widget):
        """Add a widget to the list of widgets that constitute the UI.

        @param widget:
                    the widget to add to the list.
        """
        self._widgets.append(widget)
        if self.isInherited():
            widget.setEnabled(False)
        else:
            widget.setEnabled(True)

    def _doGreyOut(self):
        """Disable the widgets (grey-out)."""
        for w in self._widgets:
            w.setEnabled(False)

    def _undoGreyOut(self):
        """Enable the widgets (grey-out)."""
        for w in self._widgets:
            w.setEnabled(True)

    def _setDefaultValue(self):
        """Display the default (inherited) value."""
        raise NotImplementedError

    def _saveValue(self):
        """Save the current value.

        Called before the default inherited value is displayed
        because the user clicked on the 'inherited' checkbox.
        This way if the user uncheck the checkbox later on, the
        previously saved value may be displayed again.
        """
        raise NotImplementedError

    def _restoreValue(self):
        """Restore (display) the previously saved value."""
        raise NotImplementedError

    def getInheritedCheckbox(self):
        """Return the 'inherited' checkbox widget."""
        return self._inherited

    def isInherited(self):
        """Check if the 'inherired' checkbox is selected.

        @return:  True if checked, False if unchecked.
        """
        return self._inherited.booleanValue()

    def getFieldWidget(self):
        """Return the parameter widget."""
        raise NotImplementedError

    def setFocus(self):
        """Set the focus on this widget."""
        raise NotImplementedError

    def updateJob(self, job):
        """Update the provided job object.

        @param job:
                the job_main (or job_main_s) object to update with
                the value of the current widget.
        @return:
                None on success or an error message if the value entered
                by the user is not valid.
        """
        raise NotImplementedError

    def buttonClick(self, event):
        """Inherited checkbox callback."""
        AutoRefresh.reset()
        enabled = event.getButton().booleanValue()
        if enabled:
            self._saveValue()
            self._setDefaultValue()
            self._doGreyOut()
        else:
            self._undoGreyOut()
            self._restoreValue()


class JobWidgetSimple(JobWidget):
    def __init__(self, jobset_ids, sql_session, sql_table,
                 workload=None, job_id=None):
        """Build a simple "TextField" widget.

        @param jobset_ids:
                    IDs (list) of the parent jobsets.
        @param sql_session:
                    SQLAlchemy session.
        @param sql_table:
                    SQL table object.
        @param workload:
                    workload to consider.
        @param job_id:
                    ID of the edited job/jobset.
        """
        super(JobWidgetSimple, self).__init__(jobset_ids, sql_session,
                                              workload, job_id)
        self._sql_table = sql_table

        # Retrieve the current and the default value
        ids = list(jobset_ids)
        if job_id is not None:
            ids.append(job_id)
        from_obj, obj = sql_hierarchy.get_by_id(sql_session, ids,
                                                sql_table, workload)
        if job_id is None or not from_obj:
            self._obj = None
            self._default_obj = obj
        else:
            self._obj = obj
            foo, self._default_obj = sql_hierarchy.get_by_id(sql_session,
                                                             jobset_ids,
                                                             sql_table,
                                                             workload)


class Host(JobWidget):
    def __init__(self, jobset_ids, sql_session, workload=None, job_id=None):
        super(Host, self).__init__(jobset_ids,
                                   sql_session,
                                   workload,
                                   job_id)
        # Retrieve the current and the default value
        ids = list(jobset_ids)
        if job_id is not None:
            ids.append(job_id)
        from_obj, obj = sql_hierarchy.get_cluster_or_host_by_id(sql_session,
                                                                ids, workload)
        if job_id is None or not from_obj:
            self._obj = None
            self._default_obj = obj.copy()
        else:
            self._obj = obj.copy()
            foo, obj = sql_hierarchy.get_cluster_or_host_by_id(sql_session,
                                                               jobset_ids,
                                                               workload)
            if obj is not None:
                self._default_obj = obj.copy()
            else:
                self._default_obj = None

        self._savedValue = None
        self._w = web.selecthostwidget.SelectHostCluster(
            sql_session,
            None,
            _("Host/Cluster on which the job/jobset must run"))
        self._addInheritedWidget(self._w)

        if self._obj is not None:
            self._w.set_value(self._obj)
        else:
            self._savedValue = self._w.get_internal_id(self._default_obj)
            self._setDefaultValue()
            self._inherited.setValue(True)
            self._doGreyOut()

    def getFieldWidget(self):
        return self._w

    def _setDefaultValue(self):
        self._w.set_value(self._default_obj)

    def _saveValue(self):
        self._savedValue = self._w.getValue()

    def _restoreValue(self):
        self._w.setValue(self._savedValue)

    def setFocus(self):
        self._w.focus()

    def get_selected(self):
        return self._w.get_selected_id()

    def get_selected_name(self):
        return self._w.get_selected_name()

    def updateJob(self, job):
        type_obj, id_obj = self._w.get_selected_id()
        if self.isInherited() or type_obj is None or id_obj is None:
            try:
                del job.job_host
                del job.job_cluster
            except:
                pass
            return None
        if type_obj == web.selecthostwidget._HOST_TYPE:
            try:
                del job.job_cluster
            except:
                pass
            if self._workload is None:
                job.job_host = job_host(id_obj)
            else:
                job.job_host = job_host_s(id_obj, self._workload)
        else:
            try:
                del job.job_host
            except:
                pass
            if self._workload is None:
                job.job_cluster = job_cluster(id_obj)
            else:
                job.job_cluster = job_cluster_s(id_obj, self._workload)
        return None


class MaxDuration(JobWidgetSimple):
    def __init__(self, jobset_ids, sql_session, workload=None, job_id=None):
        super(MaxDuration, self).__init__(
            jobset_ids,
            sql_session,
            job_max_duration_s if workload else job_max_duration,
            workload,
            job_id)
        self._savedValue = ''
        self._hb = HorizontalLayout()
        self._hb.setSpacing(True)
        self._w = TextField()
        self._w.addValidator(IntegerValidator(_("Must be a number")))
        self._w.setColumns(4)
        self._w.setDescription(_("Maximum number of minutes for the \
                                job/jobset to finish.  0 means no limit"))
        self._w.setImmediate(True)
        self._hb.addComponent(self._w)
        self._hb.addComponent(Label(_("minutes")))
        self._addInheritedWidget(self._hb)

        if self._obj is not None:
            self._w.setValue(self._obj.max_duration)
        else:
            self._savedValue = self._default_obj.max_duration
            self._setDefaultValue()
            self._inherited.setValue(True)
            self._doGreyOut()

    def getFieldWidget(self):
        return self._hb

    def _setDefaultValue(self):
        self._w.setValue(self._default_obj.max_duration)

    def _saveValue(self):
        self._savedValue = self._w.getValue()

    def _restoreValue(self):
        self._w.setValue(self._savedValue)

    def setFocus(self):
        self._w.focus()

    def updateJob(self, job):
        if self.isInherited():
            try:
                del job.job_max_duration
            except:
                pass
            return None
        # Sanity check
        try:
            v = int(self._w.getValue())
            if v < 0:
                self.setFocus()
                return _('Must be a positive number of minutes \
                        (0 means no limit)')
        except:
            self.setFocus()
            return _('Must be a number of minutes (0 means no limit)')
        if self._workload is None:
            job.job_max_duration = job_max_duration(v)
        else:
            job.job_max_duration = job_max_duration_s(v, self._workload)
        return None


class Retries(JobWidgetSimple):
    def __init__(self, jobset_ids, sql_session, workload=None, job_id=None):
        super(Retries, self).__init__(
            jobset_ids,
            sql_session,
            job_retries_s if workload else job_retries,
            workload,
            job_id)
        self._savedValue = ''
        self._w = TextField()
        self._w.addValidator(IntegerValidator(_("Must be a number")))
        self._w.setColumns(4)
        self._w.setDescription(_("Only for jobs. Number of times the \
                                job is restarted when it fails"))
        self._w.setImmediate(True)
        self._addInheritedWidget(self._w)

        if self._obj is not None:
            self._w.setValue(self._obj.retries)
        else:
            self._savedValue = self._default_obj.retries
            self._setDefaultValue()
            self._inherited.setValue(True)
            self._doGreyOut()

    def getFieldWidget(self):
        return self._w

    def _setDefaultValue(self):
        self._w.setValue(self._default_obj.retries)

    def _saveValue(self):
        self._savedValue = self._w.getValue()

    def _restoreValue(self):
        self._w.setValue(self._savedValue)

    def setFocus(self):
        self._w.focus()

    def updateJob(self, job):
        if self.isInherited():
            try:
                del job.job_retries
            except:
                pass
            return None
        # Sanity check
        try:
            v = int(self._w.getValue())
            if v < 0:
                self.setFocus()
                return _('Must be a positive number.')
        except:
            self.setFocus()
            return _('Must be a number.')
        if self._workload is None:
            job.job_retries = job_retries(v)
        else:
            job.job_retries = job_retries_s(v, self._workload)
        return None


class RetryInterval(JobWidgetSimple):
    def __init__(self, jobset_ids, sql_session, workload=None, job_id=None):
        super(RetryInterval, self).__init__(
            jobset_ids,
            sql_session,
            job_retries_interval_s if workload else job_retries_interval,
            workload,
            job_id)
        self._savedValue = ''
        self._hb = HorizontalLayout()
        self._hb.setSpacing(True)
        self._w = TextField()
        self._w.addValidator(IntegerValidator(_("Must be a number")))
        self._w.setColumns(4)
        self._w.setDescription(_("Only for jobs. Number of minutes to wait \
                                between retries"))
        self._w.setImmediate(True)
        self._hb.addComponent(self._w)
        self._hb.addComponent(Label(_("minutes")))
        self._addInheritedWidget(self._hb)

        if self._obj is not None:
            self._w.setValue(self._obj.retries_interval)
        else:
            self._savedValue = self._default_obj.retries_interval
            self._setDefaultValue()
            self._inherited.setValue(True)
            self._doGreyOut()

    def getFieldWidget(self):
        return self._hb

    def _setDefaultValue(self):
        self._w.setValue(self._default_obj.retries_interval)

    def _saveValue(self):
        self._savedValue = self._w.getValue()

    def _restoreValue(self):
        self._w.setValue(self._savedValue)

    def setFocus(self):
        self._w.focus()

    def updateJob(self, job):
        if self.isInherited():
            try:
                del job.job_retries_interval
            except:
                pass
            return None
        # Sanity check
        try:
            v = int(self._w.getValue())
            if v < 0:
                self.setFocus()
                return _('Must be a positive number of \
                          minutes between retries.')
        except:
            self.setFocus()
            return _('Must be a number of minutes between retries.')
        if self._workload is None:
            job.job_retries_interval = job_retries_interval(v)
        else:
            job.job_retries_interval = job_retries_interval_s(v,
                                                              self._workload)
        return None


class ReturnCode(JobWidgetSimple):
    def __init__(self, jobset_ids, sql_session, workload=None, job_id=None):
        super(ReturnCode, self).__init__(
            jobset_ids,
            sql_session,
            job_success_return_code_s if workload else job_success_return_code,
            workload,
            job_id)
        self._savedValue = ''
        self._w = TextField()
        self._w.addValidator(
            IntegerValidator(_("Must be a number between 0 and 254")))
        self._w.setColumns(3)
        self._w.setDescription(_("A return code greater than \
                                  this value means failure"))
        self._w.setImmediate(True)
        self._addInheritedWidget(self._w)

        if self._obj is not None:
            self._w.setValue(self._obj.success_ret)
        else:
            self._savedValue = self._default_obj.success_ret
            self._setDefaultValue()
            self._inherited.setValue(True)
            self._doGreyOut()

    def getFieldWidget(self):
        return self._w

    def _setDefaultValue(self):
        self._w.setValue(self._default_obj.success_ret)

    def _saveValue(self):
        self._savedValue = self._w.getValue()

    def _restoreValue(self):
        self._w.setValue(self._savedValue)

    def setFocus(self):
        self._w.focus()

    def updateJob(self, job):
        if self.isInherited():
            try:
                del job.job_success_return_code
            except:
                pass
            return None
        # Sanity check
        try:
            v = int(self._w.getValue())
            if v < 0 or v > 254:
                self.setFocus()
                return _('Must be a number between 0 and 254.')
        except:
            self.setFocus()
            return _('Must be a number between 0 and 254.')
        if self._workload is None:
            job.job_success_return_code = job_success_return_code(v)
        else:
            job.job_success_return_code = job_success_return_code_s(
                v, self._workload)
        return None


class StartLimit(JobWidgetSimple):
    def __init__(self, jobset_ids, sql_session, workload=None, job_id=None):
        super(StartLimit, self).__init__(
            jobset_ids,
            sql_session,
            job_start_limit_s if workload else job_start_limit,
            workload,
            job_id)
        self._savedValue = ''
        self._hb = HorizontalLayout()
        self._hb.setSpacing(True)
        self._w = TextField()
        self._w.addValidator(IntegerValidator(_("Must be a number")))
        self._w.setColumns(4)
        self._w.setDescription(_("Number of minutes for the job/jobset to \
                                start.  0 means no limit"))
        self._w.setImmediate(True)
        self._hb.addComponent(self._w)
        self._hb.addComponent(Label(_("minutes")))
        self._addInheritedWidget(self._hb)

        if self._obj is not None:
            self._w.setValue(self._obj.start_limit)
        else:
            self._savedValue = self._default_obj.start_limit
            self._setDefaultValue()
            self._inherited.setValue(True)
            self._doGreyOut()

    def getFieldWidget(self):
        return self._hb

    def _setDefaultValue(self):
        self._w.setValue(self._default_obj.start_limit)

    def _saveValue(self):
        self._savedValue = self._w.getValue()

    def _restoreValue(self):
        self._w.setValue(self._savedValue)

    def setFocus(self):
        self._w.focus()

    def updateJob(self, job):
        if self.isInherited():
            try:
                del job.job_start_limit
            except:
                pass
            return None
        # Sanity check
        try:
            v = int(self._w.getValue())
            if v < 0:
                self.setFocus()
                return _('Must be a positive number of minutes \
                        (0 means no limit)')
        except:
            self.setFocus()
            return _('Must be a number of minutes (0 means no limit)')
        if self._workload is None:
            job.job_start_limit = job_start_limit(v)
        else:
            job.job_start_limit = job_start_limit_s(v, self._workload)
        return None


class LoadEnv(JobWidgetSimple):
    def __init__(self, jobset_ids, sql_session, workload=None, job_id=None):
        super(LoadEnv, self).__init__(
            jobset_ids,
            sql_session,
            job_loadenv_s if workload else job_loadenv,
            workload,
            job_id)
        self._savedValue = ''
        self._w = CheckBox('Load user environment')
        self._w.setDescription(_("Whether the user environment (.profile, ...)\
                                must be loaded or not"))
        self._w.setImmediate(True)
        self._addInheritedWidget(self._w)

        if self._obj is not None:
            self._w.setValue(bool(self._obj.loadenv))
        else:
            self._savedValue = bool(self._default_obj.loadenv)
            self._setDefaultValue()
            self._inherited.setValue(True)
            self._doGreyOut()

    def getFieldWidget(self):
        return self._w

    def _setDefaultValue(self):
        self._w.setValue(bool(self._default_obj.loadenv))

    def _saveValue(self):
        self._savedValue = self._w.getValue()

    def _restoreValue(self):
        self._w.setValue(self._savedValue)

    def setFocus(self):
        self._w.focus()

    def updateJob(self, job):
        if self.isInherited():
            try:
                del job.job_loadenv
            except:
                pass
            return None
        v = int(self._w.getValue())
        if self._workload is None:
            job.job_loadenv = job_loadenv(v)
        else:
            job.job_loadenv = job_loadenv_s(v, self._workload)
        return None


class Detach(JobWidgetSimple):
    def __init__(self, jobset_ids, sql_session, workload=None, job_id=None):
        super(Detach, self).__init__(
            jobset_ids,
            sql_session,
            job_detach_s if workload else job_detach,
            workload,
            job_id)
        self._savedValue = ''
        #self._w = CheckBox('Start in detach mode')
        self._w = CheckBox()
        self._w.setDescription(_("Whether the completion of the job must be\
                                monitored or not (usefull to start daemons,\
                                services or long running processes)"))
        self._w.setImmediate(True)
        self._addInheritedWidget(self._w)

        if self._obj is not None:
            self._w.setValue(bool(self._obj.detach))
        else:
            self._savedValue = bool(self._default_obj.detach)
            self._setDefaultValue()
            self._inherited.setValue(True)
            self._doGreyOut()

    def getFieldWidget(self):
        return self._w

    def _setDefaultValue(self):
        self._w.setValue(bool(self._default_obj.detach))

    def _saveValue(self):
        self._savedValue = self._w.getValue()

    def _restoreValue(self):
        self._w.setValue(self._savedValue)

    def setFocus(self):
        self._w.focus()

    def updateJob(self, job):
        if self.isInherited():
            try:
                del job.job_detach
            except:
                pass
            return None
        v = int(self._w.getValue())
        if self._workload is None:
            job.job_detach = job_detach(v)
        else:
            job.job_detach = job_detach_s(v, self._workload)
        return None


class Manual(JobWidgetSimple):
    def __init__(self, jobset_ids, sql_session, workload=None, job_id=None):
        super(Manual, self).__init__(
            jobset_ids,
            sql_session,
            job_manual_s if workload else job_manual,
            workload,
            job_id)
        self._savedValue = ''
        self._w = CheckBox()
        self._w.setDescription(_("Whether a confirmation must be asked to\
                                 operators before starting the job"))
        self._w.setImmediate(True)
        self._addInheritedWidget(self._w)

        if self._obj is not None:
            self._w.setValue(bool(self._obj.manual))
        else:
            self._savedValue = bool(self._default_obj.manual)
            self._setDefaultValue()
            self._inherited.setValue(True)
            self._doGreyOut()

    def getFieldWidget(self):
        return self._w

    def _setDefaultValue(self):
        self._w.setValue(bool(self._default_obj.manual))

    def _saveValue(self):
        self._savedValue = self._w.getValue()

    def _restoreValue(self):
        self._w.setValue(self._savedValue)

    def setFocus(self):
        self._w.focus()

    def updateJob(self, job):
        if self.isInherited():
            try:
                del job.job_manual
            except:
                pass
            return None
        v = int(self._w.getValue())
        if self._workload is None:
            job.job_manual = job_manual(v)
        else:
            job.job_manual = job_manual_s(v, self._workload)
        return None


class StartTimeError(Exception):

    """Syntax error in start time exception."""

    def __init__(self, message):
        super(StartTimeError, self).__init__(message)


class StartTime(JobWidget):
    def __init__(self, jobset_ids, sql_session, workload=None, job_id=None):
        super(StartTime, self).__init__(jobset_ids, sql_session,
                                        workload, job_id)

        # Retrieve the current and default value
        ids = list(jobset_ids)
        if job_id is not None:
            ids.append(job_id)
        from_obj, obj = sql_hierarchy.get_start_time_id(sql_session, ids,
                                                        workload)
        if job_id is None or not from_obj:
            self._obj = None
            self._default_obj = obj
        else:
            self._obj = obj
            foo, self._default_obj = sql_hierarchy.get_start_time_id(
                sql_session, jobset_ids, workload)
        self._savedValue_h = ''
        self._savedValue_m = ''
        self._hb = HorizontalLayout()
        self._hb.setSpacing(True)

        self._h = TextField()
        self._h.addValidator(RegexpValidator('^([0-9]|[01][0-9]|2[0-3])$',
                                             True, "0-23"))
        self._h.setInputPrompt(_("HH"))
        self._h.setColumns(2)
        self._h.setMaxLength(2)
        self._h.setDescription(_("Hour"))
        self._h.setImmediate(True)
        self._hb.addComponent(self._h)

        self._hb.addComponent(Label(":"))

        self._m = TextField()
        self._m.addValidator(RegexpValidator('^([0-9]|[0-5][0-9])$',
                                             True, "0-59"))
        self._m.setInputPrompt(_("MM"))
        self._m.setColumns(2)
        self._m.setMaxLength(2)
        self._m.setDescription(_("Minute"))
        self._m.setImmediate(True)
        self._hb.addComponent(self._m)

        self._addInheritedWidget(self._hb)

        if self._obj is not None:
            h, m = self._obj2str(self._obj.start_time)
            self._h.setValue(h)
            self._m.setValue(m)
        else:
            h, m = self._obj2str(self._default_obj.start_time)
            self._savedValue_h = h
            self._savedValue_m = m
            self._setDefaultValue()
            self._inherited.setValue(True)
            self._doGreyOut()

    def _obj2str(self, start_time):
        return ("%02d" % (start_time / 100),
                "%02d" % (start_time % 100))

    def getFieldWidget(self):
        return self._hb

    def _setDefaultValue(self):
        h, m = self._obj2str(self._default_obj.start_time)
        self._h.setValue(h)
        self._m.setValue(m)

    def _saveValue(self):
        self._savedValue_h = self._h.getValue()
        self._savedValue_m = self._m.getValue()

    def _restoreValue(self):
        self._h.setValue(self._savedValue_h)
        self._m.setValue(self._savedValue_m)

    def setFocus(self):
        self._h.focus()

    def get_value(self):
        """Return the start time value ready to be update in the database.

        @raise StartTimeError:
                    syntax error in the start time.
        """
        if self.isInherited():
            return -1
        # Sanity check
        try:
            h = int(self._h.getValue())
            if h < 0 or h > 23:
                self._h.focus()
                raise StartTimeError(_('Hour must be between 00 and 23'))
        except:
            self._h.focus()
            raise StartTimeError(_('Hour must be a number between 00 and 23'))
        try:
            m = int(self._m.getValue())
            if m < 0 or m > 59:
                self._m.focus()
                raise StartTimeError(_('Minute must be between 00 and 59'))
        except:
            self._m.focus()
            raise StartTimeError(_('Minute must be a number \
                                    between 00 and 59'))
        return h * 100 + m

    def updateJob(self, job):
        try:
            job.start_time = self.get_value()
        except StartTimeError as e:
            return str(e)
        return None


class JobWidgetSimpleText(JobWidgetSimple):
    def __init__(self, jobset_ids, sql_session, sql_table, descr,
                 workload=None, job_id=None):
        """Build a simple "TextField" widget for a text entry.

        @param jobset_ids:
                    IDs (list) of the parent jobsets.
        @param sql_session:
                    SQLAlchemy session.
        @param sql_table:
                    SQL table object.
        @param descr:
                    description used for the tooltip.
        @param workload:
                    workload to consider.
        @param job_id:
                    ID of the edited job/jobset.
        """
        super(JobWidgetSimpleText, self).__init__(jobset_ids, sql_session,
                                                  sql_table,
                                                  workload, job_id)
        self._savedValue = ''
        self._w = TextField()
        self._w.setWidth('100%')
        self._w.setDescription(descr)
        self._addInheritedWidget(self._w)

    def getFieldWidget(self):
        return self._w

    def _saveValue(self):
        self._savedValue = self._w.getValue()

    def _restoreValue(self):
        self._w.setValue(self._savedValue)

    def setFocus(self):
        self._w.focus()


class Command(JobWidgetSimpleText):
    def __init__(self, jobset_ids, sql_session, descr,
                 workload=None, job_id=None):
        super(Command, self).__init__(
            jobset_ids,
            sql_session,
            job_command_s if workload else job_command,
            descr,
            workload,
            job_id)
        if self._obj is not None:
            self._w.setValue(self._obj.command.encode('utf-8'))
        else:
            if self._default_obj:
                self._savedValue = self._default_obj.command.encode('utf-8')
            else:
                self._savedValue = ''
            self._setDefaultValue()
            if jobset_ids:
                # Only if it's not the root jobset which is edited
                self._inherited.setValue(True)
                self._doGreyOut()

    def _setDefaultValue(self):
        if self._default_obj:
            self._w.setValue(self._default_obj.command.encode('utf-8'))
        else:
            self._w.setValue('')

    def updateJob(self, job):
        if self.isInherited():
            try:
                del job.job_command
            except:
                pass
            return None
        v = self._w.getValue()
        if v is None:
            v = ''
        if self._workload is None:
            job.job_command = job_command(v)
        else:
            job.job_command = job_command_s(v, self._workload)
        return None


class Username(JobWidgetSimpleText):
    def __init__(self, jobset_ids, sql_session, descr,
                 workload=None, job_id=None):
        super(Username, self).__init__(
            jobset_ids,
            sql_session,
            job_username_s if workload else job_username,
            descr,
            workload,
            job_id)
        if self._obj is not None:
            self._w.setValue(self._obj.username.encode('utf-8'))
        else:
            self._savedValue = self._default_obj.username.encode('utf-8')
            self._setDefaultValue()
            self._inherited.setValue(True)
            self._doGreyOut()

    def _setDefaultValue(self):
        self._w.setValue(self._default_obj.username.encode('utf-8'))

    def updateJob(self, job):
        if self.isInherited():
            try:
                del job.job_username
            except:
                pass
            return None
        v = self._w.getValue()
        if v is None:
            v = ''
        if self._workload is None:
            job.job_username = job_username(v)
        else:
            job.job_username = job_username_s(v, self._workload)
        return None


class Stdout(JobWidgetSimpleText):
    def __init__(self, jobset_ids, sql_session, descr,
                 workload=None, job_id=None):
        super(Stdout, self).__init__(
            jobset_ids,
            sql_session,
            job_file_out_s if workload else job_file_out,
            descr,
            workload,
            job_id)
        if self._obj is not None:
            self._w.setValue(self._obj.file_out.encode('utf-8'))
        else:
            self._savedValue = self._default_obj.file_out.encode('utf-8')
            self._setDefaultValue()
            self._inherited.setValue(True)
            self._doGreyOut()

    def _setDefaultValue(self):
        self._w.setValue(self._default_obj.file_out.encode('utf-8'))

    def updateJob(self, job):
        if self.isInherited():
            try:
                del job.job_file_out
            except:
                pass
            return None
        v = self._w.getValue()
        if v is None:
            v = ''
        if self._workload is None:
            job.job_file_out = job_file_out(v)
        else:
            job.job_file_out = job_file_out_s(v, self._workload)
        return None


class Stderr(JobWidgetSimpleText):
    def __init__(self, jobset_ids, sql_session, descr,
                 workload=None, job_id=None):
        super(Stderr, self).__init__(
            jobset_ids,
            sql_session,
            job_file_err_s if workload else job_file_err,
            descr,
            workload,
            job_id)
        if self._obj is not None:
            self._w.setValue(self._obj.file_err.encode('utf-8'))
        else:
            self._savedValue = self._default_obj.file_err.encode('utf-8')
            self._setDefaultValue()
            self._inherited.setValue(True)
            self._doGreyOut()

    def _setDefaultValue(self):
        self._w.setValue(self._default_obj.file_err.encode('utf-8'))

    def updateJob(self, job):
        if self.isInherited():
            try:
                del job.job_file_err
            except:
                pass
            return None
        v = self._w.getValue()
        if v is None:
            v = ''
        if self._workload is None:
            job.job_file_err = job_file_err(v)
        else:
            job.job_file_err = job_file_err_s(v, self._workload)
        return None


class CGroup(JobWidgetSimpleText):
    def __init__(self, jobset_ids, sql_session, descr,
                 workload=None, job_id=None):
        super(CGroup, self).__init__(
            jobset_ids,
            sql_session,
            job_control_group_s if workload else job_control_group,
            descr,
            workload,
            job_id)
        if self._obj is not None:
            self._w.setValue(self._obj.control_group.encode('utf-8'))
        else:
            self._savedValue = self._default_obj.control_group.encode('utf-8')
            self._setDefaultValue()
            self._inherited.setValue(True)
            self._doGreyOut()

    def _setDefaultValue(self):
        self._w.setValue(self._default_obj.control_group.encode('utf-8'))

    def updateJob(self, job):
        if self.isInherited():
            try:
                del job.job_control_group
            except:
                pass
            return None
        v = self._w.getValue()
        if v is None:
            v = ''
        if self._workload is None:
            job.job_control_group = job_control_group(v)
        else:
            job.job_control_group = job_control_group_s(v, self._workload)
        return None


class ManualCommand(JobWidgetSimpleText):
    def __init__(self, jobset_ids, sql_session, descr,
                 workload=None, job_id=None):
        super(ManualCommand, self).__init__(
            jobset_ids,
            sql_session,
            job_manual_command_s if workload else job_manual_command,
            descr,
            workload,
            job_id)
        if self._obj is not None:
            self._w.setValue(self._obj.manual_command.encode('utf-8'))
        else:
            if self._default_obj:
                self._savedValue = self._default_obj.manual_command.encode(
                    'utf-8')
            else:
                self._savedValue = ''
            self._setDefaultValue()
            if jobset_ids:
                # Only if it's not the root jobset which is edited
                self._inherited.setValue(True)
                self._doGreyOut()

    def _setDefaultValue(self):
        if self._default_obj:
            self._w.setValue(self._default_obj.manual_command.encode('utf-8'))
        else:
            self._w.setValue('')

    def updateJob(self, job):
        if self.isInherited():
            try:
                del job.job_manual_command
            except:
                pass
            return None
        v = self._w.getValue()
        if v is None:
            v = ''
        if self._workload is None:
            job.job_manual_command = job_manual_command(v)
        else:
            job.job_manual_command = job_manual_command_s(v, self._workload)
        return None
