# Schedwi
# Copyright (C) 2013 Herve Quatremain
#
# This file is part of Schedwi.
#
# Schedwi is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 3 of the License, or
# (at your option) any later version.
#
# Schedwi is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

"""Draw dependency link graphs."""

from muntjac.ui.custom_layout import CustomLayout

import parse_config
import simple_queries_job
import status_utils
import path


class LinkGraph(CustomLayout):

    """Dependency graph widget."""

    def __init__(self, job_id, canvas_id, width, height,
                 sql_session, workload=None):
        """Draw the graph for the provided job[set] in the CustomLayout widget.

        @param job_id:
                    Database ID of the job/jobset to graph.
        @param canvas_id:
                    Internal HTML canvas name (the `id' attribute)
        @param width:
                    Initial width of the canvas.
        @param height:
                    Initial height of the canvas.
        @param sql_session:
                    SQLAlchemy session.
        @param workload:
                    workload to consider.
        """
        super(LinkGraph, self).__init__()
        self._job_id = job_id
        self._canvas_id = canvas_id
        self._sql_session = sql_session
        self._workload = workload

        self._get_graph()

        #
        # Build the HTML/javascript to inject
        #
        s = '\n<canvas id="%s" width="%d" height="%d"></canvas>' % (canvas_id,
                                                                    width,
                                                                    height)
        f = open(parse_config.GUI_CONTEXT_ROOT +
                 '/VAADIN/themes/schedwi/paper.js', 'r')
        s += ('\n<script type="text/javascript">\n//<![CDATA[\n' +
              f.read() +
              '\n//]]>\n</script>' +
              '\n<script type="text/javascript">\n//<![CDATA[\n' +
              self._constants() +
              self._graph_to_javascript())
        f.close()
        f = open(parse_config.GUI_CONTEXT_ROOT +
                 '/VAADIN/themes/schedwi/graph_node_positions.js', 'r')
        s += f.read()
        f.close()
        f = open(parse_config.GUI_CONTEXT_ROOT +
                 '/VAADIN/themes/schedwi/link_graph.js', 'r')
        s += f.read() + '\n//]]>\n</script>\n'
        f.close()
        self.setTemplateContents(s.encode('utf-8'))

    def _get_graph_recursive(self, session, job_id, parent_id):
        """Recursively build the dependency graph.

        @param session:
                    an opened SQLAlchemy session.
        @param job_id:
                    database ID of the job/jobset.
        @param parent_id:
                    database ID of the parent jobset.  If the provided job_id
                    is not directly under this parent jobset, then the full
                    name (from `/') of the job/jobset is going to be stored
                    in the self._job_details dictionnary. Otherwise just the
                    basename of the job/jobset is stored.
        """
        if job_id not in self._job_details:
            job = simple_queries_job.sql_get_job(session, job_id,
                                                 self._workload)
            if job.parent == parent_id:
                self._job_details[job_id] = {'name': job.name,
                                             'type': job.type}
            else:
                # Store the full path of the job/jobset
                p = path.id2path(session, job_id, self._workload)
                self._job_details[job_id] = {'name': p.path,
                                             'type': job.type}
            if self._workload is not None:
                status = status_utils.get_status(session, job_id,
                                                 self._workload)
                # status can be '' if no status is found
                self._job_details[job_id]['state'] = status.status
            else:
                self._job_details[job_id]['state'] = 0

            self._graph[job_id] = list()
            # Recursively fetch the `parent' links
            for l in job.links:
                self._graph[job_id].append({'id': l.job_id_destination,
                                            'status': l.required_status})
                self._get_graph_recursive(session, l.job_id_destination,
                                          parent_id)
            # Then recursively fetch the `child' links
            for l in job.link_targets:
                self._get_graph_recursive(session, l.job_id_source, parent_id)

    def _get_graph(self):
        """Get the dependency graph for the provided job or jobset.

        The self._graph dictionnary is set by this method. It contains the
        dependency graph itself.  The self._job_details, which is also set
        here, contains the details of the jobs/jobsets (name, status,
        type, ...) defined in the graph.
        """
        self._graph = dict()
        self._job_details = dict()
        session = self._sql_session.open_session()
        job = simple_queries_job.sql_get_job(session, self._job_id,
                                             self._workload)
        self._get_graph_recursive(session, self._job_id, job.parent)
        self._sql_session.close_session(session)

    def _graph_to_javascript(self):
        """Convert the dependency graph to javascript.

        The self._graph and self._job_details dictionnaries (build by
        self._get_graph()), are converted to javascript.

        @return:    A string that contains the javascript code.
        """
        graph = list()
        for job_id in self._graph:
            if self._graph[job_id]:
                links = list()
                for l in range(len(self._graph[job_id])):
                    links.append('{id: %d, status: %s}' %
                                 (self._graph[job_id][l]['id'],
                                  self._graph[job_id][l]['status']))
                graph.append('  %3d: [%s]' % (job_id, ', '.join(links)))
        javascript_src = '\nvar graph = {\n' + ',\n'.join(graph) + '\n};\n\n'
        details = list()
        for job_id in self._job_details:
            details.append(
                '  %3d: {name: "%s", type: %d, state: %d, fake: false}'
                % (job_id, self._job_details[job_id]['name'],
                   self._job_details[job_id]['type'],
                   0 if not self._job_details[job_id]['state'] else
                   self._job_details[job_id]['state']))
        javascript_src += ('var job_details = {\n' +
                           ',\n'.join(details) +
                           '\n};\n\n')
        return javascript_src

    def _constants(self):
        """Build a javascript chunk that declare global constant values.

        @return:    A string that contains the javascript code.
        """
        javascript_src = """
          /* Schedwi
             Copyright (C) 2013 Herve Quatremain

             This file is part of Schedwi.

             Schedwi is free software; you can redistribute it and/or modify
             it under the terms of the GNU General Public License as published
             by the Free Software Foundation; either version 3 of the License,
             or (at your option) any later version.

             Schedwi is distributed in the hope that it will be useful,
             but WITHOUT ANY WARRANTY; without even the implied warranty of
             MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
             GNU General Public License for more details.

             You should have received a copy of the GNU General Public License
             along with this program.  If not, see
             <http://www.gnu.org/licenses/>.
          */

          var JOBSET = 0;
          var JOB = 1;

          var NONE = 0;
          var WAITING = 1;
          var RUNNING = 2;
          var COMPLETED = 3;
          var FAILED = 4;
          var C_OR_F = 5;

          var colors = [];
          colors[NONE] = 'black';
          colors[WAITING] = '#fffd80';
          colors[RUNNING] = '#0e51a7';
          colors[COMPLETED] = '#66cc66';
          colors[FAILED] = '#a62929';
          colors[C_OR_F] = 'black';

          var status_names = [];
        """
        javascript_src += (
            "status_names[NONE] = '" +
            status_utils.status2string(
                status_utils.COMPLETED_OR_FAILED).decode('utf-8') +
            "';\n")
        for state in status_utils.status_list():
            javascript_src += "status_names[%d] = '%s';\n" % \
                (state, status_utils.status2string(state).decode('utf-8'))
        javascript_src += "var string_tooltip = '%s';\n" % \
                          (_("%s must be %s").decode('utf-8') + "\\n" +
                           _("for %s to start").decode('utf-8'))
        javascript_src += "var canvas_id = '%s';\n" % self._canvas_id
        return javascript_src
