# Schedwi
# Copyright (C) 2013 Herve Quatremain
#
# This file is part of Schedwi.
#
# Schedwi is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 3 of the License, or
# (at your option) any later version.
#
# Schedwi is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

"""Agent registration commands (schedwica wrapper)"""

import os
import subprocess
import logging

import config

_schedwica_prog = config.BINDIR + "/schedwica"


def is_approval_possible():
    """Check if hosts can be approved.

    @return:
                True if the schedwica program is available, False otherwise.
    """
    return os.access(_schedwica_prog, os.X_OK)


def get_pending():
    """Get the list of pending registration requests.

    @return:
                The list of pending requests (hostnames) or None if no
                request is pending, in case of error or if the schedwica
                program is not available.
    """
    if not os.access(_schedwica_prog, os.X_OK):
        return None
    p = subprocess.Popen([_schedwica_prog, '-q', '-l'],
                         stdout=subprocess.PIPE,
                         stderr=subprocess.PIPE)
    out, err = p.communicate()
    ret = p.returncode
    if ret:
        logging.warning(_schedwica_prog + ": " + err.strip())
        return None
    if not out:
        return None
    return out.strip().split('\n')


def register(hostname):
    """Register the provided hostname in the database.

    @return:
                None is case of success or an error message.
    """
    if not hostname or not os.access(_schedwica_prog, os.X_OK):
        return None
    h = hostname.split('\t')[0]
    p = subprocess.Popen([_schedwica_prog, '-q',
                          '-d', 'Registered through the Web interface',
                          '-s', h],
                         stdout=subprocess.PIPE,
                         stderr=subprocess.PIPE)
    out, err = p.communicate()
    ret = p.returncode
    if ret:
        return err.strip()
    return None


if __name__ == "__main__":
    p = get_pending()
    for i in p:
        print "[%s]" % i
