# Schedwi
# Copyright (C) 2012 Herve Quatremain
#
# This file is part of Schedwi.
#
# Schedwi is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 3 of the License, or
# (at your option) any later version.
#
# Schedwi is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

"""Calendar selection widget."""

from muntjac.api import Panel, Tree
from muntjac.data.util.hierarchical_container import HierarchicalContainer
from muntjac.terminal.theme_resource import ThemeResource
from muntjac.data.property import IValueChangeListener
from muntjac.ui.tree import IExpandListener, ICollapseListener

from path_cal import id2list
from simple_queries_cal import sql_get_cal_children, sql_count_cal_children


# Memorize the last selected item to show the same containing dir next time
_LAST_SELECTED = None


class SelectCalendar(Panel,
                     IValueChangeListener, IExpandListener, ICollapseListener):

    """Calendar selection widget."""

    _ICON_DIR = ThemeResource('icons/dir.png')
    _ICON_CAL = ThemeResource('icons/calendar.png')
    _ROOT_DIR_ID = 1

    def __init__(self, sql_session, workload=None, selected_cal=None):
        """Build the calendar selection widget.

        @param sql_session:
                    SQLAlchemy session.
        @param workload:
                    workload to consider.
        @param selected_cal:
                    calendar ID to select by default.
        """
        global _LAST_SELECTED
        super(SelectCalendar, self).__init__()
        self._sql_session = sql_session
        self._workload = workload
        self._selected = None

        self.setCaption(_("Calendars:"))
        layout = self.getContent()
        layout.setMargin(False)
        layout.setSpacing(False)

        self._tree = Tree()
        self._tree.setImmediate(True)

        jContainer = HierarchicalContainer()
        jContainer.addContainerProperty('name', str, None)
        jContainer.addContainerProperty('type', int, None)
        jContainer.addContainerProperty('icon', ThemeResource, None)
        # Root calendar
        item = jContainer.addItem(self._ROOT_DIR_ID)
        item.getItemProperty('name').setValue('/')
        item.getItemProperty('type').setValue(1)
        item.getItemProperty('icon').setValue(self._ICON_DIR)
        self._container = jContainer
        self._tree.setContainerDataSource(self._container)
        self._tree.setItemCaptionPropertyId('name')
        self._tree.setItemIconPropertyId('icon')
        self.addComponent(self._tree)

        # Callbacks
        self._tree.addListener(self, IValueChangeListener)
        self._tree.addListener(self, IExpandListener)
        self._tree.addListener(self, ICollapseListener)

        # Expand the tree
        if selected_cal:
            p = id2list(sql_session, selected_cal, workload)
            for i in p[:-1]:
                self._tree.expandItem(i)
            self._tree.select(selected_cal)
        elif _LAST_SELECTED:
            try:
                p = id2list(sql_session, _LAST_SELECTED, workload)
            except:
                self._tree.expandItem(self._ROOT_DIR_ID)
            else:
                for i in p[:-1]:
                    self._tree.expandItem(i)
        else:
            self._tree.expandItem(self._ROOT_DIR_ID)

    def getSelected(self):
        """Return the path of the selected calendar.

        @return:
                the selected calendar ID or None if nothing is selected or
                if a directory is selected.
        """
        if not self._selected:
            return None
        item = self._container.getItem(self._selected)
        if item.getItemProperty('type').getValue():
            return None
        return self._selected

    def valueChange(self, event):
        """Callback when a calendar or a directory has been selected
           or de-selected.
        """
        global _LAST_SELECTED
        self._selected = event.getProperty().getValue()
        if self._selected:
            _LAST_SELECTED = self._selected

    def nodeExpand(self, event):
        """Callback when a directory has been expanded."""
        cal_id = event.getItemId()
        session = self._sql_session.open_session()
        for child in sql_get_cal_children(session, cal_id,
                                          False, self._workload):
            child_id = child.id
            item = self._container.addItem(child_id)
            item.getItemProperty('name').setValue(child.name.encode('utf-8'))
            item.getItemProperty('type').setValue(child.entry_type)
            if child.entry_type == 1:
                item.getItemProperty('icon').setValue(self._ICON_DIR)
            else:
                item.getItemProperty('icon').setValue(self._ICON_CAL)
            self._container.setParent(child_id, cal_id)
            # Only show the expand arrow if the dir has children
            if child.entry_type == 1 and \
               sql_count_cal_children(session, child_id, False,
                                      self._workload):
                self._container.setChildrenAllowed(child_id, True)
            else:
                self._container.setChildrenAllowed(child_id, False)
        self._sql_session.close_session(session)

    def nodeCollapse(self, event):
        """Callback when a directory has been collapsed."""
        cal_id = event.getItemId()
        # Remove all the existing children first
        if self._container.hasChildren(cal_id):
            for childId in self._container.getChildren(cal_id):
                self._tree.collapseItem(childId)
                self._container.removeItemRecursively(childId)
