# Schedwi
# Copyright (C) 2012 Herve Quatremain
#
# This file is part of Schedwi.
#
# Schedwi is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 3 of the License, or
# (at your option) any later version.
#
# Schedwi is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

"""Environment group selection widget."""

import environments_utils

from muntjac.api import NativeSelect
from muntjac.data.property import IValueChangeListener
from muntjac.data.util.indexed_container import IndexedContainer

# Memorize the last selected item to select the same group next time
_LAST_SELECTED = None


class SelectEnvGroup(NativeSelect, IValueChangeListener):

    """Environment group selection widget."""

    def __init__(self, sql_session, label=None, descr=None, env_group_id=None,
                 workload=None):
        """Build the environment group selection widget.

        @param sql_session:
                    SQLAlchemy session.
        @param label:
                    Selection box caption.
        @param descr:
                    Tooltip.
        @param env_group_id:
                    Database ID of the group to select by default.
        @param workload:
                    workload to consider.
        """
        global _LAST_SELECTED
        super(SelectEnvGroup, self).__init__()

        # Container
        c = IndexedContainer()
        c.addContainerProperty('env', str, None)
        for e in environments_utils.name2env_list(sql_session, '*', workload):
            item = c.addItem(e.id)
            item.getItemProperty('env').setValue(e.name.encode('utf-8'))
        self.setContainerDataSource(c)
        if label:
            self.setCaption(label)
        if descr:
            self.setDescription(descr)
        self.setNullSelectionAllowed(False)
        self.setItemCaptionPropertyId('env')
        if env_group_id is not None and c.containsId(env_group_id):
            self.setValue(env_group_id)
        elif _LAST_SELECTED is not None and c.containsId(_LAST_SELECTED):
            self.setValue(_LAST_SELECTED)
        else:
            self.setValue(c.firstItemId())
        self.addListener(self, IValueChangeListener)

    def getSelected(self):
        """Return the selected environment group ID and its name.

        @return: the tuple (group_id, group_name)
        """
        c = self.getContainerDataSource()
        item_id = self.getValue()
        item = c.getItem(item_id)
        return [item_id, item.getItemProperty('env').getValue()]

    def valueChange(self, event):
        """Callback when an environment group is selected from the list."""
        global _LAST_SELECTED
        _LAST_SELECTED = self.getValue()
