# Schedwi
# Copyright (C) 2012 Herve Quatremain
#
# This file is part of Schedwi.
#
# Schedwi is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 3 of the License, or
# (at your option) any later version.
#
# Schedwi is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

"""Job/jobset statistics UI."""

from muntjac.api import GridLayout, Label, Alignment
from web.jobtable import JobTableListener
from cmd_job.stat import get_stat, stat2strings


class Stats(GridLayout, JobTableListener):

    """UI component to display a job/jobset run statistics."""

    def __init__(self, jobTable, sql_session, workload=None):
        """Build the job/jobset statistics UI.

        @param jobTable:
                    the associated L{web.jobtable.JobTable} object.
        @param sql_session:
                    SQLAlchemy session.
        @param workload:
                    workload to consider (ignored)
        """
        super(Stats, self).__init__()
        self._sql_session = sql_session

        self.setMargin(True)
        self.setSpacing(True)
        self.setColumns(2)
        self.setRows(5)

        l = Label(_("Number of successful runs:"))
        self.addComponent(l, 0, 0)
        self.setComponentAlignment(l, Alignment.MIDDLE_RIGHT)
        l = Label(_("Number of failed runs:"))
        self.addComponent(l, 0, 1)
        self.setComponentAlignment(l, Alignment.MIDDLE_RIGHT)
        l = Label(_("Average duration of successful runs:"))
        self.addComponent(l, 0, 2)
        self.setComponentAlignment(l, Alignment.MIDDLE_RIGHT)
        l = Label(_("Total duration (when successful):"))
        self.addComponent(l, 0, 3)
        self.setComponentAlignment(l, Alignment.MIDDLE_RIGHT)
        l = Label(_("Last duration:"))
        self.addComponent(l, 0, 4)
        self.setComponentAlignment(l, Alignment.MIDDLE_RIGHT)

        self.num_run_success = Label("-")
        self.addComponent(self.num_run_success, 1, 0)
        self.num_run_failed = Label("-")
        self.addComponent(self.num_run_failed, 1, 1)
        self.avg_duration = Label("-")
        self.addComponent(self.avg_duration, 1, 2)
        self.total_duration = Label("-")
        self.addComponent(self.total_duration, 1, 3)
        self.last_duration = Label("-")
        self.addComponent(self.last_duration, 1, 4)

        jobTable.add_listener(self)

    def refresh(self, job, jobsetIds):
        """Display the statistics for the given job.

        @param job:
                    the job ID for which the stats must be displayed.
        @param jobsetIds:
                    an array of the IDs the parent jobsets (the last element
                    is the ID of the jobset that contains `job')
        """
        if job:
            st = get_stat(self._sql_session, job)
            a = stat2strings(st)
            self.num_run_success.setValue(a['num_success'])
            self.num_run_failed.setValue(a['num_failed'])
            self.avg_duration.setValue(a['avg_duration'])
            self.total_duration.setValue(a['total_duration'])
            self.last_duration.setValue(a['last_duration'])
        else:
            self.num_run_success.setValue("-")
            self.num_run_failed.setValue("-")
            self.avg_duration.setValue("-")
            self.total_duration.setValue("-")
            self.last_duration.setValue("-")
