/* Schedwi
   Copyright (C) 2013 Herve Quatremain

   This file is part of Schedwi.

   Schedwi is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 3 of the License, or
   (at your option) any later version.

   Schedwi is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

/* check_waiting_can_start.c -- Check if a waiting job/jobset can start */

#include <schedwi.h>
#include <schedwi_jobtree.h>
#include <schedwi_time.h>
#include <lwc_log.h>
#include <job_status_state.h>
#include <sql_status.h>
#include <manual_trigger.h>
#include <testfile.h>
#include <module.h>
#include <check_waiting_can_start.h>


/*
 * Error callback function for sql_status_update_wait_reason()
 */
static void
sql_get_error_logger (void *data, const char *msg, int err_code)
{
	if (msg != NULL) {
		lwc_writeLog (LOG_ERR, msg);
	}
	else {
		lwc_writeLog (LOG_ERR,
		_("Database error while trying to set the status of a job"));
	}
}


/*
 * Check if the provided waiting job/jobset can start now or not.  If the
 * job/jobset is not ready to start, then the reason is stored in the
 * job_status database table and ptr->wait_reason is set accordingly.
 *
 * @param workload_date:
 *		the date of the workload.
 * @param ptr:
 * 		the job/jobset pointer.
 * @param current_time:
 *		the current date/time (epoch)
 * @return:
 *		0 if the job/jobset can start, 1 if not, 2 if it's too late
 *              to start the job/jobset (start limit) or -1 on error.
 *		Also, ptr->wait_reason is updated.
 */
int
check_waiting_can_start (int workload_date,
			 schedwi_jobtree_node_ptr ptr,
			 schedwi_time current_time)
{
	int reason, i, ret;
	schedwi_time t;
	schedwi_jobtree_link_ptr link;


	if (ptr == NULL) {
		return 0;
	}

	reason = 0;

	/* Check if the parent is running */
	if (ptr->parent != NULL
	    && ptr->parent->status != JOB_STATUS_STATE_RUNNING)
	{
		/* The parent is not running */
		reason |= WAIT_REASON_MASK_PARENT;
	}

	/* Check the start time*/
	if (current_time < ptr->run_time) {
		/* It's not time yet */
		reason |= WAIT_REASON_MASK_TIME;
	}

	/* Check if it is not to late to start the job */
	if (reason == 0 && ptr->start_limit > 0) {
		if (ptr->run_time > 0) {
			t = ptr->run_time;
		}
		else {
			/* Parent start time */
			if (ptr->parent == NULL) {
				/*
				 * Root jobset
				 * Shouldn't be possible  (there is something
				 * wrong in the database for the  definition
				 * of the root jobset which shouldn't have a
				 * start time limit)
				 * The limit is then ignored here
				 */
				t = current_time;
			}
			else {
				t = ptr->parent->start_time;
			}
		}
		if (current_time > schedwi_time_add_seconds (t,
							ptr->start_limit))
		{
			/* Too late */
			ptr->wait_reason = 0;
			return 2;
		}
	}

	/* Check if all the links are okay */
	for (	link = ptr->links;
		link != NULL
		&& (link->linked_node->status == link->required_status
		    || (link->required_status ==
					JOB_STATUS_STATE_COMPLETED_OR_FAILED
			&& (link->linked_node->status ==
						JOB_STATUS_STATE_COMPLETED
			    || link->linked_node->status ==
						JOB_STATUS_STATE_FAILED)));
		link = link->next);
	if (link != NULL) {
		/* Some links are not satisfied */
		reason |= WAIT_REASON_MASK_LINKS;
	}


	/*
	 * Only do the following tests if so far everything is ready for the
	 * job/jobset to start. These tests may take some time so no need
	 * to run them if we know anyway that the job is not going to start.
	 */
	if (reason != 0) {
		if (ptr->wait_reason != reason) {
			/* Update the database (job_status table) */
			ptr->wait_reason = reason;
			sql_status_update_wait_reason (	workload_date,
							ptr->id, reason,
							sql_get_error_logger,
							NULL);
		}
		return 1;
	}

	/* Check the required files */
	if(ptr->constraint_files != NULL) {
		for (i = 0; i < ptr->num_constraint_files; i++) {
			ret = testfile (workload_date, ptr->id,
					(ptr->constraint_files)[i].host_id,
					(ptr->constraint_files)[i].file, NULL);
			if (ret < 0) {
				return -1;
			}
			if (   (!ret &&  (ptr->constraint_files)[i].exist == 0)
			    || (ret &&  (ptr->constraint_files)[i].exist != 0))
			{
				if (ptr->wait_reason !=
					WAIT_REASON_MASK_CONSTRAINT_FILE)
				{
					ptr->wait_reason =
					    WAIT_REASON_MASK_CONSTRAINT_FILE;
					sql_status_update_wait_reason (
					    workload_date, ptr->id,
					    WAIT_REASON_MASK_CONSTRAINT_FILE,
					    sql_get_error_logger, NULL);
				}
				return 1;
			}
		}
	}

	/* Call the `schedwi_check' function of the modules */
	if (module_check (workload_date, ptr) != 0) {
		if (ptr->wait_reason != WAIT_REASON_MASK_MODULE) {
			ptr->wait_reason = WAIT_REASON_MASK_MODULE;
			sql_status_update_wait_reason (workload_date, ptr->id,
					WAIT_REASON_MASK_MODULE,
					sql_get_error_logger, NULL);
		}
		return 1;
	}

	/*
	 * Check if it's a manual job and it's waiting for acknowledgement.
	 * If yes, this function will alert the operators (only the first
	 * time it is run for this specific job/jobset)
	 */
	if (check_manual_trigger (workload_date, ptr) != 0) {
		if (ptr->wait_reason != WAIT_REASON_MASK_MANUAL) {
			ptr->wait_reason = WAIT_REASON_MASK_MANUAL;
			sql_status_update_wait_reason (workload_date, ptr->id,
					WAIT_REASON_MASK_MANUAL,
					sql_get_error_logger, NULL);
		}
		return 1;
	}

	if (ptr->wait_reason != 0) {
		ptr->wait_reason = 0;
		sql_status_update_wait_reason (	workload_date, ptr->id, 0,
						sql_get_error_logger, NULL);
	}
	return 0;
}

/*-----------------============== End Of File ==============-----------------*/
