/* Schedwi
   Copyright (C) 2007-2013 Herve Quatremain

   This file is part of Schedwi.

   Schedwi is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 3 of the License, or
   (at your option) any later version.

   Schedwi is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

/* conf.c -- Configuration file management functions */

#include <schedwi.h>

#if HAVE_SYS_TYPES_H
#include <sys/types.h>
#endif

#if STDC_HEADERS
#include <stdlib.h>
#include <string.h>
#else
#if HAVE_STDLIB_H
#include <stdlib.h>
#endif
#if HAVE_STRING_H
#include <string.h>
#endif
#endif

#if HAVE_STDIO_H
#include <stdio.h>
#endif

#if HAVE_SYS_STAT_H
#include <sys/stat.h>
#endif

#if HAVE_UNISTD_H
#include <unistd.h>
#endif

#if HAVE_ERRNO_H
#include <errno.h>
#endif
#ifndef errno
extern int errno;
#endif

#if HAVE_CTYPE_H
#include <ctype.h>
#endif

#if HAVE_SYSLOG_H
#include <syslog.h>
#endif

#if HAVE_ASSERT_H
#include <assert.h>
#endif

#include <lib_functions.h>
#include <module.h>
#include <conf.h>
#include <addrmatch.h>


/*
 * Parameter list
 */
static conf_param_t *params = NULL;


/*
 * Convert a string to a syslog facility number.
 *
 * Return:
 *   0 --> No error (if facility is not NULL, it contains the facility number
 *  -1 --> Unknown facility
 */
static int
string2facility (const char *val, long int *facility)
{
#if HAVE_ASSERT_H
	assert (val != NULL);
#endif

#ifdef LOG_LOCAL0
	if (	   !schedwi_strcasecmp (val, "LOG_LOCAL0")
		|| !schedwi_strcasecmp (val, "LOCAL0"))
	{
		if (facility != NULL) {
			*facility = LOG_LOCAL0;
		}
		return 0;
	}
#endif
#ifdef LOG_LOCAL1
	if (	   !schedwi_strcasecmp (val, "LOG_LOCAL1")
		|| !schedwi_strcasecmp (val, "LOCAL1"))
	{
		if (facility != NULL) {
			*facility = LOG_LOCAL1;
		}
		return 0;
	}
#endif
#ifdef LOG_LOCAL2
	if (	   !schedwi_strcasecmp (val, "LOG_LOCAL2")
		|| !schedwi_strcasecmp (val, "LOCAL2"))
	{
		if (facility != NULL) {
			*facility = LOG_LOCAL2;
		}
		return 0;
	}
#endif
#ifdef LOG_LOCAL3
	if (	   !schedwi_strcasecmp (val, "LOG_LOCAL3")
		|| !schedwi_strcasecmp (val, "LOCAL3"))
	{
		if (facility != NULL) {
			*facility = LOG_LOCAL3;
		}
		return 0;
	}
#endif
#ifdef LOG_LOCAL4
	if (	   !schedwi_strcasecmp (val, "LOG_LOCAL4")
		|| !schedwi_strcasecmp (val, "LOCAL4"))
	{
		if (facility != NULL) {
			*facility = LOG_LOCAL4;
		}
		return 0;
	}
#endif
#ifdef LOG_LOCAL5
	if (	   !schedwi_strcasecmp (val, "LOG_LOCAL5")
		|| !schedwi_strcasecmp (val, "LOCAL5"))
	{
		if (facility != NULL) {
			*facility = LOG_LOCAL5;
		}
		return 0;
	}
#endif
#ifdef LOG_LOCAL6
	if (	   !schedwi_strcasecmp (val, "LOG_LOCAL6")
		|| !schedwi_strcasecmp (val, "LOCAL6"))
	{
		if (facility != NULL) {
			*facility = LOG_LOCAL6;
		}
		return 0;
	}
#endif
#ifdef LOG_LOCAL7
	if (	   !schedwi_strcasecmp (val, "LOG_LOCAL7")
		|| !schedwi_strcasecmp (val, "LOCAL7"))
	{
		if (facility != NULL) {
			*facility = LOG_LOCAL7;
		}
		return 0;
	}
#endif
#ifdef LOG_DAEMON
	if (	   !schedwi_strcasecmp (val, "LOG_DAEMON")
		|| !schedwi_strcasecmp (val, "DAEMON"))
	{
		if (facility != NULL) {
			*facility = LOG_DAEMON;
		}
		return 0;
	}
#endif
#ifdef LOG_USER
	if (	   !schedwi_strcasecmp (val, "LOG_USER")
		|| !schedwi_strcasecmp (val, "USER"))
	{
		if (facility != NULL) {
			*facility = LOG_USER;
		}
		return 0;
	}
#endif
#ifdef LOG_AUTH
	if (	   !schedwi_strcasecmp (val, "LOG_AUTH")
		|| !schedwi_strcasecmp (val, "AUTH"))
	{
		if (facility != NULL) {
			*facility = LOG_AUTH;
		}
		return 0;
	}
#endif
#ifdef LOG_AUTHPRIV
	if (	   !schedwi_strcasecmp (val, "LOG_AUTHPRIV")
		|| !schedwi_strcasecmp (val, "AUTHPRIV"))
	{
		if (facility != NULL) {
			*facility = LOG_AUTHPRIV;
		}
		return 0;
	}
#endif
#ifdef LOG_CRON
	if (	   !schedwi_strcasecmp (val, "LOG_CRON")
		|| !schedwi_strcasecmp (val, "CRON"))
	{
		if (facility != NULL) {
			*facility = LOG_CRON;
		}
		return 0;
	}
#endif
#ifdef LOG_KERN
	if (	   !schedwi_strcasecmp (val, "LOG_KERN")
		|| !schedwi_strcasecmp (val, "KERN"))
	{
		if (facility != NULL) {
			*facility = LOG_KERN;
		}
		return 0;
	}
#endif
#ifdef LOG_LPR
	if (	   !schedwi_strcasecmp (val, "LOG_LPR")
		|| !schedwi_strcasecmp (val, "LPR"))
	{
		if (facility != NULL) {
			*facility = LOG_LPR;
		}
		return 0;
	}
#endif
#ifdef LOG_MAIL
	if (	   !schedwi_strcasecmp (val, "LOG_MAIL")
		|| !schedwi_strcasecmp (val, "MAIL"))
	{
		if (facility != NULL) {
			*facility = LOG_MAIL;
		}
		return 0;
	}
#endif
#ifdef LOG_NEWS
	if (	   !schedwi_strcasecmp (val, "LOG_NEWS")
		|| !schedwi_strcasecmp (val, "NEWS"))
	{
		if (facility != NULL) {
			*facility = LOG_NEWS;
		}
		return 0;
	}
#endif
#ifdef LOG_SYSLOG
	if (	   !schedwi_strcasecmp (val, "LOG_SYSLOG")
		|| !schedwi_strcasecmp (val, "SYSLOG"))
	{
		if (facility != NULL) {
			*facility = LOG_SYSLOG;
		}
		return 0;
	}
#endif
#ifdef LOG_UUCP
	if (	   !schedwi_strcasecmp (val, "LOG_UUCP")
		|| !schedwi_strcasecmp (val, "UUCP"))
	{
		if (facility != NULL) {
			*facility = LOG_UUCP;
		}
		return 0;
	}
#endif
#ifdef LOG_CONSOLE
	if (	   !schedwi_strcasecmp (val, "LOG_CONSOLE")
		|| !schedwi_strcasecmp (val, "CONSOLE"))
	{
		if (facility != NULL) {
			*facility = LOG_CONSOLE;
		}
		return 0;
	}
#endif
#ifdef LOG_FTP
	if (	   !schedwi_strcasecmp (val, "LOG_FTP")
		|| !schedwi_strcasecmp (val, "FTP"))
	{
		if (facility != NULL) {
			*facility = LOG_FTP;
		}
		return 0;
	}
#endif
#ifdef LOG_SECURITY
	if (	   !schedwi_strcasecmp (val, "LOG_SECURITY")
		|| !schedwi_strcasecmp (val, "SECURITY"))
	{
		if (facility != NULL) {
			*facility = LOG_SECURITY;
		}
		return 0;
	}
#endif
	return -1;
}


/*
 * Check if the string correspond to a syslog facility
 *
 * Return:
 *   NULL if yes or an error message if no
 */
const char *
check_syslog_facility (const char *s)
{
#if HAVE_ASSERT_H
	assert (s != NULL);
#endif

	if (string2facility (s, NULL) != 0){
		return _("unknown facility name (see syslog(3))");
	}
	return NULL;
}


/*
 * Does the string correspond to a boolean TRUE value?
 *
 * Return:
 *   0 --> No
 *  -1 --> Yes
 */
static char
is_bool_true (const char *value)
{
#if HAVE_ASSERT_H
	assert (value != NULL);
#endif

	if (	   value[0] == '1'
		|| value[0] == 'y' || value[0] == 'Y'		/* Yes */
		|| value[0] == 't' || value[0] == 'T'		/* True */
		|| schedwi_strncasecmp (value, "ON", 2) == 0)	/* On */
	{
		return 1;
	}
	return 0;
}

/*
 * Does the string correspond to a boolean FALSE value?
 *
 * Return:
 *   0 --> No
 *  -1 --> Yes
 */
static char
is_bool_false (const char *value)
{
#if HAVE_ASSERT_H
	assert (value != NULL);
#endif

	if (	   value[0] == '0'
		|| value[0] == 'n' || value[0] == 'N'		/* No */
		|| value[0] == 'f' || value[0] == 'F'		/* False */
		|| schedwi_strncasecmp (value, "OFF", 3) == 0)	/* Off */
	{
		return 1;
	}
	return 0;
}

/*
 * Check if the provided string contain a boolean value
 *
 * Return:
 *   NULL if yes or an error message if no
 */
const char *
check_bool (const char *s)
{
	if (is_bool_true (s) == 0 && is_bool_false (s) == 0) {
		return
	_("boolean value must be True, 1, Yes, On, False, 0, No or Off");
	}
	return NULL;
}

/*
 * Check if the provided string contain an integer
 *
 * Return:
 *   NULL if yes or an error message if no
 */
const char *
check_int (const char *s)
{
	char *ptr;

#if HAVE_ASSERT_H
	assert (s != NULL);
#endif

	(void)strtol (s, &ptr, 0);
	while (*ptr != 0 && isspace (*ptr) != 0) {
		ptr++;
	}
	if (*ptr == '\0' || *ptr == '#' || *ptr == ';') {
		return NULL;
	}
	return _("invalid integer value");
}

/*
 * Check if the provided string contain an valid directory name
 *
 * Return:
 *   NULL if yes or an error message if no
 */
const char *
check_dirname (const char *s)
{
	struct stat file_info;

#if HAVE_ASSERT_H
	assert (s != NULL);
#endif

	if (stat (s, &file_info) != 0) {
		return strerror (errno);
	}

#ifndef STAT_MACROS_BROKEN
	if (S_ISDIR (file_info.st_mode) == 0) {
		return _("the parameter value is not a directory name");
	}
#endif
	return NULL;
}

/*
 * Check if the provided string contain an valid address family (any, inet
 * or inet6)
 *
 * Return:
 *   NULL if yes or an error message if no
 */
const char *
check_addrfamily (const char *s)
{
#if HAVE_ASSERT_H
	assert (s != NULL);
#endif

	if (	   schedwi_strcasecmp (s, "any") == 0
		|| schedwi_strcasecmp (s, "inet") == 0
		|| schedwi_strcasecmp (s, "inet6") == 0)
	{
		return NULL;
	}
	return _("the parameter value should be `any', `inet' (use IPv4 only), or `inet6' (use IPv6 only)");
}

/*
 * Check if the provided network list is properly built.
 *
 * Return:
 *   NULL if yes or an error message if no
 */
const char *
check_netlist (const char *s)
{
#if HAVE_ASSERT_H
	assert (s != NULL);
#endif

	if (addr_match_list (NULL, s) != 0) {
		return _("Inconsistent mask length in one of the address");
	}
	return NULL;
}


/*
 * strtok (see strtok(3)).
 * Differences:
 *    - The delimiter is a character (and not a string)
 *    - Spaces at the begining and the end of the token are removed
 *    - Empty lines (two consecutives delimiters) are returned
 */
static char *
my_strtok (char *s, char delim)
{
	static char *save = NULL;
	char *end, *ret;

	if (s == NULL && save == NULL) {
		return NULL;
	}
	if (s != NULL) {
		save = s;
	}

	end = strchr (save, delim);
	ret = save;
	if (end != NULL) {
		*end = '\0';
		save = end + 1;
	}
	else {
		end = save + schedwi_strlen (save);
		save = NULL;
	}

	while (*ret != '\0' && isspace (*ret) != 0) {
		ret++;
	}

	for (--end; end > ret && isspace (*end) != 0; end--);
	end[1] = '\0';

	return ret;
}


/*
 * Split the provided module key name into the module name and the key.  The
 * format of the provided name should be:
 *     module:<module_name>:<key>
 * The provided mod_name and mod_key parameters will be set with respectively
 * the module name and the key value (mod_name and mod_key are allocated by
 * this function and must be freed by the caller with free())
 *
 * Return:
 *   0 --> No error. mod_name and mod_key are set and must be freed by the
 *         caller with free()
 *  -1 --> Memory allocation error
 *   1 --> Wrong format
 */
static int
split_module (	const char *name, size_t name_len,
		char **mod_name, char **mod_key)
{
	const char *mod_s;
	char *mod_k, *s, *k;
	size_t l;

#if HAVE_ASSERT_H
	assert (   name != NULL && name_len > 0
		&& mod_name != NULL && mod_key != NULL);
#endif

	if (	   name_len < 10
		|| schedwi_strncasecmp ("module:", name, 7) != 0)
	{
		return 1;
	}

	mod_s = name + 7;
	mod_k = strchr (mod_s, ':');
	if (	   mod_k == NULL
		|| mod_k[1] == '\0'
		|| isspace (mod_k[1]) != 0
		|| mod_k - name + 1 >= name_len)
	{
		return 1;
	}

	l = mod_k - mod_s;
	s = (char *) malloc (l + 1);
	if (s == NULL) {
		return -1;
	}
	strncpy (s, mod_s, l);
	s[l] = '\0';

	l = name_len - (mod_k - name) - 1;
	k = (char *) malloc (l + 1);
	if (k == NULL) {
		free (s);
		return -1;
	}
	strncpy (k, mod_k + 1, l);
	k[l] = '\0';

	*mod_name = s;
	*mod_key = k;
	return 0;
}


/*
 * Destroy (free) the parameter array
 */
void
conf_destroy ()
{
	int i;

	if (params != NULL) {
		for (i = 0;  params[i].name != NULL; i++) {
			if (	   params[i].type == STRING
				&& params[i].value.value_string != NULL)
			{
				free (params[i].value.value_string);
				params[i].value.value_string = NULL;
			}
			params[i].is_set = 0;
		}
		params = NULL;
	}
}


/*
 * Fill the parameter array by reading the provided configuration file.  If
 * file_name is NULL, no configuration  file is read; the params_array
 * object is stored internally
 *
 * Return:
 *   0 --> No error
 *  -1 --> Memory allocation error
 *  -2 --> System error while accessing the configuration file (errno is set)
 *  -3 --> Syntax error in the configuration file.  The errors are sent to
 *         stderr
 */
int
conf_init (const char *file_name, conf_param_t *params_array)
{
	FILE *fd;
	struct stat file_info;
	char *file_content, *save_file_content;
	char *line, *name, *value, *save_line, *mod_name, *mod_key;
	const char *err_msg;
	size_t nb_read, name_len;
	int save_errno, num_line, nb_errors, i, ret;
	long int facility;

#if HAVE_ASSERT_H
	assert (params_array != NULL);
#endif

	if (file_name == NULL) {
		params = params_array;
		return 0;
	}

	if (stat (file_name, &file_info) != 0) {
		return -2;
	}

	/* Buffer to read all the file in once */
	file_content = (char *) malloc (file_info.st_size + 1);
	if (file_content == NULL) {
		return -1;
	}

	fd = fopen (file_name, "r");
	if (fd == NULL) {
		free (file_content);
		return -2;
	}

	clearerr (fd);
	nb_read = fread (file_content, 1, file_info.st_size, fd);
	if (nb_read != file_info.st_size && ferror (fd) != 0) {
		save_errno = errno;
		free (file_content);
		fclose (fd);
		errno = save_errno;
		return -2;
	}
	file_content[nb_read] = '\0';

	fclose (fd);

	conf_destroy ();
	params = params_array;
	num_line = nb_errors = 0;
	save_file_content = file_content;
	while ((line = my_strtok (file_content, '\n')) != NULL) {
		file_content = NULL;
		num_line++;
		save_line = line;

		/* Empty line or comment */
		if (*line == '\0' || *line == '#' || *line == ';') {
			continue;
		}

		/* Parameter name */
		name = line;
		while (*line != '\0' && *line != '=' && isspace (*line) == 0) {
			line++;
		}

		name_len = line - name;
		if (name_len == 0) {
			/* ie. '= value' (missing parameter name) */
			nb_errors++;
			fprintf (stderr, _("%s: syntax error on line %d: "),
					file_name, num_line);
			fputs (_("missing parameter name:\n"), stderr);
			fputs (save_line, stderr);
			fputc ('\n', stderr);
			continue;
		}

		while (*line != '\0' && isspace (*line) != 0) {
			line++;
		}

		if (*line != '=') {
			/* ie. 'name  value' (missing '=') */
			nb_errors++;
			fprintf (stderr, _("%s: syntax error on line %d: "),
					file_name, num_line);
			fputs (_("missing `=' sign between name and value:\n"),
					stderr);
			fputs (save_line, stderr);
			fputc ('\n', stderr);
			continue;
		}
		line++;

		/* Skip the spaces in front of the value */
		while (*line != '\0' && isspace (*line) != 0) {
			line++;
		}

		if (*line == '\0' || *line == '#' || *line == ';') {
			/* ie. 'name =' (missing value) */
			nb_errors++;
			fprintf (stderr, _("%s: syntax error on line %d: "),
					file_name, num_line);
			fputs (_("missing value:\n"), stderr);
			fputs (save_line, stderr);
			fputc ('\n', stderr);
			continue;
		}

		value = line;

		/* Find the parameter name */
		for (i = 0;  params[i].name != NULL; i++) {
			if (schedwi_strncasecmp (	params[i].name,
							name, name_len) == 0)
			{
				break;
			}
		}

		if (params[i].name == NULL) {
			/*
			 * The parameter is not a core variable.  Check if
			 * its a module variable
			 * (i.e. module:<modname>:<varname>)
			 */
			switch (split_module (	name, name_len,
						&mod_name, &mod_key))
			{
				case -1:
					free (save_file_content);
					conf_destroy ();
					return -1;
				case 1:
					nb_errors++;
					fprintf (stderr,
				_("%s: unknown parameter name on line %d:\n"),
						file_name, num_line);
					fputs (save_line, stderr);
					fputc ('\n', stderr);
					continue;
			}

			ret = module_conf (mod_name, mod_key, value);
			free (mod_name);
			free (mod_key);
			if (ret != 0) {
				nb_errors++;
			}
			continue;
		}

		/* Check the parameter */
		if (params[i].check != NULL) {
			err_msg = (params[i].check)(value);
			if (err_msg != 0) {
				nb_errors++;
				fprintf (stderr,
			_("%s: erroneous value parameter on line %d: "),
					file_name, num_line);
				fputs (err_msg, stderr);
				fputs (":\n", stderr);
				fputs (save_line, stderr);
				fputc ('\n', stderr);
				continue;
			}
		}

		/* Save the parameter */
		switch (params[i].type) {
			case INT:
				params[i].value.value_int = strtol (	value,
									NULL,
									0);
				break;

			case BOOL:
				params[i].value.value_bool = is_bool_true (value);
				break;

			case STRING:
				params[i].value.value_string = (char *) malloc (
						schedwi_strlen (value) + 1);
				if (params[i].value.value_string == NULL) {
					free (save_file_content);
					conf_destroy ();
					return -1;
				}
				strcpy (params[i].value.value_string, value);
				break;
			case SYSLOG_FACILITY:
				facility = -1;
				string2facility (value, &facility);
				params[i].value.value_int = facility;
				break;
			case FUNCTION:
				if (	   params[i].set != NULL
					&& (*(params[i].set)) (value) != 0)
				{
					free (save_file_content);
					conf_destroy ();
					return -3;
				}
				break;
		}
		params[i].is_set = 1;
	}
	free (save_file_content);
	if (nb_errors != 0) {
		conf_destroy ();
		return -3;
	}
	return 0;
}

/*
 * Get the value associated with the provided parameter name
 *
 * Return:
 *   0 --> If not NULL, value is set with the parameter value
 *  -1 --> Unknown parameter name (or wrong type)
 */
int
conf_get_param_string (const char *name, const char **value)
{
	int i;

	if (params == NULL) {
		return -1;
	}

	for (i = 0;  params[i].name != NULL; i++) {
		if (	   params[i].type == STRING
			&& schedwi_strcasecmp (params[i].name, name) == 0)
		{
			if (value != NULL) {
				if (params[i].is_set == 0) {
					*value = params[i].default_value;
				}
				else {
					*value = params[i].value.value_string;
				}
			}
			return 0;
		}
	}
	return -1;
}

/*
 * Get the value associated with the provided parameter name
 *
 * Return:
 *   0 --> If not NULL, value is set with the parameter value
 *  -1 --> Unknown parameter name (or wrong type)
 */
int
conf_get_param_int (const char *name, long int *value)
{
	int i;

	if (params == NULL) {
		return -1;
	}

	for (i = 0;  params[i].name != NULL; i++) {
		if (	   params[i].type == INT
			&& schedwi_strcasecmp (params[i].name, name) == 0)
		{
			if (value != NULL) {
				if (params[i].is_set == 0) {
					if (params[i].default_value == NULL) {
						*value = 0;
					}
					else {
						*value = strtol (
						params[i].default_value,
						NULL, 0);
					}
				}
				else {
					*value = params[i].value.value_int;
				}
			}
			return 0;
		}
	}
	return -1;
}

/*
 * Get the value associated with the provided parameter name
 *
 * Return:
 *   0 --> If not NULL, value is set with the parameter value
 *  -1 --> Unknown parameter name (or wrong type)
 */
int
conf_get_param_bool (const char *name, char *value)
{
	int i;

	if (params == NULL) {
		return -1;
	}

	for (i = 0;  params[i].name != NULL; i++) {
		if (	   params[i].type == BOOL
			&& schedwi_strcasecmp (params[i].name, name) == 0)
		{
			if (value != NULL) {
				if (params[i].is_set == 0) {
					if (params[i].default_value == NULL) {
						*value = 0;
					}
					else {
						*value = is_bool_true (
						params[i].default_value);
					}
				}
				else {
					*value = params[i].value.value_bool;
				}
			}
			return 0;
		}
	}
	return -1;
}

/*
 * Get the value associated with the provided parameter name
 *
 * Return:
 *   0 --> If not NULL, value is set with the parameter value
 *  -1 --> Unknown parameter name (or wrong type)
 */
int
conf_get_param_syslog_facility (const char *name, long int *value)
{
	int i;

	if (params == NULL) {
		return -1;
	}

	for (i = 0;  params[i].name != NULL; i++) {
		if (	   params[i].type == SYSLOG_FACILITY
			&& schedwi_strcasecmp (params[i].name, name) == 0)
		{
			if (value != NULL) {
				if (params[i].is_set == 0) {
					if (params[i].default_value == NULL) {
						*value = -1;
					}
					else {
						*value = -1;
						string2facility (
						params[i].default_value,
						value);
					}
				}
				else {
					*value = params[i].value.value_int;
				}
			}
			return 0;
		}
	}
	return -1;
}

/*------------------------======= End Of File =======------------------------*/
