/* Schedwi
   Copyright (C) 2013 Herve Quatremain

   This file is part of Schedwi.

   Schedwi is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 3 of the License, or
   (at your option) any later version.

   Schedwi is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

/* manual_trigger.c -- Alert someone that a manual job is waiting */

#include <schedwi.h>

#if STDC_HEADERS
#include <stdlib.h>
#include <string.h>
#else
#if HAVE_STDLIB_H
#include <stdlib.h>
#endif
#if HAVE_STRING_H
#include <string.h>
#endif
#endif

#if HAVE_ERRNO_H
#include <errno.h>
#endif
#ifndef errno
extern int errno;
#endif

#if HAVE_STDIO_H
#include <stdio.h>
#endif

#if HAVE_UNISTD_H
#include <unistd.h>
#endif

#if HAVE_SYS_STAT_H
#include <sys/stat.h>
#endif

#if HAVE_SYS_TYPES_H
#include <sys/types.h>
#endif

#if HAVE_ASSERT_H
#include <assert.h>
#endif

#include <lib_functions.h>
#include <lwc_log.h>
#include <cert_utils.h>
#include <utils.h>
#include <schedwi_time.h>
#include <schedwi_system.h>
#include <sql_acknowledge_manual.h>
#include <conf.h>
#include <manual_trigger.h>


/*
 * Replace all occurences of keywords in a file.
 *
 * @param file_in:
 *        The source file in which keywords will be looked for.  This file
 *        is not modified.
 * @param fd_out:
 *        Output file descriptor of the output file.
 * @param keys:
 *        List of keywords to look for.  In `file_in' this keywords must
 *        be surrounded by %'s. For instance, for a SCHEDWI_JOBID keyword, all
 *        the occurences of %SCHEDWI_JOBID% in the source file will be replaced
 *        in `fd_out' by the associated value.
 * @param values:
 *        List of values.  `keys' and `values' work together.  The value
 *        at position 1 in associated with the keyword at the same position.
 * @param len:
 *        Number of keywords/values in the `keys' and `values' arrays.
 * @return:
 *        0 in case of success or -1 in case of failure (in this case,
 *        an error message has already been logged by lwc_writeLog())
 */
static int
my_sed(	const char *file_in, int fd_out,
	const char * const *keys, const char * const *values, size_t len)
{
	char *file_content, *s, *d, *k, *save_s;
	size_t length;
	size_t *values_len;
	char **match_keys;
	size_t l, i, j;

#ifdef HAVE_ASSERT_H
	assert (file_in != NULL && file_in[0] != '\0' && fd_out >= 0);
#endif

	/* Store the whole content of the input file in a string */
	length = 0;
	file_content = read_file (file_in, &length);
	if (file_content == NULL) {
		return -1;
	}

	if (len == 0) {
		/* No keys/values. Just copy the file */
		(void)write (fd_out, file_content, length);
		free (file_content);
		return 0;
	}

	/* Array to store the length of each value */
	values_len = (size_t *) malloc (sizeof (size_t) * len);
	if (values_len == NULL) {
		free (file_content);
		lwc_writeLog (LOG_CRIT, _("Memory allocation error"));
		return -1;
	}

	/* Array to store the keywords to look for (with surrounding %'s) */
	match_keys = (char **) malloc (sizeof (char *) * len);
	if (match_keys == NULL) {
		free (values_len);
		free (file_content);
		lwc_writeLog (LOG_CRIT, _("Memory allocation error"));
		return -1;
	}

	for (i = 0; i < len; i++) {
		values_len[i] = (size_t) schedwi_strlen (values[i]);
		l = (size_t) schedwi_strlen (keys[i]) + 2;
		match_keys[i] = (char *) malloc (l + 1);
		if (match_keys[i] == NULL) {
			for (j = 0; j < i; j++) {
				free (match_keys[j]);
			}
			free (match_keys);
			free (values_len);
			free (file_content);
			lwc_writeLog (LOG_CRIT, _("Memory allocation error"));
			return -1;
		}
		/* The key to search for is surrounded by `%' (ie. %JOBID%) */
		match_keys[i][0] = '%';
		strcpy (match_keys[i] + 1, keys[i]);
		match_keys[i][l - 1] = '%';
		match_keys[i][l] = '\0';

		/*
		 * Compute the size of the final array (the output string)
		 * by finding the number of occurences of the keyword to
		 * substitute.
		 */
		s = file_content;
		while ((s = strstr(s, match_keys[i])) != NULL) {
			length += values_len[i];
			s += l;
		}
	}

	/* Do the substitutions of each keyword */
	s = file_content;
	for (i = 0; i < len; i++) {
		save_s = s;
		d = (char *) malloc (length);
		if (d == NULL) {
			for (; i < len; i++) {
				free (match_keys[i]);
			}
			free (match_keys);
			free (values_len);
			free (save_s);
			lwc_writeLog (LOG_CRIT, _("Memory allocation error"));
			return -1;
		}
		l = 0;
		while ((k = strstr (s, match_keys[i])) != NULL) {
			strncpy (d + l, s, (size_t)(k - s));
			l += k - s;
			strcpy (d + l, values[i]);
			l += values_len[i];
			s = k + schedwi_strlen (match_keys[i]);
		}
		strcpy (d + l, s);
		s = d;
		free (save_s);
		free (match_keys[i]);
	}
	free (match_keys);
	free (values_len);

	/* Write the result to the output file */
	(void)write (fd_out, s, (size_t)schedwi_strlen (s));
	free (s);
	return 0;
}


/*
 * Generate a random string.
 *
 * @param num_chars:
 *	Number of characters in the returned string.
 * @return:
 *	The null-terminated random string to be freed by the caller by free()
 *	or NULL in case of memory allocation error.
 */
static char *
rand_string (unsigned int num_chars)
{
	const char *set[3] =	{	"bcdfghjkmnpqrstvwxz",
					"aeiouy",
					"23456789-"
				};
	char *s;
	unsigned int i;
	unsigned int l[3];

	l[0] = schedwi_strlen(set[0]);
	l[1] = schedwi_strlen(set[1]);
	l[2] = schedwi_strlen(set[2]);

	s = (char *) malloc (num_chars + 1);
	if (s == NULL) {
		return NULL;
	}

	for (i = 0; i < num_chars - 1; i++) {
		s[i] = set[i % 2][rand () % l[i % 2]];
	}
	s[num_chars - 1 ] = set[2][rand () % l[2]];
	s[num_chars] = '\0';
	return s;
}


/*
 * Generate a template string for the mkstemp() function.
 *
 * @return:
 *       the string (to be freed by the caller by free()) that contains the
 *       template or NULL in case of memory allocation error.
 */
static
char *get_template_for_mkstemp()
{
	char *tmp, *s;

#if HAVE_GETENV
	tmp = getenv ("TMPDIR");
	if (tmp == NULL) {
		tmp = "/tmp";
	}
#else
	tmp = "/tmp";
#endif

	s = (char *) malloc (  schedwi_strlen (tmp) + schedwi_strlen (DIR_SEP)
			     + 15);
	if (s == NULL) {
		return NULL;
	}
	strcpy (s, tmp);
	strcat (s, DIR_SEP);
	strcat (s, "schedwiXXXXXX");
	return s;
}


/*
 * Error callback function for sql_acknowledge_get()
 */
static void
sql_get_error_logger (void *data, const char *msg, int err_code)
{
	if (msg != NULL) {
		lwc_writeLog (LOG_ERR, msg);
	}
	else {
		lwc_writeLog (LOG_ERR,
_("Database error while trying to retrieve manual status of a job/jobset"));
	}
}


/*
 * Check if the provided job needs a manual validation to start.  If yes,
 * check that it has been validated.  If no, add an entry in the
 * acknowledge_manual database table (with the urn and password to use
 * to acknowledge the message from the Web interface) and execute the
 * command to alert operators (job->manual_command)
 *
 * @param workload_date:
 * 	The workload date of the job.
 * @param job:
 *	Job/jobset details.
 * @return:
 *	0 if the job can be started (it's not a manual job or it has been
 *	validated.
 *      1 if the job is waiting for validation and should not start yet.
 *	-1 in case of error (an error message has already been logged with
 *      lwc_writeLog())
 */
int
check_manual_trigger (int workload_date, schedwi_jobtree_node_ptr job)
{
	const char *template_file;
	int status, i, ret;
	schedwi_time t, ack_time;
	char *username;
	char *password, *urn, *s, *err_msg, *tmp_file;
	const char *keys[6] = {	"SCHEDWI_JOBPATH", "SCHEDWI_JOBID",
				"SCHEDWI_START_TIME", "SCHEDWI_TIME_LIMIT",
				"SCHEDWI_URN", "SCHEDWI_PASSWORD" };
        char *values[6];
	size_t num_keys = 6;  /* Keep in sync with the num of items in keys! */
	char *variable_system[1];


	if (job == NULL || job->manual == 0) {
		return 0;
	}

	/* Retrieve the acknowledge status from the database */
	username = NULL;
	if (sql_acknowledge_get (workload_date, job->id,  &status,
				 &ack_time, &username,
				 sql_get_error_logger, NULL) != 0)
	{
		return -1;
	}

	/* Not yet acknowledged */
	if (status == 0) {
		if (username != NULL) {
			free (username);
		}
		return 1;
	}

	/* Acknowledged */
	if (status == 1) {
		s = schedwi_time_strftime (SCHEDWI_DEFAULT_DATEFORMAT,
						  ack_time);
		if (s == NULL) {
			lwc_writeLog (LOG_CRIT, _("Memory allocation error"));
			/* Memory error but we know that the job is ack */
			return 0;
		}
		lwc_writeLog (	LOG_INFO,
	_("Workload %d: %s (id %lld): start acknowledged by %s at %s"),
				workload_date, job->path, job->id,
				(username != NULL && username[0] != '\0') ?
					username : _("unkwnown"),
				s);
		free (s);
		if (username != NULL) {
			free (username);
		}
		return 0;
	}

	/* No entry in the database.  Create one */
	lwc_writeLog (	LOG_INFO,
_("Workload %d: %s (id %lld): manual %s ready: requesting acknowledgement"),
			workload_date, job->path, job->id,
			(job->node_type == 0) ? _("jobset") : _("job"));

	/* Add a new entry in the database */
	password = rand_string(6);
	if (password == NULL) {
		lwc_writeLog (LOG_CRIT, _("Memory allocation error"));
		return -1;
	}
	s = rand_string(6);
	if (s == NULL) {
		free (password);
		lwc_writeLog (LOG_CRIT, _("Memory allocation error"));
		return -1;
	}
	urn = (char *) malloc (20);   /* schedwi-<s> (s is 6 chars long) */
	if (urn == NULL) {
		free (s);
		free (password);
		lwc_writeLog (LOG_CRIT, _("Memory allocation error"));
		return -1;
	}
	strcpy (urn, "schedwi-");
	strcat (urn, s);
	free (s);

	err_msg = NULL;
	ret = sql_acknowledge_add (	workload_date, job->id, urn, password,
					&err_msg);
	if (ret != 0) {
		free (urn);
		free (password);
		if (err_msg != NULL) {
			lwc_writeLog (LOG_CRIT, err_msg);
			free (err_msg);
		}
		else {
			lwc_writeLog (	LOG_CRIT,
_("Workload %d: %s (id %lld): cannot add the acknowledgement req in database"),
			workload_date, job->path, job->id);
		}
		return -1;
	}

	if (job->manual_command == NULL || (job->manual_command)[0] == '\0') {
		free (urn);
		free (password);
		return 1;
	}

	/* Convert the template */
	ret = conf_get_param_string (	"MANUAL_TRIGGER_TEMPLATE_FILE",
					&template_file);
#if HAVE_ASSERT_H
	assert (ret == 0);
#endif
	if (access (template_file, R_OK) != 0) {
		lwc_writeLog (	LOG_INFO, _("%s: %s"),
				template_file, strerror (errno));
		free (urn);
		free (password);
		tmp_file = NULL;
		i = 0;
	}
	else {
		/* SCHEDWI_JOBPATH */
		values[0] = (char *) malloc (schedwi_strlen (job->path) + 1);
		if (values[0] == NULL) {
			free (urn);
			free (password);
			lwc_writeLog (LOG_CRIT, _("Memory allocation error"));
			return -1;
		}
		strcpy (values[0], job->path);
			
		/* SCHEDWI_JOBID */
		values[1] = (char *) malloc (100);
		if (values[1] == NULL) {
			free (values[0]);
			free (urn);
			free (password);
			lwc_writeLog (LOG_CRIT, _("Memory allocation error"));
			return -1;
		}
		sprintf (values[1], "%d_%lld", workload_date, job->id);

		/* SCHEDWI_START_TIME */
		values[2] = schedwi_time_strftime ("%H:%M", job->run_time);
		if (values[2] == NULL) {
			free (values[1]);
			free (values[0]);
			free (urn);
			free (password);
			lwc_writeLog (LOG_CRIT, _("Memory allocation error"));
			return -1;
		}
	
		/* SCHEDWI_TIME_LIMIT */
		if (job->start_limit > 0) {
			if (job->run_time > 0) {
				t = job->run_time;
			}
			else {
				if (job->parent == NULL) {
					/* Should not be possible */
					t = job->run_time;
				}
				else {
					t = job->parent->start_time; 
				}
			}

			/*
			 * job->start_limit - 1 because if it's 24 hours the
			 * displayed HH:MM will show the current hour/min
			 * and this may confuse the user.
			 */
			values[3] = schedwi_time_strftime ("%H:%M",
				schedwi_time_add_seconds (t,
							job->start_limit - 1));
			if (values[3] == NULL) {
				free (values[2]);
				free (values[1]);
				free (values[0]);
				free (urn);
				free (password);
				lwc_writeLog (	LOG_CRIT,
						_("Memory allocation error"));
				return -1;
			}
		}
		else {
			values[3] = (char *) malloc (100);
			if (values[3] == NULL) {
				free (values[2]);
				free (values[1]);
				free (values[0]);
				free (urn);
				free (password);
				lwc_writeLog (	LOG_CRIT,
						_("Memory allocation error"));
				return -1;
			}
			strncpy (values[3], _("no limit"), 99);
			values[3][99] = '\0';
		}
	
		/* SCHEDWI_URN */
		values[4] = urn;
	
		/* SCHEDWI_PASSWORD */
		values[5] = password;

		/* Get a temporary file name to convert the template */
		tmp_file = get_template_for_mkstemp();
		if (tmp_file == NULL) {
			for (i = 0; i < num_keys; i++) {
				free (values[i]);
			}
			lwc_writeLog (LOG_CRIT, _("Memory allocation error"));
			return -1;
		}

		i = mkstemp (tmp_file);
		if (i < 0) {
			lwc_writeLog (	LOG_ERR, _("%s: %s"),
					tmp_file, strerror (errno));
			free (tmp_file);
			for (i = 0; i < num_keys; i++) {
				free (values[i]);
			}
			return -1;
		}

		ret = my_sed (	template_file, i, keys,
				(const char * const *)values, num_keys);
		close (i);
		for (i = 0; i < num_keys; i++) {
			free (values[i]);
		}
		if (ret != 0) {
			my_unlink (tmp_file);
			free (tmp_file);
			return -1;
		}

		variable_system[0] = (char *) malloc (
					  schedwi_strlen ("SCHEDWI_TEMPLATE")
					+ schedwi_strlen (tmp_file) + 2);
		if (variable_system[0] == NULL) {
			my_unlink (tmp_file);
			free (tmp_file);
			return -1;
		}
		strcpy (variable_system[0], "SCHEDWI_TEMPLATE");
		strcat (variable_system[0], "=");
		strcat (variable_system[0], tmp_file);
		i = 1;
	}

	/* Execute the command to alert of a waiting manual job */
	ret = schedwi_system (	job->manual_command, workload_date, job,
				(const char * const *)variable_system, i);
	if (i == 1) {
		free (variable_system[0]);
	}
	if (tmp_file != NULL) {
		my_unlink (tmp_file);
		free (tmp_file);
	}
	switch (ret) {
		case 0: break;
		case 127:
			lwc_writeLog (	LOG_ERR,
			_("Workload %d: %s (id %lld): %s: failed to start"),
					workload_date, job->path,
					job->id, job->manual_command);
			break;
		case 300:
			lwc_writeLog (	LOG_ERR,
			_("Workload %d: %s (id %lld): %s: killed"),
					workload_date, job->path,
					job->id, job->manual_command);
			break;
		case -1:
			lwc_writeLog (	LOG_CRIT,
			_("Workload %d: %s (id %lld): %s: system error"),
					workload_date, job->path,
					job->id, job->manual_command);
			break;
		default:
			lwc_writeLog (	LOG_ERR,
			_("Workload %d: %s (id %lld): %s: exit code: %d"),
					workload_date, job->path,
					job->id, job->manual_command, ret);
			break;
	}
	return 1;
}

/*-----------------============== End Of File ==============-----------------*/
