/* Schedwi
   Copyright (C) 2013 Herve Quatremain

   This file is part of Schedwi.

   Schedwi is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 3 of the License, or
   (at your option) any later version.

   Schedwi is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

/* control_groups_linux.c -- Associate a process to a Control Group */


#include <schedwi.h>

#if HAVE_LIBCGROUP

#if STDC_HEADERS
#include <stdlib.h>
#include <string.h>
#else
#if HAVE_STDLIB_H
#include <stdlib.h>
#endif
#if HAVE_STRING_H
#include <string.h>
#endif
#endif

#if HAVE_SYS_STAT_H
#include <sys/stat.h>
#endif

#if HAVE_UNISTD_H
#include <unistd.h>
#endif

#include <libcgroup.h>

#include <lib_functions.h>
#include <control_groups_linux.h>



/*
 * Add a process to a Linux Control Group.
 *    cgroup_name: Control Group name relative to the cgroup mount point
 *                 (for instance, for the mount point /cgroup/memory/test -
 *                 for the memory controler - cgroup_name must be test.
 *                 For /cgroup/cpu/schedwi/test - for the cpu controller -
 *                 cgroup_name must be schedwi/test)
 *            pid: PID of the process
 *
 * Return:
 *        0 --> No error
 *    other --> Error
 */
#if HAVE_PID_T
int
cgroup_set (const char *cgroup_name, pid_t pid)
#else
int
cgroup_set (const char *cgroup_name, int pid)
#endif
{
	int ret;
	void *handle;
	struct cgroup_mount_point info;
	char *s;
	const char *ctrls[2];
	unsigned int len_cgroup_name, len_dir_sep;
        struct stat buf;


	if (cgroup_name == NULL || cgroup_name[0] == '\0') {
		return 0;
	}

	ret = cgroup_init ();
	if (ret != 0) {
		/*
		fprintf (stderr, "libcgroup init failed: %s\n",
			 cgroup_strerror(ret));
		*/
		return ret;
	}

	/*
	 * Scan all the mount points/controllers to look for the specified
	 * Control Group name.
	 */
	ctrls[1] = NULL;
	len_cgroup_name = schedwi_strlen (cgroup_name);
	len_dir_sep = schedwi_strlen (DIR_SEP);
	ret = cgroup_get_controller_begin (&handle, &info);
	if (ret && ret != ECGEOF) {
		/*
		fprintf (stderr, "cannot read controller data: %s\n",
			 cgroup_strerror(ret));
		*/
		return ret;
	}
	while (ret != ECGEOF) {
		s = (char *) malloc (  schedwi_strlen (info.path)
				     + len_dir_sep + len_cgroup_name + 1);
		if (s == NULL) {
			cgroup_get_controller_end (&handle);
			return -1;
		}

		strcpy (s, info.path);
		strcat (s, DIR_SEP);
		strcat (s, cgroup_name);
		if (stat (s, &buf) == 0 && S_ISDIR (buf.st_mode)) {
			ctrls[0] = info.name;
			cgroup_change_cgroup_path (cgroup_name, pid, ctrls);
			/*
			 * Don't care about the return code because
			 * it's not going to be reported.
			 *
			ret = cgroup_change_cgroup_path (cgroup_name, pid,
							 ctrls);
			if (ret != 0) {
				fprintf (stderr, "cannot change cgroup: %s\n",
					 cgroup_strerror(ret));
				cgroup_get_controller_end (&handle);
				return ret;
			}
			*/
		}
		free (s);
		ret = cgroup_get_controller_next (&handle, &info);
		if (ret && ret != ECGEOF) {
			/*
			fprintf(stderr,
				"cgroup_get_controller_next failed (%s)\n",
				cgroup_strerror(ret));
			*/
			cgroup_get_controller_end (&handle);
			return ret;
		}
	}
	cgroup_get_controller_end (&handle);
	return 0;
}

#endif /* HAVE_LIBCGROUP */

/*-----------------============== End Of File ==============-----------------*/
