/* Schedwi
   Copyright (C) 2007-2015 Herve Quatremain

   This file is part of Schedwi.

   Schedwi is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 3 of the License, or
   (at your option) any later version.

   Schedwi is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

#include <schedwi.h>

#if STDC_HEADERS
#include <stdlib.h>
#include <string.h>
#endif

#if HAVE_SYS_TIME_H
#include <sys/time.h>
#endif

#if HAVE_SYS_RESOURCE_H
#include <sys/resource.h>
#endif

#if HAVE_SYS_WAIT_H
# include <sys/wait.h>
#endif

#if HAVE_UNISTD_H
#include <unistd.h>
#endif

#if HAVE_ERRNO_H
#include <errno.h>
#endif
#ifndef errno
extern int errno;
#endif

#if HAVE_FCNTL_H
#include <fcntl.h>
#endif

#if HAVE_ASSERT_H
#include <assert.h>
#endif

#include <job_run.h>
#include <job_parameters.h>
#include <job_launcher_main.h>
#include <utils.h>
#include <net_module.h>
#include <net_utils_init.h>
#include <lib_functions.h>
#include <control_groups_linux.h>
#include <lwc_log.h>
#include <job_launcher.h>


/*
 * System calls which trigger an error in the child process
 */
typedef enum
{
	CHILD_NOERROR = 0,
	CHILD_SETPRIORITY,
	CHILD_SETRLIMIT,
	CHILD_SETUID,
	CHILD_SETEUID,
	CHILD_SETGID,
	CHILD_SETEGID,
	CHILD_CHDIR,
	CHILD_OPEN,
	CHILD_EXEC,
	CHILD_PIPE,
	CHILD_FORK
} child_errcode_t;


/*
 * Compose the error message.  Return this message in *out (which must
 * be freed by the caller).
 */
static void
build_errmsg_errno (int errnum, const char *msg, char **out)
{
	char *strerr;


#if HAVE_ASSERT_H
	assert (msg != NULL);
#endif

	if (out != NULL) {
		strerr = strerror (errnum);
		*out = (char *)malloc (   schedwi_strlen (msg)
					+ schedwi_strlen (strerr) + 1);
		if (*out != NULL) {
			strcpy (*out, msg);
			strcat (*out, strerr);
		}
	}
}


/*
 * Compose the error message.  Return this message in *out (which must
 * be freed by the caller).
 */
static void
build_errmsg (const char *msg1, const char *msg2, char **out)
{
#if HAVE_ASSERT_H
	assert (msg2 != NULL);
#endif

	if (out != NULL) {
		*out = (char *)malloc (   ((msg1 != NULL) ?
						schedwi_strlen (msg1) : 0)
					+ schedwi_strlen (msg2)
					+ schedwi_strlen (": ")
					+1);
		if (*out != NULL) {
			(*out)[0] = '\0';
			if (msg1 != NULL) {
				strcpy (*out, msg1);
				strcat (*out, ": ");
			}
			strcat (*out, msg2);
		}
	}
}


/*
 * Read the status of a child from the provided pipe descriptor.
 * If the child fails, it will write to this pipe a errno value and the
 * function which failed. If it is successful it will simply close the pipe.
 * This function will read the pipe (if it can't then it's because the child
 * close the pipe and then everithing if successful) and set err_msg (if not
 * NULL) to the corresponding error message.
 *
 * Return:
 *   0 --> No error (the child is running)
 *  -1 --> Error (errno is set and err_msg, if not NULL, contains an error
 *         message
 */
#if HAVE_PID_T
static int
get_child_status (int pipe_fd, pid_t child, char **err_msg)
#else
static int
get_child_status (int pipe_fd, int child, char **err_msg)
#endif
{
	ssize_t nb_read;
	int result_errno;
	child_errcode_t child_err;


	/* Read the error code sent by the child */
	nb_read = read (pipe_fd, &result_errno, sizeof (int));
	if (nb_read > 0) {
		nb_read = read (pipe_fd, &child_err, sizeof (child_errcode_t));
	}

	/* No error,  the job is running */
	if (nb_read == -1 || nb_read == 0) {
		return 0;
	}

	/* Error */
	waitpid (child, NULL, 0);
	switch (child_err) {
		case CHILD_SETPRIORITY:
			build_errmsg_errno (result_errno,
	_("Failed to change the scheduling priority (nice): setpriority: "),
						err_msg);
			break;
		case CHILD_SETRLIMIT:
			build_errmsg_errno (result_errno,
			_("Failed to set the file size limit: setrlimit: "),
						err_msg);
			break;
		case CHILD_SETUID:
			build_errmsg_errno (result_errno,
				_("Failed to change the user ID: setuid: "),
				err_msg);
			break;
		case CHILD_SETEUID:
			build_errmsg_errno (result_errno,
				_("Failed to change the user ID: seteuid: "),
				err_msg);
			break;
		case CHILD_SETGID:
			build_errmsg_errno (result_errno,
				_("Failed to change the group ID: setgid: "),
				err_msg);
			break;
		case CHILD_SETEGID:
			build_errmsg_errno (result_errno,
				_("Failed to change the group ID: setegid: "),
				err_msg);
			break;
		case CHILD_CHDIR:
			build_errmsg_errno (result_errno,
			_("Failed to change the working directory: chdir: "),
						err_msg);
			break;
		case CHILD_OPEN:
			build_errmsg_errno (result_errno,
				_("Failed to open the output files: open: "),
				err_msg);
			break;
		case CHILD_EXEC:
			build_errmsg_errno (result_errno,
				_("Failed to start the command: execve: "),
				err_msg);
			break;
		case CHILD_PIPE:
			build_errmsg_errno (result_errno, "pipe: ", err_msg);
			break;
		case CHILD_FORK:
			build_errmsg_errno (result_errno,
					_("Failed to start the job: fork: "),
					err_msg);
			break;
		default:
				break;
	}
	errno = result_errno;
	return -1;
}


/*
 * Start the job manager (launcher).  It will start the job (fork and exec)
 * and manage its return code.
 */
static void
launcher (	job_run_t *j, int status_pipe_father[2], const char *job_id,
		const char *job_name)
{
#if HAVE_PID_T
	pid_t child;
#else
	int child;
#endif
	int result_errno;
	int status_pipe_child[2], out, err;
	child_errcode_t child_err;
	ssize_t nb_read;
	char open_log;
#if HAVE_SETRLIMIT
	struct rlimit rlim;
#endif
#if HAVE_GETPRIORITY && HAVE_SETPRIORITY
	int old_nice;
#endif


#if HAVE_ASSERT_H
	assert (j != NULL);
#endif

	/* Clean the process */
	destroy_modules ();
	net_free ();
	open_log = (char) lwc_logIsFile ();
	sanitise (status_pipe_father[1]);
#if HAVE_SETSID
	setsid();
#endif

	/*
	 * A pipe is opened between this process and its child.  If something
	 * goes wrong in the child (fail to exec, to change the user id, ...)
	 * the child writes the errno code of the error in the pipe so this
	 * process can report the error to its father.
	 */
	if (pipe (status_pipe_child) != 0) {
		write (status_pipe_father[1], &errno, sizeof (int));
		child_err = CHILD_PIPE;
		write (status_pipe_father[1], &child_err,
						sizeof (child_errcode_t));
		close (status_pipe_father[1]);
		exit (1);
	}

	child = fork ();
	if (child < 0) {
		write (status_pipe_father[1], &errno, sizeof (int));
		child_err = CHILD_FORK;
		write (status_pipe_father[1], &child_err,
						sizeof (child_errcode_t));
		close (status_pipe_father[1]);
		close (status_pipe_child[0]);
		close (status_pipe_child[1]);
		exit (1);
	}


	/*
	 * Father
	 */
	if (child > 0) {
		close (status_pipe_child[1]);

		/* Read the error code sent by the child */
		nb_read = read (status_pipe_child[0], &result_errno,
							sizeof (int));
		if (nb_read > 0) {
			nb_read = read (status_pipe_child[0], &child_err,
						sizeof (child_errcode_t));
		}
		close (status_pipe_child[0]);

		/* No error, the job is running */
		if (nb_read == -1 || nb_read == 0) {
			close (status_pipe_father[1]);
			if (j->detach == 0) {
				/*
				 * Run the main function of the launcher to
				 * wait for the command result
				 */
				launcher_main (	child, job_id, job_name,
						open_log);
			}
			exit (0);
		}

		/* Send back the error code to the father */
		waitpid (child, NULL, 0);
		write (status_pipe_father[1], &result_errno, sizeof (int));
		write (	status_pipe_father[1], &child_err,
			sizeof (child_errcode_t));
		close (status_pipe_father[1]);
		exit (1);
	}


	/*
	 * Child
	 */

	close (status_pipe_father[1]);
	close (status_pipe_child[0]);
	if (status_pipe_child[1] != 3) {
		dup2 (status_pipe_child[1], 3);
		close (status_pipe_child[1]);
	}
	fcntl (3, F_SETFD, FD_CLOEXEC);

#if HAVE_LIBCGROUP
	/* Linux Control Group */
	cgroup_set (j->cgroup_name, getpid());
#endif

	/* umask */
#if HAVE_UMASK
	umask (j->mask);
#endif

#if HAVE_GETPRIORITY && HAVE_SETPRIORITY
	/* nice */
	old_nice = getpriority (PRIO_PROCESS, 0);
	/*
	 * In GNU/Hurd, getpriority() returns strange values (not in -20/+19).
	 * In this case, do nothing.
	 */
	if (   old_nice >= -20 && old_nice <= 19 && old_nice != j->nice
	    && j->nice >= -20 && j->nice <= 19
	    && setpriority (PRIO_PROCESS, 0, j->nice) != 0)
	{
		write (3, &errno, sizeof (int));
		child_err = CHILD_SETPRIORITY;
		write (3, &child_err, sizeof (child_errcode_t));
		close (3);
		exit (1);
	}
#endif

#if HAVE_SETRLIMIT && HAVE_GETRLIMIT
	/* ulimit */
	if (getrlimit (RLIMIT_FSIZE, &rlim) == 0) {
		if (j->lim == 0) {
			j->lim = RLIM_INFINITY;
		}
		if (rlim.rlim_cur != j->lim) {
			rlim.rlim_cur = j->lim;
			rlim.rlim_max = j->lim;
			if (setrlimit (RLIMIT_FSIZE, &rlim) != 0) {
				write (3, &errno, sizeof (int));
				child_err = CHILD_SETRLIMIT;
				write (	3, &child_err,
					sizeof (child_errcode_t));
				close (3);
				exit (1);
			}
		}
	}
#endif

	/* gid */
#if HAVE_SETREGID
	if (setregid (j->gid, j->gid) != 0) {
		child_err = CHILD_SETGID;
		write (3, &errno, sizeof (int));
		write (3, &child_err, sizeof (child_errcode_t));
		close (3);
		exit (1);
	}
#else
	if (setgid (j->gid) != 0) {
		child_err = CHILD_SETGID;
		write (3, &errno, sizeof (int));
		write (3, &child_err, sizeof (child_errcode_t));
		close (3);
		exit (1);
	}
#if HAVE_SETEGID
	if (setegid (j->gid) != 0) {
		child_err = CHILD_SETEGID;
		write (3, &errno, sizeof (int));
		write (3, &child_err, sizeof (child_errcode_t));
		close (3);
		exit (1);
	}
#endif
#endif

	/* uid */
#if HAVE_SETREUID
	if (setreuid (j->uid, j->uid) != 0) {
		child_err = CHILD_SETUID;
		write (3, &errno, sizeof (int));
		write (3, &child_err, sizeof (child_errcode_t));
		close (3);
		exit (1);
	}
#else
	if (setuid (j->uid) != 0) {
		child_err = CHILD_SETUID;
		write (3, &errno, sizeof (int));
		write (3, &child_err, sizeof (child_errcode_t));
		close (3);
		exit (1);
	}
#if HAVE_SETEUID
	if (seteuid (j->uid) != 0) {
		child_err = CHILD_SETEUID;
		write (3, &errno, sizeof (int));
		write (3, &child_err, sizeof (child_errcode_t));
		close (3);
		exit (1);
	}
#endif
#endif

	/* Change dir */
#if HAVE_CHDIR
	if (chdir (j->work) != 0) {
		write (3, &errno, sizeof (int));
		child_err = CHILD_CHDIR;
		write (3, &child_err, sizeof (child_errcode_t));
		close (3);
		exit (1);
	}
#endif

	/* Open stdout (if not set, open /dev/null) */
	if (j->file_stdout == NULL || (j->file_stdout)[0] == '\0') {
		out = open ("/dev/null", O_WRONLY);
	}
	else {
		out = open (	j->file_stdout,
				O_CREAT|O_WRONLY|O_APPEND,
				S_IRUSR|S_IWUSR|
				S_IRGRP|S_IWGRP|
				S_IROTH|S_IWOTH);
	}
	if (out < 0) {
		write (3, &errno, sizeof (int));
		child_err = CHILD_OPEN;
		write (3, &child_err, sizeof (child_errcode_t));
		close (3);
		exit (1);
	}
	if (out != 1) {
		dup2 (out, 1);
		close (out);
	}

	/*
	 * Open stderr (if not set, open /dev/null.  If the file
	 * name is the same as for stdout, duplicate the stdout
	 * file descriptor)
	 */
	if (j->file_stderr == NULL || (j->file_stderr)[0] == '\0') {
		err = open ("/dev/null", O_WRONLY);
		if (err < 0) {
			write (3, &errno, sizeof (int));
			child_err = CHILD_OPEN;
			write (3, &child_err, sizeof (child_errcode_t));
			close (1);
			close (3);
			exit (1);
		}
		dup2 (err, 2);
		close (err);
	}
	else {
		if (	   j->file_stdout != NULL
			&& strcmp (j->file_stdout, j->file_stderr) == 0)
		{
			dup2 (1, 2);
		}
		else {
			err = open (	j->file_stderr,
					O_CREAT|O_WRONLY|O_APPEND,
					S_IRUSR|S_IWUSR|
					S_IRGRP|S_IWGRP|
					S_IROTH|S_IWOTH);
			if (err < 0) {
				write (3, &errno, sizeof (int));
				child_err = CHILD_OPEN;
				write (3, &child_err,
						sizeof (child_errcode_t));
				close (1);
				close (3);
				exit (1);
			}
			if (err != 2) {
				dup2 (err, 2);
				close (err);
			}
		}
	}

	/* Exec the command */
	execve (j->command, (j->arguments).arg, (j->environ).env);

	/* Error */
	write (3, &errno, sizeof (int));
	child_err = CHILD_EXEC;
	write (3, &child_err, sizeof (child_errcode_t));
	close (1);
	close (2);
	close (3);
	exit (1);
}


/*
 * Execute the job (fork and exec).
 * A child is created (the launcher).  It will start the job (a second fork
 * and an exec) and manage it (getting its return code, writing the result of
 * the job in a result file, ...).
 * If err_msg is not NULL, it is set with the error message in case of error
 * (it must be freed by the caller).  If no error, *err_msg is set to NULL
 * If job_pid is not NULL, it is set with the PID of the running child.
 *
 *
 * Return:
 *    0 --> No error (job_pid is set if not NULL)
 *   -1 --> System error (errno and err_msg (if not NULL) are set).  err_msg
 *          may be NULL if malloc failed to allocate enough space for this
 *          error message.
 */
#if HAVE_PID_T
static int
exec_command (	job_run_t *j, const char *job_id, pid_t *job_pid,
		const char *job_name, char **err_msg)
#else
static int
exec_command (	job_run_t *j, const char *job_id, int *job_pid,
		const char *job_name, char **err_msg)
#endif
{
#if HAVE_PID_T
	pid_t child;
#else
	int child;
#endif
	int status_pipe[2];
	int save_errno;
	int ret;


#if HAVE_ASSERT_H
	assert (j != NULL);
#endif

	if (err_msg != NULL) {
		*err_msg = NULL;
	}

	/*
	 * A pipe is opened between the father and the child.  If something
	 * goes wrong in the child (fail to exec, to change the user id, ...)
	 * the child writes the errno code of the error in the pipe so the
	 * father can report the error to the caller.
	 */
	if (pipe (status_pipe) != 0) {
		save_errno = errno;
		build_errmsg_errno (save_errno, "pipe: ", err_msg);
		errno = save_errno;
		return -1;
	}

	child = fork ();
	if (child < 0) {
		save_errno = errno;
		close (status_pipe[0]);
		close (status_pipe[1]);
		build_errmsg_errno (save_errno, "fork: ", err_msg);
		errno = save_errno;
		return -1;
	}

	/*
	 * Father
	 */
	if (child > 0) {
		close (status_pipe[1]);
		ret = get_child_status (status_pipe[0], child, err_msg);
		save_errno = errno;
		close (status_pipe[0]);
		if (ret == 0 && job_pid != NULL) {
			*job_pid = child;
		}
		errno = save_errno;
		return ret;
	}

	/*
	 * Child
	 */
	launcher (j, status_pipe, job_id, job_name);
	return 0; /* Never reached - launcher() uses exit()) */
}


/*
 * Submit a job
 * If not NULL, job_pid is set with the PID of the runnning job
 * In case of error, if err_msg is not NULL, the error message is returned in
 * *err_msg (it must be freed by the caller).
 * If no error, *err_msg is set to NULL.
 * Warning: even in case of error, *err_msg may be NULL if malloc failed to
 * allocate enough space for this error message.
 *
 * Return:
 *    0 --> No error (job_pid is set if not NULL)
 *   -1 --> Memory allocation error
 *   -2 --> Other error (*err_msg is set if err_msg is not NULL)
 */
#if HAVE_PID_T
int
submit_job (	job_parameters_t *j, const char *job_id, pid_t *job_pid,
		char **err_msg)
#else
int
submit_job (	job_parameters_t *j, const char *job_id, int *job_pid,
		char **err_msg)
#endif
{
	int save_errno, idx;
	job_run_t jr;


	if (j == NULL) {
		return 0;
	}

	if (err_msg != NULL) {
		*err_msg = NULL;
	}

	/* Build the job_run structure */
	init_job_run (&jr);

	switch (add_username_job_run (&jr, j->username)) {
		case 1:
			destroy_job_run (&jr);
			build_errmsg (j->username, _("Unknown user"), err_msg);
			return -2;
		case -1:
			destroy_job_run (&jr);
			build_errmsg (	NULL, _("Memory allocation error"),
					err_msg);
			return -1;
	}

	switch (add_env_job_run (&jr, &(j->environ), &idx)) {
		case -2: case -3:
			destroy_job_run (&jr);
			build_errmsg ((j->environ).env[idx], _("Syntax error"),
					err_msg);
			return -2;
		case -1:
			destroy_job_run (&jr);
			build_errmsg (	NULL, _("Memory allocation error"),
					err_msg);
			return -1;
	}

	switch (add_outfile_job_run (&jr, j->file_stdout, j->file_stderr)) {
		case -1:
			destroy_job_run (&jr);
			build_errmsg (	NULL, _("Memory allocation error"),
					err_msg);
			return -1;
		case -2:
			destroy_job_run (&jr);
			build_errmsg (j->file_stdout, _("Syntax error"),
					err_msg);
			return -2;
		case -3:
			destroy_job_run (&jr);
			build_errmsg (j->file_stderr, _("Syntax error"),
					err_msg);
			return -2;
	}

	if (add_cgroup_job_run (&jr, j->cgroup_name) != 0) {
		destroy_job_run (&jr);
		build_errmsg (NULL, _("Memory allocation error"), err_msg);
		return -1;
	}

	switch (add_command_job_run (&jr, j->command, &(j->arguments),
					j->load_env, &idx))
	{
		case -1:
			destroy_job_run (&jr);
			build_errmsg (	NULL, _("Memory allocation error"),
					err_msg);
			return -1;
		case -2:
			destroy_job_run (&jr);
			if (idx == -1) {
				build_errmsg (	j->command, _("Syntax error"),
						err_msg);
			}
			else {
				build_errmsg (	(j->arguments).arg[idx],
						_("Syntax error"), err_msg);
			}
			return -2;
		case -3:
			destroy_job_run (&jr);
			build_errmsg (j->command, _("No such file"), err_msg);
			return -2;
	}
	add_detach_job_run (&jr, j->detach);

	/* Run the command */
	if (exec_command (&jr, job_id, job_pid, j->path, err_msg) != 0) {
		save_errno = errno;
		destroy_job_run (&jr);
		errno = save_errno;
		return -2;
	}
	destroy_job_run (&jr);
	return 0;
}

/*-----------------============== End Of File ==============-----------------*/
