/* Schedwi
   Copyright (C) 2007-2010 Herve Quatremain

   This file is part of Schedwi.

   Schedwi is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 3 of the License, or
   (at your option) any later version.

   Schedwi is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

/*
 * net_utils_sock.c -- Useful network functions (basic socket connection
 * without SSL - which is not available on this host)
 */

#include <schedwi.h>

#if STDC_HEADERS
#include <stdlib.h>
#include <string.h>
#endif

#if HAVE_SYS_SOCKET_H
#include <sys/socket.h>
#endif

#if HAVE_NETINET_IN_H
#include <netinet/in.h>
#endif

#if HAVE_ARPA_INET_H
#include <arpa/inet.h>
#endif

#if HAVE_NETDB_H
#include <netdb.h>
#endif

#if HAVE_SYS_SELECT_H
#include <sys/select.h>
#else
#if HAVE_SYS_TIME_H
#include <sys/time.h>
#endif
#endif

#if HAVE_UNISTD_H
#include <unistd.h>
#endif

#if HAVE_TIME_H
#include <time.h>
#endif

#if HAVE_CTYPE_H
#include <ctype.h>
#endif

#if HAVE_ERRNO_H
#include <errno.h>
#endif
#ifndef errno
extern int errno;
#endif

#if HAVE_TCPD_H
#include <tcpd.h>
#endif

#if HAVE_SYSLOG_H
#include <syslog.h>
#endif

#if HAVE_ASSERT_H
#include <assert.h>
#endif

#ifndef HAVE_RECV
#define recv(a,b,c,d) read(a,b,c)
#endif

#ifndef HAVE_SEND
#define send(a,b,c,d) write(a,b,c)
#endif

#ifndef PF_INET
#define PF_INET AF_INET
#endif

#ifndef HAVE_CLOSESOCKET
#define closesocket(x) close(x)
#endif

#ifndef IPV6_V6ONLY
#define IPV6_V6ONLY 26
#endif

#ifndef IPPROTO_IPV6
#define IPPROTO_IPV6 41
#endif

#ifndef NI_MAXHOST
#define NI_MAXHOST 1025
#endif

#include <lib_functions.h>
#include <lwc_log.h>
#include <utils.h>
#include <net_utils_sock.h>
#include <addrmatch.h>
#include <JSON_parser.h>
#include <parse_json_result.h>


#define BUFFER_LEN 1024

#if HAVE_LIBWRAP
int allow_severity;
int deny_severity;
#endif


#ifndef HAVE_GETADDRINFO
/* Protocol number for TCP */
static int protocol = -1;

/*
 *  Convert a port number/name in an unsigned short int suitable for the
 *  member sin_port of the sockaddr_in structure
 *
 *  Return :
 *    The port number or 0 in case of error
 */
static unsigned short int
get_port_number (const char *port)
{
	int i;
#if HAVE_GETSERVBYNAME
	struct servent *entry;
#endif

#if HAVE_ASSERT_H
	assert (port != NULL);
#endif

	/* Skip the spaces */
	while (*port != '\0' && isspace (*port) != 0) {
		port++;
	}

	/* Empty string ! */
	if (*port == '\0') {
		return 0;
	}

	/* Check if the string contains only digits */
	for (i = 0; port[i] != '\0' && isdigit (port[i]) != 0; i++);

	/* Only digits */
	if (port[i] == '\0') {
		return htons ((unsigned short int) atoi (port));
	}

#if HAVE_GETSERVBYNAME
	/*
	 * The string is a port name. This name will be searched in
	 * /etc/services (or equivalent)
	 */
	entry = getservbyname (port, "tcp");
	if (entry != NULL) {
		return (unsigned short int) entry->s_port;
	}
#endif
	return  0;
}


/*
 * Convert the provided port string (a service name or number) to a port
 * number
 *
 * Return:
 *   The new string (to be freed by the caller by free() or
 *   NULL in case of memory allocation error
 */
char *
port2number (const char *port)
{
	char *s;

#if HAVE_ASSERT_H
	assert (port != NULL);
#endif
	s = (char *) malloc (7);
	if (s == NULL) {
		return NULL;
	}
	copy_ulltostr (ntohs (get_port_number (port)), s);
	return s;
}
#else /* HAVE_GETADDRINFO */
char *
port2number (const char *port)
{
	char *s;
	int try_no, ret;
	struct addrinfo hints;
	struct addrinfo *result;
#if HAVE_NANOSLEEP
	struct timespec req;

	req.tv_sec = SCHEDWI_HOSTLOOKUP_SLEEP;
	req.tv_nsec = 0;
#endif


#if HAVE_ASSERT_H
	assert (port != NULL);
#endif
	s = (char *) malloc (7);
	if (s == NULL) {
		return NULL;
	}

	schedwi_memset (&hints, 0, sizeof (struct addrinfo));
	hints.ai_family    = AF_UNSPEC;
	hints.ai_socktype  = SOCK_STREAM; /* Stream socket */
	hints.ai_flags     = 0;
        hints.ai_protocol  = 0;           /* Any protocol */
	hints.ai_canonname = NULL;
	hints.ai_addr      = NULL;
	hints.ai_next      = NULL;

	try_no = 0;
	do {
		ret = getaddrinfo (NULL, port, &hints, &result);
		/*
		 * If the name server returned a temporary failure, try
		 * again.
		 */
		if (ret == EAI_AGAIN) {
			try_no++;
			if (try_no < SCHEDWI_HOSTLOOKUP_RETRIES) {
#if HAVE_NANOSLEEP
				nanosleep (&req, NULL);
#elif HAVE_SLEEP
				sleep (SCHEDWI_HOSTLOOKUP_SLEEP);
#elif HAVE_USLEEP
				usleep (SCHEDWI_HOSTLOOKUP_SLEEP * 1000000);
#endif
			}
		}
		else {
			if (ret != 0) {
				strncpy (s, port, 6);
				s[6] = '\0';
				return s;
			}
		}
	} while (ret != 0 && try_no < SCHEDWI_HOSTLOOKUP_RETRIES);

	if (ret != 0) {
		strncpy (s, port, 6);
		s[6] = '\0';
		return s;
	}

	ret = getnameinfo (	result->ai_addr, result->ai_addrlen,
				NULL, 0,
				s, 7,
				NI_NUMERICSERV);
	freeaddrinfo (result);
	if (ret != 0) {
		strncpy (s, port, 6);
		s[6] = '\0';
		return s;
	}
	return s;
}
#endif /* HAVE_GETADDRINFO */


/*
 * Retrieve the next waiting connection from the provided set.
 * sock_set and nb_set are updated accordingly.
 *   sock_set --> Set of waiting sockets
 *   max_set  --> Highest socket number + 1 in the set
 *   nb_set   --> Number of waiting sockets in the set
 *   allow_from --> Only connection from the server are allowed
 *
 * Return :
 *   The next socket
 *   -1 in case of error (errno is set and a message is displayed using
 *      lwc_writeLog())
 *   -2 No more socket waiting (the set is empty)
 */
static int
net_accept_next (fd_set *sock_set, int max_set, int *nb_set,
		 const char *allow_from)
{
	int i, nb, save_errno, s, ret;
#if HAVE_GETADDRINFO
	struct sockaddr_storage sockname;
#else
	struct sockaddr_in sockname;
#endif
#if HAVE_GETNAMEINFO
	char name[NI_MAXHOST];
#endif
	socklen_t len;
#if HAVE_LIBWRAP
	struct request_info req;
#endif

	nb = *nb_set;
	for (i = 0 ; i < max_set && nb > 0; i++) {
		if (FD_ISSET (i, sock_set) == 0) {
			continue;
		}
		nb--;

		do {
			len = sizeof (sockname);
			s = accept (i, (struct sockaddr *)&sockname, &len);
			if (s < 0) {
#if CYGWIN_IN_USE
				/*
				 * Sometimes, using Cygwin, the errno value is
				 * not set to EINTR nor EAGAIN when accept has
				 * been interrupted by a signal.  When accept()
				 * returns -1, the error is then ignored and
				 * accept() is run again
				 */
				continue;
#else
				if (errno != EINTR && errno != EAGAIN) {
					save_errno = errno;
					lwc_writeLog (	LOG_CRIT,
						_("Network error: accept: %s"),
						strerror (errno));
					errno = save_errno;
					return -1;
				}
#endif
			}
		} while (s < 0);

		FD_CLR (i, sock_set);
		(*nb_set)--;

		/*
		 * Only the server (defined by the configuration option
		 * ALLOW_FROM) is allowed.
		 * It's always refused if the variable is not set (it should
		 * not happened as if the variable is not set in the
		 * configuration file it is set in schedwiclnt.c to the value
		 * of the SERVER variable)
		 */
		if (allow_from == NULL || allow_from[0] == '\0') {
			net_close_sock (s);
			continue;
		}

#if HAVE_GETNAMEINFO
		ret = getnameinfo (	(struct sockaddr *)(&sockname), len,
					name, sizeof (name), NULL, 0,
					NI_NUMERICHOST);
		if (ret != 0) {
			lwc_writeLog (	LOG_NOTICE,
					_("getnameinfo: %s"),
					gai_strerror (ret));
			net_close_sock (s);
			continue;
		}
#else /* no HAVE_GETNAMEINFO */
		ret = sizeof (name) - 1;
		strncpy (name, inet_ntoa (sockname.sin_addr), ret);
		name[ret] = '\0';
#endif /* HAVE_GETNAMEINFO */

#if HAVE_LIBWRAP
		/* Check the network access using the TCP Wrappers */
		allow_severity = LOG_INFO;
		deny_severity  = LOG_WARNING;

		request_init (	&req,
				RQ_DAEMON, SCHEDWI_TCP_WRAPPERS_DAEMON,
				RQ_FILE, s, 0);
		fromhost (&req);
		if (hosts_access (&req) == 0) {
			lwc_writeLog (	LOG_DEBUG,
			_("Connection refused by the TCP Wrappers for %s"),
					name);
			net_close_sock (s);
			continue;
		}
#endif /* HAVE_LIBWRAP */

		/*
		 * Check through the internal network access list if the client
		 * (the Schedwi server) is allowed.  This list is set through
		 * the configuration variable ALLOW_FROM.
		 */
		ret = addr_match_list (name, allow_from);
		switch (ret) {
			case 1:
				/* Allowed */
				return s;
			case 0:
				lwc_writeLog (	LOG_DEBUG,
						_("Connection refused for %s"),
						name);
				break;
			case -1:
				lwc_writeLog (	LOG_CRIT,
						_("Memory allocation error"));
				break;
			case -2:
				/*
				 * Should not occur (checked at start time
				 * by check_netlist())
				 */
				lwc_writeLog (	LOG_ERR,
	_("Inconsistent mask length in one of the address in ALLOW_FROM"));
				break;
		}

		net_close_sock (s);
	}
	return -2;
}


/*
 * Wait for a network request and return the socket associated with the
 * accepted connection.  The client host is check to be sure that it's the
 * Schedwi server
 *
 * Return:
 *   The new socket or
 *   -1 in case of error (errno is set and a message is displayed using
 *      lwc_writeLog())
 */
int
net_accept_sock (fd_set *sock_set, int sock_max, const char *allow_from)
{
	static fd_set waiting_set;
	static int still_waiting = 0;
	int s, ret, save_errno;
	fd_set rfds;

#if HAVE_ASSERT_H
	assert (sock_set != NULL);
#endif

	/* First, manage the connections still waiting from a previous call */
	s = net_accept_next (	&waiting_set, sock_max, &still_waiting,
				allow_from);
	if (s == -1) {
		return -1;
	}
	if (s >= 0) {
		return s;
	}

	/* No more pending connections from a previous call */

	while (1) {
		rfds = *sock_set;
		ret = select (sock_max, &rfds, NULL, NULL, NULL);
		if (ret < 0) {
#if CYGWIN_IN_USE
			/*
			 * Sometimes, using Cygwin, the errno value is
			 * not set to EINTR nor EAGAIN when select has
			 * been interrupted by a signal.  When select()
			 * returns -1, the error is then ignored and select()
			 * is run again
			 */
			continue;
#else
			if (errno == EINTR || errno == EAGAIN) {
				continue;
			}
			save_errno = errno;
			lwc_writeLog (	LOG_CRIT,
					_("Network error: select: %s"),
					strerror (errno));
			errno = save_errno;
			return -1;
#endif
		}
		waiting_set = rfds;
		still_waiting = ret;

		s = net_accept_next (	&waiting_set, sock_max, &still_waiting,
					allow_from);
		if (s == -1) {
			return -1;
		}
		if (s >= 0) {
			return s;
		}
	}
	return -1;
}


/*
 * Create a socket (plus bind and listen)
 * domain, type, protocol_s : see socket(2)
 * addr, addrlen  : see bind(2)
 *
 * Return:
 *   The socket OR
 *   -1 in case of error
 */
static int
net_create_sock (int domain, int type, int protocol_s,
		const struct sockaddr *addr, socklen_t addrlen)
{
	int sock, opt;

	/* Create the socket */
	sock = socket (domain, type, protocol_s);
	if (sock < 0) {
		lwc_writeLog (LOG_ERR, _("socket: %s"), strerror (errno));
		return -1;
	}

	/* Set the REUSEADDR flag */
	opt = 1;
	if (setsockopt(sock, SOL_SOCKET, SO_REUSEADDR, &opt, sizeof (opt)) != 0)
	{
		lwc_writeLog (LOG_ERR, _("setsockopt: %s"), strerror (errno));
	}

	/* Allow the IPv6 bind to the same port as in IPv4 */
	if (	   domain == AF_INET6
		&& setsockopt(sock, IPPROTO_IPV6, IPV6_V6ONLY,
				&opt, sizeof (opt)) != 0)
	{
		lwc_writeLog (LOG_ERR, _("setsockopt: %s"), strerror (errno));
	}

	/* Bind to the address */
	if (bind (sock, addr, addrlen) != 0) {
		lwc_writeLog (LOG_ERR, _("bind: %s"), strerror (errno));
		closesocket (sock);
		return -1;
	}

	/* Prepare to accept requests */
	if (listen (sock, SCHEDWI_LISTEN_BACKLOG) != 0) {
		lwc_writeLog (LOG_ERR, _("listen: %s"), strerror (errno));
		closesocket (sock);
		return -1;
	}

	return sock;
}


/*
 * Create a socket ready to accept requests
 *
 * port_number is the port number to listen to (a number or a service name
 *             in /etc/services)
 *       iface is the network interface to listen to (NULL or empty means all
 *             the interfaces)
 *
 * Return:
 *    0 --> No error (sock_set and sock_max are set)
 *   -1 --> Error (an error message is displayed using lwc_writeLog())
 */
int
net_server_sock (const char *port_number, const char *iface,
		const char *address_family,
		fd_set *sock_set, int *sock_max)
{
	int sock, try_no;
	fd_set fds;
#if HAVE_GETADDRINFO
	struct addrinfo hints;
	struct addrinfo *result, *rp;
	int ret;
	int max_sock;
#else /* not HAVE_GETADDRINFO */
#if HAVE_GETPROTOBYNAME
	struct protoent *prot;
#endif
	struct  sockaddr_in sad;
	unsigned long int iface_addr;
	unsigned short int port;
	struct hostent *he;
#endif /* HAVE_GETADDRINFO */
#if HAVE_NANOSLEEP
	struct timespec req;

	req.tv_sec = SCHEDWI_HOSTLOOKUP_SLEEP;
	req.tv_nsec = 0;
#endif

#if HAVE_ASSERT_H
	assert (port_number != NULL);
#endif

/*
 * If the getaddrinfo() function is available lets use it (support for IPv6)
 */
#if HAVE_GETADDRINFO

	schedwi_memset (&hints, 0, sizeof (struct addrinfo));
	hints.ai_family = AF_UNSPEC;   /* Allow IPv4 and IPv6 */
	if (address_family != NULL) {
		if (schedwi_strcasecmp (address_family, "inet") == 0) {
			hints.ai_family = AF_INET; /* Only allow IPv4 */
		}
		else {
			if (schedwi_strcasecmp (address_family, "inet6") == 0) {
				hints.ai_family = AF_INET6; /* Only IPv6 */
			}
		}
	}
	hints.ai_socktype  = SOCK_STREAM; /* Stream socket */
	hints.ai_flags     = AI_PASSIVE;
        hints.ai_protocol  = 0;           /* Any protocol */
	hints.ai_canonname = NULL;
	hints.ai_addr      = NULL;
	hints.ai_next      = NULL;

	try_no = 0;
	do {
		ret = getaddrinfo (
			(iface == NULL || iface[0] == '\0') ? NULL : iface,
			port_number, &hints, &result);
		/*
		 * If the name server returned a temporary failure, try
		 * again.
		 */
		if (ret == EAI_AGAIN) {
			try_no++;
			if (try_no < SCHEDWI_HOSTLOOKUP_RETRIES) {
#if HAVE_NANOSLEEP
				nanosleep (&req, NULL);
#elif HAVE_SLEEP
				sleep (SCHEDWI_HOSTLOOKUP_SLEEP);
#elif HAVE_USLEEP
				usleep (SCHEDWI_HOSTLOOKUP_SLEEP * 1000000);
#endif
			}
		}
		else {
			if (ret != 0) {
				lwc_writeLog (LOG_ERR, _("getaddrinfo: %s"),
						gai_strerror (ret));
				return -1;
			}
		}
	} while (ret != 0 && try_no < SCHEDWI_HOSTLOOKUP_RETRIES);

	if (ret != 0) {
		lwc_writeLog (LOG_WARNING, _("Name server unavailable"));
		return -1;
	}

	/* Create the sockets for all the returned addresses */
	FD_ZERO (&fds);
	max_sock = -1;
	for (rp = result; rp != NULL; rp = rp->ai_next) {
		sock = net_create_sock (rp->ai_family,
					rp->ai_socktype,
					rp->ai_protocol,
					rp->ai_addr,
					rp->ai_addrlen);
		if (sock == -1) {
			/* Close all the previously opened sockets */
			while (max_sock >= 0) {
				if (FD_ISSET (max_sock, &fds) != 0) {
					closesocket (max_sock);
				}
				max_sock--;
			}
			freeaddrinfo (result);
			return -1;
		}
		FD_SET (sock, &fds);
		if (sock > max_sock) {
			max_sock = sock;
		}
	}

	freeaddrinfo (result);

	*sock_set = fds;
	*sock_max = max_sock + 1;
	return 0;


#else /* not HAVE_GETADDRINFO */

	/* Get the port number */
	port = get_port_number (port_number);
	if (port == 0) {
		lwc_writeLog (LOG_ERR,
		 	_("Cannot get the port number for `%s'"),
			port_number);
		return -1;
	}

	/* Get the IP address of the network interface to listen to */
	if (iface == NULL || iface[0] == '\0') {
		iface_addr = htonl (INADDR_ANY);
	}
	else {
		try_no = 0;
		do {
			he = gethostbyname (iface);
			if (he == NULL) {
				if (h_errno == TRY_AGAIN) {
					try_no++;
					if (try_no < SCHEDWI_HOSTLOOKUP_RETRIES)
					{
#if HAVE_NANOSLEEP
						nanosleep (&req, NULL);
#elif HAVE_SLEEP
						sleep (
						    SCHEDWI_HOSTLOOKUP_SLEEP);
#elif HAVE_USLEEP
						usleep (
						    SCHEDWI_HOSTLOOKUP_SLEEP
						  * 1000000);
#endif
					}
				}
				else {
				if (h_errno == NO_RECOVERY) {
					lwc_writeLog (	LOG_ERR,
							_("Name server error"));
					return -1;
				}
				else {
					lwc_writeLog (	LOG_ERR,
						_("Cannot get address for %s"),
						iface);
					return -1;
				}}
			}
		} while (he == NULL && try_no < SCHEDWI_HOSTLOOKUP_RETRIES);

		if (he == NULL) {
			lwc_writeLog (	LOG_WARNING,
					_("Name server unavailable"));
			return -1;
		}

		iface_addr = *((unsigned long *)(he->h_addr_list[0]));
	}

	/* Get the protocol number for tcp */
	if (protocol == -1) {
#if HAVE_GETPROTOBYNAME
		prot = getprotobyname ("tcp");
		if (prot == NULL) {
			lwc_writeLog (LOG_ERR,
		    _("getprotobyname: cannot get protocol number for tcp"));
			return -1;
		}
		protocol = prot->p_proto;
#else
		protocol = 0;
#endif
	}

	/* Compose the address */
	schedwi_memset (&sad, 0, sizeof(sad));
        sad.sin_family = AF_INET;
        sad.sin_addr.s_addr = iface_addr;
	sad.sin_port = port;

	/* Create the socket */
	sock = net_create_sock (PF_INET, SOCK_STREAM, protocol,
				(struct sockaddr *)&sad, sizeof(sad));

	if (sock < 0) {
		return -1;
	}

	FD_ZERO (&fds);
	FD_SET (sock, &fds);
	*sock_set = fds;
	*sock_max = sock + 1;
	return 0;
#endif /* HAVE_GETADDRINFO */
}


/*
 * Create a socket connected to a server
 * port_number is the port number to connect to (a number or a service name
 *             in /etc/services)
 *
 * Return:
 *   The socket or
 *   -1 in case of error (an error message is displayed using lwc_writeLog())
 */
int
net_client_sock (const char *port_number, const char *hostname)
{
	int sock, try_no;
#if HAVE_GETADDRINFO
	struct addrinfo hints;
	struct addrinfo *result, *rp;
	int ret;
#else /* not HAVE_GETADDRINFO */
#if HAVE_GETPROTOBYNAME
	struct protoent *prot;
#endif
	struct  sockaddr_in sad;
	unsigned long int iface_addr;
	unsigned short int port;
	struct hostent *he;
#endif /* HAVE_GETADDRINFO */

#if HAVE_NANOSLEEP
	struct timespec req;

	req.tv_sec = SCHEDWI_HOSTLOOKUP_SLEEP;
	req.tv_nsec = 0;
#endif

#if HAVE_ASSERT_H
	assert (port_number != NULL && hostname != NULL);
#endif

/*
 * If the getaddrinfo() function is available lets use it (support for IPv6)
 */
#if HAVE_GETADDRINFO

	schedwi_memset (&hints, 0, sizeof (struct addrinfo));
	hints.ai_family   = AF_UNSPEC;     /* Allow IPv4 or IPv6 */
	hints.ai_socktype = SOCK_STREAM;   /* Stream socket */
#ifdef AI_ADDRCONFIG
	hints.ai_flags    = AI_ADDRCONFIG;
#endif
	hints.ai_protocol = 0;             /* Any protocol */

	try_no = 0;
	do {
		ret = getaddrinfo (hostname, port_number, &hints, &result);
		/*
		 * If the name server returned a temporary failure, try
		 * again.
		 */
		if (ret == EAI_AGAIN) {
			try_no++;
			if (try_no < SCHEDWI_HOSTLOOKUP_RETRIES) {
#if HAVE_NANOSLEEP
				nanosleep (&req, NULL);
#elif HAVE_SLEEP
				sleep (SCHEDWI_HOSTLOOKUP_SLEEP);
#elif HAVE_USLEEP
				usleep (SCHEDWI_HOSTLOOKUP_SLEEP * 1000000);
#endif
			}
		}
		else {
			if (ret != 0) {
				lwc_writeLog (LOG_ERR, _("getaddrinfo: %s"),
						gai_strerror (ret));
				return -1;
			}
		}
	} while (ret != 0 && try_no < SCHEDWI_HOSTLOOKUP_RETRIES);

	if (ret != 0) {
		lwc_writeLog (LOG_WARNING, _("Name server unavailable"));
		return -1;
	}

	/* Try the addresses one after the other until one is working */
	sock = -1;
	for (rp = result; rp != NULL; rp = rp->ai_next) {
		sock = socket(rp->ai_family, rp->ai_socktype, rp->ai_protocol);
		if (sock < 0) {
			continue;
		}

		if (connect (sock, rp->ai_addr, rp->ai_addrlen) == 0) {
			break;
		}
		closesocket (sock);
		sock = -1;
	}

	freeaddrinfo (result);
	if (sock < 0) {
		lwc_writeLog (LOG_ERR, _("Failed to connect to %s: %s"),
				hostname, port_number);
		return -1;
	}

	return sock;

#else /* not HAVE_GETADDRINFO */
		
	/* Get the port number */
	port = get_port_number (port_number);
	if (port == 0) {
		lwc_writeLog (LOG_ERR,
		 	_("Cannot get the port number for `%s'"),
			port_number);
		return -1;
	}

	/* Get the IP address of the server */
	try_no = 0;
	do {
		he = gethostbyname (hostname);
		if (he == NULL) {
			if (h_errno == TRY_AGAIN) {
				try_no++;
				if (try_no < SCHEDWI_HOSTLOOKUP_RETRIES)
				{
#if HAVE_NANOSLEEP
					nanosleep (&req, NULL);
#elif HAVE_SLEEP
					sleep (SCHEDWI_HOSTLOOKUP_SLEEP);
#elif HAVE_USLEEP
					usleep (  SCHEDWI_HOSTLOOKUP_SLEEP
						* 1000000);
#endif
				}
			}
			else {
			if (h_errno == NO_RECOVERY) {
				lwc_writeLog (	LOG_ERR,
						_("Name server error"));
				return -1;
			}
			else {
				lwc_writeLog (	LOG_ERR,
					_("Cannot get address for %s"),
					hostname);
				return -1;
			}}
		}
	} while (he == NULL && try_no < SCHEDWI_HOSTLOOKUP_RETRIES);

	if (he == NULL) {
		lwc_writeLog (	LOG_WARNING,
				_("Name server unavailable"));
		return -1;
	}

	iface_addr = *((unsigned long *)(he->h_addr_list[0]));

	/* Get the protocol number for tcp */
	if (protocol == -1) {
#if HAVE_GETPROTOBYNAME
		prot = getprotobyname ("tcp");
		if (prot == NULL) {
			lwc_writeLog (LOG_ERR,
		    _("getprotobyname: cannot get protocol number for tcp"));
			return -1;
		}
		protocol = prot->p_proto;
#else
		protocol = 0;
#endif
	}

	/* Create the socket */
	sock = socket (PF_INET, SOCK_STREAM, protocol);
	if (sock < 0) {
		lwc_writeLog (LOG_ERR, _("socket: %s"), strerror (errno));
		return -1;
	}

	/* Compose the remote address */
	schedwi_memset(&sad, 0, sizeof(sad));
        sad.sin_family = AF_INET;
        sad.sin_addr.s_addr = iface_addr;
	sad.sin_port = port;

	/* Connect to the address */
	if (connect (sock, (struct sockaddr *)&sad, sizeof(sad)) != 0) {
		lwc_writeLog (LOG_ERR, _("Failed to connect to %s: %s"),
				hostname, strerror (errno));
		closesocket (sock);
		return -1;
	}

	return sock;
#endif /* HAVE_GETADDRINFO */
}


/*
 * JSON callback function
 */
static int
json_char_callback (void* ctx, int type, const JSON_value* value)
{
	int *nb_open_brackets = (int *)ctx;

	switch (type) {
		case JSON_T_ARRAY_BEGIN:
		case JSON_T_OBJECT_BEGIN:
			if (*nb_open_brackets < 0) {
				*nb_open_brackets = 1;
			}
			else {
				(*nb_open_brackets)++;
			}
			break;
		case JSON_T_ARRAY_END:
		case JSON_T_OBJECT_END:
			(*nb_open_brackets)--;
			break;
	}
	return 1;
}


/*
 * Read data from the provided socket handler.
 * The data are store in *buff (of size *len). If buffer is NULL or not big
 * enough, it is reallocated to a bigger size. It must be freed by the caller
 *
 * Return:
 *    The total number of characters read and stored in *buff or
 *    -1 on error (a message is displayed using lwc_writeLog())
 *    -2 if too many data are received (a message is displayed using
 *       lwc_writeLog())
 */
ssize_t
net_read_sock (int sock, char **buff, size_t *len)
{
	ssize_t nb_read, total;
	size_t to_read;
	char *b, *tmp;
	fd_set rfds;
	struct timeval tv;
	int ret, nb_open_brackets, i;
	JSON_config config;
	struct JSON_parser_struct *jc;

#if HAVE_ASSERT_H
	assert (buff != NULL && len != NULL);
#endif

	if (*buff == NULL || *len == 0) {
		*buff = (char *) malloc (BUFFER_LEN);
		if (*buff == NULL) {
			*len = 0;
			lwc_writeLog (LOG_CRIT, _("Memory allocation error"));
			return -1;
		}
		*len = BUFFER_LEN;
	}

	/* JSON parser initialization */
	init_JSON_config (&config);

	nb_open_brackets              = -1;
	config.depth                  = 20;
	config.callback               = &json_char_callback;
	config.allow_comments         = 0;
	config.handle_floats_manually = 0;
	config.callback_ctx           = &nb_open_brackets;

	jc = new_JSON_parser (&config);

	total = 0;
	to_read = *len;
	b = *buff;
	do {
		do {
			FD_ZERO (&rfds);
			FD_SET (sock, &rfds);
			tv.tv_sec = 5;
			tv.tv_usec = 0;
			ret = select (sock + 1, &rfds, NULL, NULL, &tv);
		} while (ret < 0 && (errno == EINTR || errno == EAGAIN));

		if (ret < 0) {
			lwc_writeLog (LOG_ERR,
					_("Error while reading data: %s"),
					strerror (errno));
			delete_JSON_parser (jc);
			return -1;
		}

		if (ret == 0) {
			lwc_writeLog (LOG_NOTICE,
			_("Timeout while waiting to read network data"));
			delete_JSON_parser (jc);
			return -1;
		}

		do {
			nb_read = recv (sock, b, to_read, 0);
		} while (nb_read < 0 && (errno == EINTR || errno == EAGAIN));

		if (nb_read == 0) {
			delete_JSON_parser (jc);
			return total;
		}

		if (nb_read < 0) {
			lwc_writeLog (	LOG_ERR,
					_("Error while reading data: %s"),
					strerror (errno));
			delete_JSON_parser (jc);
			return -1;
		}

		total += nb_read;

		/*
		 * Count the numbers of open/close brackets.  If this number
		 * drops to zero it means that the full JSON request has
		 * been received.
		 */
		for (i = 0; i < nb_read; i++) {
			if (! JSON_parser_char (jc, (unsigned char)(b[i]))) {
				lwc_writeLog (	LOG_ERR,
					_("Syntax error in received data"));
				delete_JSON_parser (jc);
				return -1;
			}
			if (nb_open_brackets == 0) {
				delete_JSON_parser (jc);
				return total;
			}
		}
		
		if ((size_t)nb_read < to_read) {
			to_read -= nb_read;
			b += nb_read;
		}
		else {
			if (*len + BUFFER_LEN > SCHEDWI_MAX_BUFFER_LEN) {
				/* Too many data */
				lwc_writeLog (LOG_NOTICE,
		_("Failed to read the network request: buffer overflow"));
				delete_JSON_parser (jc);
				return -2;
			}

			tmp = (char *) realloc (*buff, *len + BUFFER_LEN);
			if (tmp == NULL) {
				lwc_writeLog (	LOG_CRIT,
						_("Memory allocation error"));
				delete_JSON_parser (jc);
				return -1;
			}
			*buff = tmp;
			*len += BUFFER_LEN;
			b = *buff + total;
			to_read = BUFFER_LEN;
		}
	} while (1);

	delete_JSON_parser (jc);

	return total;
}


/*
 * Read the reply
 *
 * The format of the JSON reply looks like:
 *     {
 *         "success" : false,
 *          "data" : "Database unavailable"
 *     }
 *
 * No reply (nothing to read) means failure.
 *
 * Return:
 *   0 --> The reply means the remote action was successfull. If result_msg is
 *         not NULL, it is set with the message sent by the remote socket. It
 *         must be freed by the caller.
 *   1 --> The reply means the remote action failed. If result_msg is not NULL,
 *         it is set with the message sent by the remote socket. It may be NULL
 *         if no message was sent. It must be freed by the caller.
 *  -1 --> Error (a message is diplayed using lwc_writeLog())
 */
int
net_read_result_sock (int sock, char **result_msg)
{
	char *buff;
	size_t len;
	ssize_t nb_read;
	int ret;

	if (result_msg != NULL) {
		*result_msg = NULL;
	}

	/* Read the reply */
	buff = NULL;
	len = 0;
	nb_read = net_read_sock (sock, &buff, &len);
	if (nb_read < 0 ) {
		if (buff != NULL) {
			free (buff);
		}
		return -1;
	}

	if (nb_read > 0) {
		ret = parse_json_result (buff, nb_read, result_msg, NULL);
	}
	else {
		ret = 1;
	}
	if (buff != NULL) {
		free (buff);
	}
	return ret;
}


/*
 * Write data to the provided socket handler.
 *
 * Return:
 *   0 --> No error
 *  -1 --> Write error (a message is displayed using lwc_writeLog())
 */
int
net_write_sock (int sock, const char *buff, size_t len)
{
	ssize_t nb_write;

	if (buff == NULL || len == 0) {
		return 0;
	}

	do {
		nb_write = send (sock, buff, len, 0);
		if ((size_t)nb_write == len) {
			return 0;
		}

		if (nb_write < 0) {
			if (errno != EINTR && errno != EAGAIN) {
				lwc_writeLog (LOG_ERR,
					_("Error while sending data: %s"),
					strerror (errno));
				return -1;
			}
		}
		else {
			buff += nb_write;
			len -= nb_write;
		}

	} while (1);

	return 0;
}


/*
 * Close the socket
 */
int
net_close_sock (int sock)
{
	if (sock >= 0) {
#if HAVE_SHUTDOWN
		shutdown (sock, SHUT_RDWR);
#endif
		return closesocket (sock);
	}
	return 0;
}

/*------------------------======= End Of File =======------------------------*/
