/* Schedwi
   Copyright (C) 2007-2015 Herve Quatremain

   This file is part of Schedwi.

   Schedwi is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 3 of the License, or
   (at your option) any later version.

   Schedwi is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

/* utils.c -- Several functions */

#include <schedwi.h>

#if HAVE_SYS_TYPES_H
#include <sys/types.h>
#endif

#if STDC_HEADERS
#include <stdlib.h>
#include <string.h>
#else
#if HAVE_STDLIB_H
#include <stdlib.h>
#endif
#if HAVE_STRING_H
#include <string.h>
#endif
#endif

#if HAVE_STDIO_H
#include <stdio.h>
#endif

#if HAVE_SYS_STAT_H
#include <sys/stat.h>
#endif

#ifdef HAVE_SYS_FILE_H
#include <sys/file.h>
#endif

#if HAVE_UNISTD_H
#include <unistd.h>
#endif

#if HAVE_SIGNAL_H
#include <signal.h>
#endif

#if HAVE_FCNTL_H
#include <fcntl.h>
#endif

#if HAVE_ASSERT_H
#include <assert.h>
#endif

#if HAVE_ERRNO_H
#include <errno.h>
#endif
#ifndef errno
extern int errno;
#endif

#include <lib_functions.h>
#include <xmem.h>
#include <utils.h>

#if _GNU_SOURCE && !HAVE_CLEARENV
extern char **environ;
#endif


/*
 * Convert a unsigned long long int to a string
 *
 * Return:
 *   The number of characters copied in str
 */
unsigned int
copy_ulltostr (unsigned long long int v, char *str)
{
	char c;
	unsigned int len, i, j;

	if (str == NULL) {
	    return 0;
	}

	i = 0;
	do {
		str[i++] = '0' + v % 10;
		v /= 10;
	} while (v > 0);
	str[i] = '\0';
	len = i;

	/* Reverse the string */
	j = 0;
	i--;
	while (j < i) {
	    c = str[i];
	    str[i--] = str[j];
	    str[j++] = c;
	}

	return len;
}


/*
 * Return the filename part of a path
 *
 * Return:
 *   A pointer to the filename part in the provided path
 */
const char *
base_name (const char *path)
{
	const char * s, * prev;
	size_t sep_length;


#ifdef HAVE_ASSERT_H
	assert (path != 0);
#endif
	prev = 0;
	s    = path;
	sep_length = strlen (DIR_SEP);
	while ((s = strstr (s, DIR_SEP)) != 0) {
		s += sep_length;
		prev = s;
	}
	if (prev == 0) {
		return path;
	}
	return prev;
}


/*
 * Print copyright and version information to stdout
 */
void
version (const char *prog_name)
{
	printf ("%s (Schedwi) %s\n", prog_name, PACKAGE_VERSION);
	puts("Copyright (C) 2007-2010 Herve Quatremain");
	puts("Schedwi is free software, covered by the GNU");
	puts("General Public License version 3 or later.");
	puts("You may redistribute copies of Schedwi under");
	puts("the terms of this license.");
}


/*
 * Remove a file. If the remove fails, then try to truncate (size=0) the
 * file
 *
 * Return:
 *   0 --> File removed or truncated
 *  -1 --> Error (errno is set)
 */
int
my_unlink (const char *file_name)
{
	FILE *f;
	int save_errno;

	if (file_name == NULL || file_name[0] == '\0') {
		return 0;
	}

	/* Remove the file */
	if (unlink (file_name) != 0) {

		/* File does not exist */
		if (errno == ENOENT) {
			return 0;
		}

		/* Can't remove the file.  Try to truncate it */
		save_errno = errno;
		f = fopen (file_name, "wb");
		if (f == NULL) {
			/* Fail to truncate the file.  Return an error */
			errno = save_errno;
			return -2;
		}
		fclose (f);
	}
	return 0;
}


/*
 * RAZ the process structures:
 *    - Close all open files (except the provided file descriptor)
 *    - Empty the environment
 *    - Set the default action for all the signals
 *    - Remove the signal mask
 *
 * Failure of one of this step is not reported
 *
 * Note:
 *    This function is called in two places by the Schedwi agent (schedwiclnt):
 *       - At the start of the agent, when it goes into background
 *       - Every time a job is started
 *
 *    Clear the environment may cause issues.  In particular, some variables
 *    may be required by the job subprocess.  For instance, using Cygwin, the
 *    PATH environment variable contains the path to the Cygwin DLL
 *    (cygwin1.dll) which is required to fork/exec the job command.  This is
 *    why the environment is not cleared for Cygwin only.
 *    On other systems, other variables may be required (LD_LIBRARY_PATH for
 *    instance for SunOS and Linux) but no special care is taken and they
 *    will be erased by this function.  If they are required they can be
 *    defined as an environment for the job in the Schedwi server.
 */
void
sanitise (int fd)
{
	int i;
	struct sigaction sa;
	sigset_t mask;


	/* Close all files (except fd) */
	for (i = getdtablesize () - 1; i >= 3; i--) {
		if (i != fd) {
			close (i);
		}
	}

	/* Clear the environment */
#if !defined CYGWIN_IN_USE && !defined NETBSD_IN_USE
#if HAVE_CLEARENV
	clearenv ();
#elif _GNU_SOURCE
	environ = NULL;
#endif
#endif

	/* Clear the signal actions and mask */
	schedwi_memset (&sa, 0, sizeof (struct sigaction));
	sa.sa_handler = SIG_DFL;
	sigemptyset (&sa.sa_mask);
	for (i = 0; i < 128; i++) {
		sigaction (i, &sa, NULL);
	}
	sigfillset (&mask);
	sigprocmask (SIG_UNBLOCK, &mask, NULL);
}


/*
 * Clean the process
 */
void
clean_process ()
{
	/* Change default directory to `/' */
#if HAVE_CHDIR
	(void)chdir ("/");
#endif /* HAVE_CHDIR */

	/* Clean the process */
	sanitise (-1);

#if HAVE_UMASK
	umask (0027);
#endif
}


/*
 * Start the daemon
 *
 * Return:
 *   0 --> No error
 *  -1 --> fork error (errno is set)
 */
int
go_daemon()
{
#if HAVE_PID_T
	pid_t p;
#else
	int p;
#endif


	p = fork ();
	if (p < 0) {
		return -1;
	}
	if (p > 0) {
		exit (0);
	}

#if HAVE_SETSID
	setsid();
#endif

	clean_process ();
	return 0;
}


/*
 * Create the intermediate directories leading to `file_path'.  The last
 * component is not created.
 *
 * @param[in] file_path Full file name.
 * @return 0 on success,
 *        -1 if one component already exists and is not a directory or
 *        -2 on failure to create a directory (errno is set)
 */
int
mkpath (const char *file_path)
{
#if HAVE_MKDIR
	char *path, *p;
	size_t sep_length;
	struct stat stat_str;
	int save_errno;


#ifdef HAVE_ASSERT_H
	assert (file_path != NULL && file_path[0] != '\0');
#endif

	path = (char *) xmalloc (strlen (file_path) + 1);
	strcpy (path, file_path);

	sep_length = strlen (DIR_SEP);
	for (p = strstr (path, DIR_SEP);
	     p != NULL;
	     p = strstr (p + sep_length, DIR_SEP))
	{
		if (p == path) {
			continue;
		}
		*p = '\0';
		if (stat (path, &stat_str) == 0) {
			if (S_ISDIR (stat_str.st_mode) == 0) {
				/* Not a directory */
				free (path);
				return -1;
			}
			*p = DIR_SEP[0];
			continue;
		}
		if (mkdir (path, 0755) != 0) {
			save_errno = errno;
			free (path);
			errno = save_errno;
			return -2;
		}
		*p = DIR_SEP[0];
	}
	free (path);
#endif /* HAVE_MKDIR */
	return 0;
}


static int fd_pid_file = -1;
static char *pid_file_name = NULL;

/*
 * Write the PID of the current process to the provided file.
 *
 * Return:
 *   0 --> No error
 *  -1 --> Fail to open the file (errno is set)
 *   1 --> An other daemon is already running
 */
int
write_pid_file (const char *file_name)
{
	char pid_str[26];


	if (file_name == NULL || file_name[0] == '\0') {
		return 0;
	}

	if (pid_file_name != NULL) {
		close_pid_file ();
	}
	pid_file_name = (char *) xmalloc (strlen (file_name) + 1);
	strcpy (pid_file_name, file_name);

	/* If need be, create the intermediate directories */
	mkpath (file_name);

	/* Open the PID file */
	fd_pid_file = open (file_name, O_RDWR|O_CREAT, 00644);
	if (fd_pid_file < 0) {
		return -1;
	}

	/*
	 * Try to lock it.  If it fails it's probably because an other
	 * daemon is already running
	 */
#if HAVE_LOCKF && F_TLOCK && !HURD_IN_USE
	if (lockf (fd_pid_file, F_TLOCK, 0) < 0) {
		return 1;
	}
#elif HAVE_FLOCK && LOCK_EX && LOCK_NB
	if (flock (fd_pid_file, LOCK_EX | LOCK_NB) < 0) {
		return 1;
	}
#endif

	/* Write the PID to the file */
	snprintf (pid_str, 26, "%ld\n", (long int)getpid ());
	pid_str [26 - 1] = '\0';
	write (fd_pid_file, pid_str, strlen (pid_str));
	return 0;
}


/*
 * Close and remove the file containing the PID of the daemon
 */
void
close_pid_file ()
{
	if (pid_file_name != NULL) {
		my_unlink (pid_file_name);
		free (pid_file_name);
		pid_file_name = NULL;
	}

	if (fd_pid_file >= 0) {
		close (fd_pid_file);
	}
	fd_pid_file = -1;
}


/*
 * Split the given jobid string into the workload part and the job ID part.
 *
 * Return:
 *    The job ID as a string and the workload in `workload' (if not NULL)
 */
char *
split_jobid (const char *jobid, int *workload)
{
	int wl;
	char *s;


	wl = strtol (jobid, &s, 0);
	if (workload != NULL) {
		*workload = wl;
	}
	if (*s != '\0') {
		s++;
	}
	return s;
}


/*
 * Read a file and return a newly allocated string with the content,
 * and set `*length' to the length of the string.  The string is
 * zero-terminated, but the terminating zero byte is not counted in
 * `*length'.
 *
 * Return:
 *   The content of the file (to be freed by free()) OR
 *   NULL in case of error (errno is set and `*length' is undefined)
 */
char *
read_file (const char *filename, size_t *length)
{
	struct stat stat_str;
	FILE *f;
	char *file_content;
	size_t nb_read;
	int save_errno;


#if HAVE_ASSERT_H
	assert (filename != NULL);
#endif

	/* Retrieve the file size */
	if (stat (filename, &stat_str) != 0) {
		return NULL;
	}

	/* Create a buffer for reading the whole file */
	file_content = (char *) xmalloc (stat_str.st_size + 1);

	/* Open and read the whole file */
	f = fopen (filename, "rb");
	if (f == NULL) {
		save_errno = errno;
		free (file_content);
		errno = save_errno;
		return NULL;
	}

	clearerr (f);
	nb_read  = fread (file_content, 1, stat_str.st_size, f);
	if (nb_read != stat_str.st_size && ferror (f) != 0) {
		save_errno = errno;
		fclose (f);
		free (file_content);
		errno = save_errno;
		return NULL;
	}
	fclose (f);

	file_content[stat_str.st_size] = '\0';
	if (length != NULL) {
		*length = stat_str.st_size;
	}
	return file_content;
}

/*------------------------======= End Of File =======------------------------*/
