;; properties.scm: Property declarations and predicates for SCSS
;; Copyright (C) 2011 Julian Graham

;; SCSS is free software: you can redistribute it and/or modify
;; it under the terms of the GNU General Public License as published by
;; the Free Software Foundation, either version 3 of the License, or
;; (at your option) any later version.

;; This program is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;; GNU General Public License for more details.

;; You should have received a copy of the GNU General Public License
;; along with this program.  If not, see <http://www.gnu.org/licenses/>.

#!r6rs

(library (scss properties)
  (export validate-property 
	  property-hash-table
	  color-table

	  specify
	  
	  css-property:default
	  css-property:inherited?)

  (import (rnrs)
	  (scss common)
	  (only (srfi :1) drop)
	  (srfi :2)
	  (srfi :13)
	  (srfi :14))

  (define (ident? x) (or (char-alphabetic? x) (memv x '(#\_ #\-))))
  (define (css-string? x) (memv x '(#\" #\')))
  (define (css-integer? x) (let ((xx (string->number x))) (and xx (exact? xx))))
  (define (pos-integer? x) (let ((xx (string->number x))) (and xx (exact? xx))))
  (define (css-number? x) (string->number x))
  (define (pos-css-number? x) (string->number x))
  
  (define (frequency? x)
    (define (f y) (string-suffix? y x))
    (and-let* ((s (find f  '("Hz" "kHz"))))
	      (pos-css-number? (substring x 0 (- (string-length x) s)))))
  
  (define (angle? x)
    (define (f y) (string-suffix? y x)) 
    (and-let* ((s (find f '("deg" "grad" "rad"))))
	      (css-number? (substring x 0 (- (string-length x) s)))))
  
  (define (percentage? x) 
    (and-let* ((p (string-index x #\%))) (string->number (substring x 0 p))))
  
  (define (length? x) 
    (define (f y) (string-suffix? y x))
    (or (and (find f '("em" "ex" "px" "in" "cm" "mm" "pt" "pc"))
	     (css-number? (substring x 0 (- (string-length x) 2))))
	(equal? x "0")))
  
  (define (cue? x) 
    (for-all (lambda (x) (or (uri? x) (member x '("none" "inherit"))))
	     (string-tokenize x)))
  
  (define color-table '(("maroon" "#800000") 
			("red" "#ff0000") 
			("orange" "#ffa500") 
			("yellow" "#ffff00") 
			("olive" "#808000") 
			("purple" "#800080")
			("fuchsia" "#ff00ff") 
			("white" "#ffffff") 
			("lime" "#00ff00") 
			("green" "#008000") 
			("navy" "#000080") 
			("blue" "#0000ff") 
			("aqua" "#00ffff") 
			("teal" "#008080") 
			("black" "#000000") 
			("silver" "#c0c0c0") 
			("gray" "#808080")))
  
  (define (color? x)
    (define dx (string-downcase x))
    (define (pct? x) (string-suffix? "%" x))
    (define (int? x) (not (or (string-index x #\.) (pct? x))))
    (define ccs (char-set-adjoin char-set:digit #\. #\%))
    (define (f y) (equal? (car y) dx))
    (or (find f color-table)
	(and (string-prefix? "rgb(" x)
	     (let ((s (string-tokenize 
		       (substring x 4 (- (string-length x) 1)) ccs)))
	       (and (= (length s) 3) (or (for-all pct? s) (for-all int? s)))))
	(eqv? (string-ref x 0) #\#)))
  
  (define border-styles
    '("none" 
      "hidden" 
      "dotted" 
      "dashed" 
      "solid" 
      "double"
      "groove" 
      "ridge" 
      "inset" 
      "outset"))
  
  (define (border-*-style? x) (equal? x "inherit") (member x border-styles))
  (define (border-*-width? x) 
    (or (length? x) (member x '("thin" "medium" "thick"))))
  
  (define (svoice? x) (or (ident? x) (css-string? x)))
  (define gvoices '("male" "female" "child"))
  (define gfonts '("serif" "sans-serif" "cursive" "fantasy" "monospace"))
  
  (define comma-charset (char-set #\,))
  (define (font-family? x)
    (define (font-family-pred y)
      (let ((z (string-trim y)))
	(or (member z gfonts) (string-every char-set:graphic z))))
    (for-all font-family-pred (string-tokenize x comma-charset)))
  
  (define (font-size? x) 
    (or (length? x) (percentage? x) (member x '("xx-small"
						"x-small"
						"small"
						"medium"
						"large"
						"x-large"
						"xx-large"
						"larger"
						"smaller"
						"inherit"))))
  (define (font-style? x) (member x '("normal" "italic" "oblique" "inherit")))
  (define (font-variant? x) (member x '("normal" "small-caps" "inherit")))
  (define (font-weight? x) (member x '("normal" 
				       "bold"
				       "bolder"
				       "lighter" 
				       "100" 
				       "200" 
				       "300" 
				       "400"
				       "500" 
				       "600" 
				       "700" 
				       "800" 
				       "900" 
				       "inherit")))
  
  (define (line-height? x) (or (css-number? x) 
			       (length? x) 
			       (percentage? x)
			       (member x '("normal" "inherit"))))
  
  ;; This isn't 100% accurate, but...
  (define (uri? x) (string-prefix? "url(" x))
  (define (ls-image? x) (or (uri? x) (equal? x "none")))
  (define ls-positions '("inside" "outside"))
  (define ls-types '("disc" "circle" "square" "decimal" "decimal-leading-zero" 
		     "lower-roman" "upper-roman" "lower-greek" "lower-latin" 
		     "lower-alpha" "upper-latin" "upper-alpha" "armenian" 
		     "georgian" "none"))
  
  (define (counter? x) 
    (and (string-prefix? "counter(" x)
	 (let* ((args (substring x 8 (- (string-length x) 1)))
		(sx (string-tokenize args (char-set #\,))))
	   (for-all ident? sx))))
  (define (ocolor? x) (or (color? x) (equal? x "invert")))
  (define (css-time? x) (or (string-suffix? "ms" x) (string-suffix? "s" x)))  
  (define (pcolor? x) (or (color? x) (member x '("transparent" "inherit"))))
  (define (pimage? x) (or (uri? x) (member x '("none" "inherit"))))
  (define prepeats '("repeat" "repeat-x" "repeat-y" "no-repeat" "inherit"))
  (define pattaches '("scroll" "fixed" "inherit"))

  (define (lposition-inner? xts)
    (define hori '("left" "center" "right"))
    (define vert '("top" "center" "bottom"))
    (let ((cxts (car xts)))
      (cond ((and (> (length xts) 1)
		  (or (percentage? cxts) (length? cxts) (member cxts hori))
		  (let ((cddx (cadr xts)))
		    (or (percentage? cddx) (length? cddx) (member cddx vert))))
	     (drop xts 2))
	    ((member cxts hori)
	     (let ((cdxts (cdr xts)))
	       (cond ((null? cdxts) cdxts)
		     ((member (car cdxts) vert) (cdr cdxts))
		     (else #f))))
	    ((member cxts vert)
	     (let ((cdxts (cdr xts)))
	       (cond ((null? cdxts) cdxts)
		     ((member (car cdxts) hori) (cdr cdxts))
		     (else #f))))
	    ((equal? cxts "inherit") (cdr xts))
	    (else #f))))

  (define (lposition? x) (null? (lposition-inner? (string-tokenize x))))

  (define (background? x)
    (let loop ((xts (string-tokenize x))
	       (preds (list pcolor? pimage? prepeats pattaches 'lposition)))
      (if (null? xts) #t
	  (let ((cxts (car xts)))
	    (cond ((find (lambda (p) (and (procedure? p) (p cxts))) preds) =>
		   (lambda (p) (loop (cdr xts) (remq p preds))))
		  ((find (lambda (m) (and (list? m) (member cxts m))) preds) =>
		   (lambda (m) (loop (cdr xts) (remq m preds))))
		  ((and (memq 'lposition preds) (lposition-inner? xts)) =>
		   (lambda (xts) (loop xts (remq 'lposition preds))))
		  (else #f))))))
  
  (define (aomf x y)
    (define (f x y)
      (define (g z) (if (procedure? z) (z (car x)) (member (car x) z)))
      (and y (or (null? x) 
		 (and-let* ((a (find g y))) (f (cdr x) (remove a y))))))
    (let ((sx (string-tokenize x)))
      (and (<= (length sx) (length y)) (f sx y))))
  
  (define (mf expr) (lambda (input) (member input expr)))
  
  (define-record-type (:css-property css-property:new css-property?)
    (fields (immutable match-fn css-property:match-fn)
	    (immutable default css-property:default)
	    (immutable inherited? css-property:inherited?)))
  
  (define (azimuth? x)
    (or (angle? x)
	(for-all (lambda (y) (member y '("left-side" 
					 "far-left" 
					 "left" 
					 "center-left" 
					 "center" 
					 "center-right" 
					 "right" 
					 "far-right"
					 "right-side")))
		 (string-tokenize x))
	(member x '("leftwards" "rightwards" "inherit"))))
  (define (border-collapse? x) (member x '("collapse" "separate" "inherit")))
  (define (border-color? x)
    (define (f x) (or (equal? x "transparent") (pcolor? x)))
    (or (equal? x "inherit") (for-all f (string-tokenize x))))
  (define (border-spacing? x) (for-all length? (string-tokenize x)))
  (define (border-style? x) (for-all border-*-style? (string-tokenize x)))
  (define (border-width? x) (for-all border-*-width? (string-tokenize x)))
  (define (bottom? x) 
    (or (length? x) (percentage? x) (member x '("auto" "inherit"))))
  (define (caption-side? x) (member x '("top" "bottom" "inherit")))
  (define (clip? x) 
    (or (member x '("auto" "inherit")) (string-prefix? "(rect" x)))
  (define (content? x)
    (define (f x str-chr)
      (or (null? x)
	  (let ((cx (car x)))
	    (if str-chr
		(if (eqv? (string-ref cx (- (string-length cx) 1)) str-chr)
		    (f (cdr x) #f)
		    (f (cdr x) str-chr))
		(let ((sc (string-ref cx 0)))
		  (if (memv sc '(#\" #\'))
		      (f (cdr x) sc)
		      (and (or (uri? cx)
			       (counter? cx)
			       (member cx '("open-quote"
					    "close-quote"
					    "no-open-quote"
					    "no-close-quote")))
			   (f (cdr x) #f))))))))
    (or (member x '("normal" "inherit")) (f (string-tokenize x) #f)))
  (define (counter-*? x) 
    (or (ident? x) (css-integer? x) (member x '("none" "inherit"))))
  (define (cursor? x)
    (or (equal? x "inherit")
	(let* ((sx (string-tokenize x))
	       (lc (last-pair sx)))
	  (and (for-all uri? (remove lc sx))
	       (member lc '("auto"
			    "crosshair"
			    "default"
			    "pointer"
			    "move"
			    "e-resize"
			    "ne-resize"
			    "nw-resize"
			    "n-resize"
			    "se-resize"
			    "sw-resize"
			    "s-resize"
			    "w-resize"
			    "text"
			    "wait"
			    "help"
			    "progress"))))))
  (define display-types
    '("inline" 
      "block" 
      "list-item" 
      "run-in" 
      "inline-block" 
      "table"
      "inline-table"
      "table-row-group"
      "table-header-group"
      "table-footer-group"
      "table-row"
      "table-column-group"
      "table-column"
      "table-cell"
      "table-caption"
      "none"))
  (define (display? x) (or (member x display-types) (equal? x "inherit")))
  (define (elevation? x)
    (or (angle? x)
	(member x '("below" "level" "above" "higher" "lower"))
	(equal? x "inherit")))	      
  (define (height? x) 
    (or (length? x) (percentage? x) (member x '("auto" "inherit"))))
  (define (letter-spacing? x) 
    (or (length? x) (member x '("normal" "inherit")))) 
  (define (list-style? x)
    (or (aomf x `(,ls-types ,ls-positions ,ls-image?)) (equal? x "inherit")))
  (define (list-style-image? x) (or (equal? x "inherit") (ls-image? x)))
  (define (list-style-position? x) 
    (or (equal? x "inherit") (member x ls-positions)))
  (define (margin-*? x) 
    (or (length? x) (percentage? x) (member x '("auto" "inherit"))))
  (define (margin? x) 
    (or (equal? x "inherit") (for-all margin-*? (string-tokenize x))))
  (define (max-*? x) 
    (or (length? x) (percentage? x) (member x '("none" "inherit"))))
  (define (min-*? x) (or (length? x) (percentage? x) (equal? x "inherit")))
  (define (orphans? x) (or (css-integer? x) (equal? x "inherit")))
  (define (outline? x)
    (or (aomf x `(,ocolor? 
		  ,border-style?
		  ,border-width?))
	(equal? x "inherit")))
  (define (outline-color? x) (or (ocolor? x) (equal? x "inherit")))
  (define (outline-style? x) (or (equal? x "inherit") (member x border-styles)))
  (define (outline-width? x) 
    (or (border-width? x) (member x '("inherit" "medium"))))
  (define (overflow? x)
    (member x '("visible" "hidden" "scroll" "auto" "inherit")))
  (define (padding? x) 
    (or (equal? x "inherit") (for-all padding-*? (string-tokenize x))))
  (define (padding-*? x) (or (length? x) (percentage? x) (equal? x "inherit")))
  (define (page-break-*? x) 
    (member x '("auto" "always" "avoid" "left" "right" "inherit")))
  (define (pause-*? x) (or (equal? x "inherit") (css-time? x) (percentage? x)))
  (define (pause? x) (for-all pause-*? (string-tokenize x)))
  (define (pitch? x) 
    (or (frequency? x) 
	(member x '("x-low" "low" "medium" "high" "x-high" "inherit"))))
  (define (pitch-range? x) (or (css-number? x) (equal? x "inherit")))
  (define (position? x) 
    (member x '("static" "relative" "absolute" "fixed" "inherit")))
  (define (quotes? x)
    (or (member x '("none" "inherit"))
	(let ((sx (string-tokenize x))) ;; This is wrong...
	  (and (even? (length sx)) (for-all css-string? sx)))))
  (define (richness? x) (or (equal? x "inherit") (css-number? x)))
  (define (speech-rate? x)
    (or (css-number? x) 
	(member x '("x-slow" "slow" "medium" "fast" "x-fast" "faster" "slower"))
	(equal? x "inherit")))
  (define (system-font? x)
    (member x '("caption" 
		"icon" 
		"menu" 
		"message-box" 
		"small-caption" 
		"status-bar")))
  (define (text-align? x) 
    (member x '("left" "right" "center" "justify" "inherit")))
  (define (text-decoration? x)
    (define (f x) (member x '("underline" "overline" "line-through" "blink")))
    (or (member x '("inherit" "none")) (for-all f (string-tokenize x))))
  (define (text-indent? x) 
    (or (length? x) (percentage? x) (equal? x "inherit")))
  (define (text-transform? x) 
    (member x '("capitalize" "uppercase" "lowercase" "none" "inherit")))
  (define (unicode-bidi? x) 
    (member x '("normal" "embed" "bidi-override" "inherit")))
  (define (vertical-align? x)
    (or (length? x)
	(percentage? x)
	(member x '("baseline"
		    "sub"
		    "super"
		    "top"
		    "text-top" 
		    "middle" 
		    "bottom" 
		    "text-bottom" 
		    "inherit"))))
  (define (white-space? x)
    (member x '("normal" "pre" "nowrap" "pre-wrap" "pre-line" "inherit")))
  (define (widows? x) (or (equal? x "inherit") (css-integer? x)))
  (define (word-spacing? x) (or (length? x) (member x '("normal" "inherit"))))
  (define (visibility? x) (member x '("visible" "hidden" "collapse" "inherit")))
  (define (voice-family? x)
    (define (f x) (or (svoice? x) (member x gvoices)))
    (or (equal? x "inherit"))) ; (for-all f (string-tokenizer ", "))))
  (define (volume? x)
    (or (css-number? x)
	(percentage? x)
	(member x '("silent" "x-soft" "soft" "medium" "loud" "xloud"))
	(equal? x "inherit")))
  (define (z-index? x) (or (css-integer? x) (member x '("auto" "inherit"))))
  
  (define properties
    `(((azimuth) ,azimuth? "center" #t)
      ((background-attachment) ,(mf '("scroll" "fixed" "inherit")) "scroll" #f)
      ((background-color) ,pcolor? "transparent" #f)
      ((background-image) ,pimage? "none" #f)
      ((background-position) ,lposition? "0% 0%" #f)
      ((background-repeat) ,(mf prepeats) "repeat" #f)
      ((background) ,background? "black none no-repeat fixed 0 0" #f)
      ((border-collapse) ,border-collapse? "separate" #t)
      ((border-color) ,border-color? "black" #t)
      ((border-spacing) ,border-spacing? "0" #t)
      ((border-style) ,border-style? "none" #f)
      ((border-top border-right border-bottom border-left)
       ,(lambda (x) (or (aomf x (list border-width? border-style? pcolor?))
			(equal? "inherit" x)))
       "medium none black" #f)
      ((border-top-color 
	border-right-color 
	border-left-color 	       
	border-bottom-color) ,pcolor? "black" #t)
      ((border-top-style 
	border-right-style 
	border-left-style 
	border-bottom-style) ,border-*-style? "none" #f)
      ((border-top-width 
	border-right-width 
	border-bottom-width 
	border-left-width) ,border-*-width? "medium" #f)
      ((border-width) ,border-width? "medium" #f)
      ((border) ,(lambda (x) (or (equal? x "inherit")
				 (aomf x `(,border-width?
					   ,border-style? 
					   ,pcolor?))))
       "medium none black" #f)
      ((bottom) ,bottom? "auto" #f)
      ((caption-side) ,caption-side? "top" #t)
      ((clear) ,(mf '("none" "left" "right" "both" "inherit")) "none" #f)
      ((clip) ,clip? "auto" #f)
      ((color) ,pcolor? "black" #t)
      ((content) ,content? "normal" #f)
      ((counter-increment counter-reset) ,counter-*? "none" #f)
      ((cue-after cue-before) ,cue? "none" #f)
      ((cue) ,cue? "none none" #f)
      ((cursor) ,cursor? "auto" #t)
      ((direction) ,(mf '("ltr" "rtl" "inherit")) "ltr" #t)
      ((display) ,display? "inline" #f)
      ((elevation) ,elevation? "level" #t)
      ((empty-cells) ,(mf '("show" "hide" "inherit")) "show" #t)
      ((float) ,(mf '("left" "right" "none" "inherit")) "none" #f)
      ((font-family) ,font-family? "monospace" #t)
      ((font-size) ,font-size? "medium" #t)
      ((font-style) ,font-style? "normal" #t)
      ((font-variant) ,font-variant? "normal" #t)
      ((font-weight) ,font-weight? "normal" #t)
      ((font) ,(lambda (x)
		 (let ((f (lambda (x) (or (font-style? x)
					  (font-variant? x)
					  (font-weight? x))))
		       (g (lambda (x)
			    (let ((si (string-index x #\/)))
			      (if si 
				  (let ((a (substring x 0 si))
					(b (substring x (+ si 1))))
				    (and (font-size? a)
					 (line-height? b)))
				  (font-size? x))))))
		   (or (system-font? x)
		       (equal? x "inherit")		     
		       (let ((sx (string-tokenize x)))
			 (and (>= (length sx) 2)
			      (g (car sx))
			      (font-family? (cadr sx)))))))
       "normal normal normal medium monospace" #t)
      ((height left right width bottom top) ,height? "auto" #f)
      ((letter-spacing) ,letter-spacing? "normal" #t)
      ((line-height) ,line-height? "normal" #t)
      ((list-style-image) ,list-style-image? "none" #t)
      ((list-style-position) ,list-style-position? "outside" #t)
      ((list-style-type) ,(mf (cons "inherit" ls-types)) "disc" #t)
      ((list-style) ,list-style? "disc outside none" #t)
      ((margin-right margin-left margin-top margin-bottom) ,margin-*? "0" #f)
      ((margin) ,margin? "0" #f)
      ((max-height max-width) ,max-*? "none" #f)
      ((min-height min-width) ,min-*? "0" #f)
      ((orphans) ,orphans? "2" #t)
      ((outline-color) ,outline-color? "invert" #f)
      ((outline-style) ,outline-style? "none" #f)
      ((outline-width) ,outline-width? "medium" #f)
      ((outline) ,outline? "invert none medium" #f)
      ((overflow) ,overflow? "visible" #f)
      ((padding-top 
	padding-right 
	padding-bottom 
	padding-left) ,padding-*? "0" #f)
      ((padding) ,padding? "0" #f)
      ((page-break-after page-break-before) ,page-break-*? "auto" #f)
      ((page-break-inside) ,(mf '("avoid" "auto" "inherit")) "auto" #t)
      ((pause-after pause-before) ,pause-*? "0" #f)
      ((pause) ,pause? "0 0" #f)
      ((pitch-range) ,pitch-range? "50" #t)
      ((pitch) ,pitch? "medium" #t)
      ((play-during) ,(lambda (x) (or (member x '("auto" "none" "inherit"))
				      (let ((sx (string-tokenize x)))
					(and (uri? (car x))
					     (aomf (cdr x) '(("mix") 
							     ("repeat")))))))
       "auto" #f)
      ((position) ,position? "static" #f)
      ((quotes) ,quotes? "none" #t)
      ((richness) ,richness? "50" #t)
      ((speak-header) ,(mf '("once" "always" "inherit")) "once" #t)
      ((speak-numeral) ,(mf '("digits" "continuous" "inherit")) "continuous" #t)
      ((speak-punctuation) ,(mf '("code" "none" "inherit")) "none" #t)
      ((speak) ,(mf '("normal" "none" "spell-out" "inherit")) "normal" #t)
      ((speech-rate) ,speech-rate? "medium" #t)
      ((stress) ,(lambda (x) (or (equal? x "inherit") (number? x))) "50" #t)
      ((table-layout) ,(mf '("auto" "fixed" "inherit")) "auto" #f)
      ((text-align) ,text-align? "left" #t)
      ((text-decoration) ,text-decoration? "none" #f)
      ((text-indent) ,text-indent? "0" #t)
      ((text-transform) ,text-transform? "none" #t)
      ((unicode-bidi) ,unicode-bidi? "normal" #f)
      ((vertical-align) ,vertical-align? "baseline" #f)
      ((visibility) ,visibility? "visible" #t)
      ((voice-family) ,voice-family? "female" #t)
      ((volume) ,volume? "medium" #t)
      ((white-space) ,white-space? "normal" #t)
      ((widows) ,widows? "2" #t)
      ((word-spacing) ,word-spacing? "normal" #t)
      ((z-index) ,z-index? "auto" #f)))

  (define (validate-property property value)
    (let ((pe (hashtable-ref 
	       property-hash-table (symbol-downcase property) #f)))
      (and pe (apply (css-property:match-fn pe) 
		     `(,(string-downcase value))))))
  
  (define property-hash-table
    (let ((pht (make-eq-hashtable)))
      (for-each (lambda (x)
		  (let ((rec (apply css-property:new (cdr x))))
		    (for-each (lambda (y) (hashtable-set! pht y rec)) (car x))))
		properties)
      pht))
  
  ;; This function is a bit complicated.  The general idea is that for certain
  ;; properties, if they're not explicitly defined in the matched selector 
  ;; block, you can derive their value by looking at a more general property 
  ;; that is defined in that block.
  
  (define (specify sprop val)
    (define vals (string-tokenize val))
    (define (g pred d x)
      (let ((ml (filter pred (string-tokenize x))))
	(case (length ml)
	  ((1) (car ml)) 
	  ((2) (if (member d '("top" "bottom")) (car ml) (cadr ml)))
	  ((3) (cond ((equal? d "top") (car ml))
		     ((member d '("left" "right")) (cadr ml))
		     (else (caddr ml))))
	  ((4) (list-ref ml (list-index (lambda (y) (equal? y d))
					'("top" "right" "bottom" "left"))))
	  (else #f))))
    
    (let* ((prop-bits (string-tokenize sprop char-set:letter))
	   (c (car prop-bits))
	   (cc (cadr prop-bits)))
      
      ;; The way values are mapped to specific properties depends on the 
      ;; number of values as per 
      ;; http://www.w3.org/TR/CSS21/box.html#propdef-border-top-color
      
      (and (> (length prop-bits) 1)
	   (cond 
	    ((equal? (car prop-bits) "border")
	     (if (= (length prop-bits) 2)
		 (cond ((equal? cc "color") (find color? vals))
		       ((equal? cc "width") (find border-width? vals))
		       ((equal? cc "style") (find border-style? vals))
		       
		       ;; Else it's one of top, left, bottom, right
		       
		       ((equal? cc "top") val)
		       ((equal? cc "right") val)
		       ((equal? cc "bottom") val)
		       ((equal? cc "left") val)
		       (else #f))
		 (let* ((ccc (caddr prop-bits)))
		   (cond ((equal? ccc "color") (find color? vals))
			 ((equal? ccc "width") (find border-width? vals))
			 ((equal? ccc "style") (find border-style? vals))))))
	    ((equal? c "padding") (g padding? cc val))
	    ((equal? c "margin") (g margin? cc val))
	    
	    ;; The `background' property is different from some of the other
	    ;; shorthand properties in that it's possible to give it a value 
	    ;; that doesn't give computed values to all of the more specific
	    ;; properties for which it is a shorthand.  In these cases, the
	    ;; specific properties should be explicitly set to their default
	    ;; values.
	    
	    ((equal? c "background")
	     (cond ((equal? cc "color") (find color? vals))
		   ((equal? cc "attach") (and (member val pattaches) val))
		   ((equal? cc "image") (and (pimage? val) val))
		   ((equal? cc "repeat") (and (member val prepeats) val))
		   ;; FIX THIS!!!
		   ((equal? cc "position"))
		   (else #f)))
	    ((and (equal? c "list") (equal? cc "style"))
	     (let ((ccc (caddr prop-bits)))
	       (cond ((equal? ccc "position")
		      (and (member val (cons "inherit" ls-positions)) val))
		     ((equal? ccc "image")
		      (or (ls-image? val) (equal? val "inherit")))
		     ((equal? ccc "type") 
		      (and (member val (cons "inherit" ls-types)) val))
		     (else #f))))
	    ((equal? c "outline")
	     (cond ((equal? cc "color") (find color? vals))
		   ((equal? cc "style") (find border-style? vals))
		   ((equal? cc "width") (find border-width? vals))
		   (else #f)))
	    
	    ;; Like `background', the shorthand `font' property sets all values
	    ;; that can't be deduced from its computed value to their defaults.
	    
	    ;; TODO: Actually parse the computed values!
	    
	    ((equal? c "font")
	     (cond ((equal? cc "family") 
		    (or (and (font-family? val) val)
			(get-default-prop-value (get-prop-entry 'font-family))))
		   ((equal? cc "size") 
		    (or (and (font-size? val) val)
			(get-default-prop-value (get-prop-entry 'font-size))))
		   ((equal? cc "style") 
		    (or (and (font-style? val) val)
			(get-default-prop-value (get-prop-entry 'font-style))))
		   ((equal? cc "variant") 
		    (or (and (font-variant? val) val)
			(get-default-prop-value 
			 (get-prop-entry 'font-variant))))
		   ((equal? cc "weight") 
		    (or (and (font-weight? val) val)
			(get-default-prop-value (get-prop-entry 'font-weight))))
		   (else #f)))
	    (else #f)))))
)
