;; sdom.scm: main module exports and implementations for SDOM
;; Copyright (C) 2004 Julian Graham

;; SDOM is free software; you can redistribute it and/or modify
;; it under the terms of the GNU General Public License as published by
;; the Free Software Foundation; either version 2 of the License, or
;; (at your option) any later version.
;;
;; SDOM is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;; GNU General Public License for more details.
;;
;; You should have received a copy of the GNU General Public License
;; along with SDOM; if not, write to the Free Software
;; Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

(define-module (sdom core)
  #:use-module (ice-9 slib)
  #:use-module (srfi srfi-1)
  #:use-module (sxml ssax)
  #:export     (sdom:sxml->sdom 
		sdom:set-dom-property!
		sdom:get-dom-property
		sdom:node?
		sdom:node-attributes
		sdom:node-name
		sdom:node-type

		sdom:node-type-element
		sdom:node-type-attr
		sdom:node-type-text
		sdom:node-type-cdata-section
		sdom:node-type-entity-reference
		sdom:node-type-entity
		sdom:node-type-processing-instruction
		sdom:node-type-comment
		sdom:node-type-document
		sdom:node-type-document-type
		sdom:node-type-document-fragment
		sdom:node-type-notation

		sdom:exception-code-index-size-err
		sdom:exception-code-domstring-size-err
		sdom:exception-code-hierarchy-request-err
		sdom:exception-code-wrong-document-err
		sdom:exception-code-invalid-character-err
		sdom:exception-code-no-data-allowed-err
		sdom:exception-code-no-modification-allowed-err
		sdom:exception-code-not-found-err
		sdom:exception-code-not-supported-err
		sdom:exception-code-inuse-attribute-err
		sdom:exception-code-invalid-state-err
		sdom:exception-code-syntax-err
		sdom:exception-code-invalid-modification-err
		sdom:exception-code-namespace-err
		sdom:exception-code-invalid-access-err
		sdom:exception-code-validation-err
		sdom:exception-code-type-mismatch-err

		sdom:document-position-disconnected
		sdom:document-position-contained-by
		sdom:document-position-contains
		sdom:document-position-following
		sdom:document-position-preceding
		sdom:document-position-implementation-specific
		
		sdom:user-data-event-node-cloned
		sdom:user-data-event-node-imported
		sdom:user-data-event-node-deleted
		sdom:user-data-event-node-renamed
		sdom:user-data-event-node-adopted

		sdom:register-feature!
		sdom:has-feature?
		sdom-feature-registry

		sdom:add-dom-config-parameter!
		sdom:get-dom-config-parameter
		sdom:set-dom-config-parameter!
		sdom:can-set-dom-config-parameter?

		sdom:insert-before!
		sdom:insert-after!
		sdom:remove-child!
		sdom:replace-child!
		sdom:append-child!
		sdom:clone-node
		sdom:normalize!
		sdom:normalize-document!
		sdom:adopt-node!
		sdom:import-node
		sdom:rename-node!

		sdom:same-node?
		sdom:equal-node?
		sdom:has-child-nodes?
		sdom:supported?

		sdom:default-namespace?		
		sdom:lookup-prefix
		sdom:lookup-namespace-uri

		sdom:set-user-data!
		sdom:get-user-data

		sdom:create-node
		sdom:create-document
		sdom:create-document-type
		
		sdom:get-elements-by-tag-name
		sdom:get-element-by-id
		sdom:set-id-attribute!
		sdom:set-id-attribute-node!
		sdom:get-attribute-node
		sdom:set-attribute-node!
		sdom:remove-attribute-node!
		sdom:get-attribute
		sdom:set-attribute!
		sdom:remove-attribute!

		sdom:compare-document-position
		
		add-namespace
		extract-children
		extract-attributes
		ancestors
		annotate!
		get-prefix
		get-local-name
		document-order
		annotations
		annotation
		single-at-finder))


;; The following constructs get added to the SXML tree to make it into a DOM-
;; compliant document tree. !!! We're going to use sub @-annotations to store
;; our data!

(define sdom:node-type-node 0)
(define sdom:node-type-character-data 100)

(define sdom:node-type-element 			1)
(define sdom:node-type-attr	 		2)
(define sdom:node-type-text 			3)
(define sdom:node-type-cdata-section 		4)
(define sdom:node-type-entity-reference 	5)
(define sdom:node-type-entity 			6)
(define sdom:node-type-processing-instruction 	7)
(define sdom:node-type-comment 			8)
(define sdom:node-type-document			9)
(define sdom:node-type-document-type 		10)
(define sdom:node-type-document-fragment 	11)
(define sdom:node-type-notation 		12)

(define sdom:exception-code-index-size-err 		1)
(define sdom:exception-code-domstring-size-err 		2)
(define sdom:exception-code-hierarchy-request-err 	3)
(define sdom:exception-code-wrong-document-err 		4)
(define sdom:exception-code-invalid-character-err 	5)
(define sdom:exception-code-no-data-allowed-err 	6)
(define sdom:exception-code-no-modification-allowed-err 7)
(define sdom:exception-code-not-found-err 		8)
(define sdom:exception-code-not-supported-err 		9)
(define sdom:exception-code-inuse-attribute-err 	10)
(define sdom:exception-code-invalid-state-err 		11)
(define sdom:exception-code-syntax-err 			12)
(define sdom:exception-code-invalid-modification-err 	13)
(define sdom:exception-code-namespace-err 		14)
(define sdom:exception-code-invalid-access-err 		15)
(define sdom:exception-code-validation-err 		16)
(define sdom:exception-code-type-mismatch-err 		17)

(define sdom:document-position-disconnected 		1)
(define sdom:document-position-preceding 		2)
(define sdom:document-position-following 		4)
(define sdom:document-position-contains 		8)
(define sdom:document-position-contained-by 		16)
(define sdom:document-position-implementation-specific 	32)

(define sdom:user-data-event-node-cloned 	1)
(define sdom:user-data-event-node-imported	2)
(define sdom:user-data-event-node-deleted	3)
(define sdom:user-data-event-node-renamed	4)
(define sdom:user-data-event-node-adopted	5)

(define sdom:event-exception-code-unspecified-event-type-err	0)
(define sdom:event-exception-code-dispatch-request-err		1)

(define sdom:event-phase-capturing	1)
(define sdom:event-phase-target		2)
(define sdom:event-phase-bubbling	3)

(define xml-ns-uri "http://www.w3.org/XML/1998/namespace")
(define xmlns-ns-uri "http://www.w3.org/2000/xmlns")

;; This guy stores the user data across all nodes / documents.  We're not going
;; to export it, because we have a pair of functions that handle access to it.

(define initial-user-data-hash-size 16)
(define user-data-hash (make-hash-table initial-user-data-hash-size))

(define default-dom-error-handler
  (lambda (severity msg excep data loc)
    (display (string-append "SDOM: "
			    (cond ((eqv? severity sdom:error-severity-warning)
				   "warning: ")
				  ((eqv? severity sdom:error-severity-error)
				   "error: ")
				  ((eqv? severity 
					 sdom:error-severity-fatal-error)
				   "fatal error: "))
			    msg))
    (newline)))

;; How to extract the annotations from a particular node?
;; If the node is an element node, annotations are stored in sub attributes
;; like so '(@ (@ ...)).  If the node is an attribute, then the sub-annotation
;; looks like an attribute itself -- but we know it's not, because attributes
;; don't have attributes.

;; (attr (@ (sdom:name "value")))
;; (node (@ (attribute "value")))
;; (node (@ (attribute "value") (@ (sdom:name "value"))))

(define fold
  (lambda (_kons_38 _knil_39 _lis1_40)
    ((letrec ((_lp_41 (lambda (_lis_42 _ans_43)
                        (if (null? _lis_42)
                          _ans_43
                          (_lp_41 (cdr _lis_42)
                                  (_kons_38 (car _lis_42) _ans_43))))))
       _lp_41)
     _lis1_40
     _knil_39)))

;; I feel like this will be useful -- but I don't feel like writing it just yet

(define sdom:node?
  (lambda (x) #t))

(define single-at-finder 
  (lambda (node)
    (find (lambda (item) (and (list? item) (eq? (car item) '@))) node)))

(define multi-at-finder
  (lambda (node)
    (filter (lambda (item) (eq? (car item) '@)) node)))

(define annotations
  (lambda (node)
    (let* ((type (sdom:node-type node))
	   (at-list (single-at-finder node)))
      (if (and at-list (not (null? at-list)))
	  (if (eqv? type sdom:node-type-element)
	      (let ((sub-at-list (single-at-finder (cdr at-list))))
		(if (and sub-at-list (not (null? sub-at-list))) 
		    (cdr sub-at-list) '()))
	      (cdr at-list))
	  '()))))

(define whole-annotation
  (lambda (node annotation-name)
    (let ((result (find (lambda (item) (eq? (car item) annotation-name))
			(annotations node))))
      (if result result '()))))

(define annotation
  (lambda (node annotation-name)
    (let ((whole (whole-annotation node annotation-name)))
      (if (null? whole) whole (cadr whole)))))

(define annotate!
  (lambda (node new-annotation)
    (let ((anntns (annotations node)))
      (if (null? anntns)
	  (let ((type (sdom:node-type node)))
	    (if (eqv? type sdom:node-type-element)
		(let ((attrs (single-at-finder node)))
		  (if (eq? attrs #f)
		      (append! node `((@ (@ ,new-annotation))))
		      (let ((sub-attrs (single-at-finder attrs)))
			(if sub-attrs
			    (let ((x (find (lambda (y) 
					     (eq? (car y) 
						  (car new-annotation)))
					   sub-attr)))
			      (if x 
				  (set-cdr! x (cdr new-annotation))
				  (append! sub-attrs `(,new-annotation))))
			    (append! attrs `((@ ,new-annotation)))))))
		(set-cdr! (last-pair node) `((@ ,new-annotation)))))
	  (let ((x (find (lambda (item) (eq? (car item) (car new-annotation)))
			 anntns)))
	    (if x (set-cdr! x (cdr new-annotation))
		(append! anntns `(,new-annotation))))))))

;; This not only removes the annotation itself, but may also remove the entire
;; annotative node if there are no more annotations left in it.

(define remove-annotation!
  (lambda (node sym)
    (let ((type (sdom:node-type node))
	  (annt (find (lambda (x) (eq? (car x) sym)) (annotations node)))
	  (at-list (single-at-finder node)))
      (if (eqv? type sdom:node-type-element)
	  (for-each (lambda (item)
		      (begin (if (eq? (car item) '@)
				 (delq! annt item))
			     (if (= (length item) 1)
				 (delq! item at-list))))
		    (cdr at-list))
	  (begin 
	    (delq! annt at-list)
	    (if (= (length at-list) 1)
		(delq! at-list node)))))))

;;---------------------------------------------------------------------------;;
;;                                                                           ;;
;; Internal functions for manipulating namespaces                            ;;
;;                                                                           ;;
;;---------------------------------------------------------------------------;;

(define add-namespace
  (lambda (node prefix uri) 
    (let* ((doc (if (eqv? (sdom:node-type node) sdom:node-type-document)
		    node 
		    (sdom:get-dom-property node 'sdom:owner-document)))
	   (new-decl (if (sdom:get-dom-config-parameter 
			  doc "sdom:resolve-new-prefixes")
			 `(,(string->symbol uri) ,uri ,prefix)
			 `(,prefix ,uri ,prefix)))
	  (w (whole-annotation node '*NAMESPACES*)))
      (if (null? w)
	  (annotate! node `(*NAMESPACES* ,new-decl))
	  (let ((n (lookup-prefix-at-node node prefix)))
	    (if (null? n) 
		(append! w `(,new-decl))
		(begin (if (= (length n) 3) 
			   (set-car! (caddr n) prefix) 
			   (append! n `(,prefix)))
		       (set-car! (cadr n) uri))))))))

(define remove-namespace-by-prefix
  (lambda (node prefix) ()))

(define remove-namespace-by-namespace
  (lambda (node uri) ()))

(define lookup-prefix-at-node
  (lambda (node prefix) 
    (let* ((w (whole-annotation node '*NAMESPACES*))
	   (doc (if (eqv? (sdom:node-type node) sdom:node-type-document)
		    node
		    (sdom:get-dom-property node 'sdom:owner-document)))
	   (p (sdom:get-dom-config-parameter doc "sdom:prefer-orig-prefix")))
      (if (null? w) 
	  '() 
	  (let ((ns (find (if p 
			      (lambda (x) (if (= (length x) 3)
					      (eq? (caddr x) prefix)
					      (eq? (car x) prefix)))
			      (lambda (x) (eq? (car x) prefix)))
			  (cdr w))))
	    (if ns ns '()))))))

(define lookup-namespace-at-node
  (lambda (node uri) 
    (let ((w (whole-annotation node '*NAMESPACES*)))
      (if (null? w) 
	  '() 
	  (let ((ns (find (lambda (x) (equal? (cadr x) uri)) (cdr w))))
	    (if ns ns '()))))))

;;---------------------------------------------------------------------------;;
;;                                                                           ;;
;; DOM feature management functions                                          ;;
;;                                                                           ;;
;;---------------------------------------------------------------------------;;

(define sdom-feature-registry '(("Core" . "3.0") ("XML" . "1.0")))
(define sdom:register-feature! 
  (lambda (feature version)
      (if (not (sdom:has-feature? feature version)) 
	  (append! sdom-feature-registry `((,feature . ,version))))))
(define sdom:has-feature? 
  (lambda (feature version) 
    (let ((f (find (lambda (x) (equal? x `(,feature . ,version))) 
		   sdom-feature-registry))) 
      (if (not f) #f #t))))

;;---------------------------------------------------------------------------;;
;;                                                                           ;;
;; DOM configuration functions                                               ;;
;;                                                                           ;;
;;---------------------------------------------------------------------------;;

;; The values after the option name are required supported values; the first
;; is the default.

(define sdom-config-defaults
  `(("canonical-form" #f ,(lambda (x) (not x)))
    ("cdata-sections" #t ,boolean?)
    ("check-character-normalization" #f ,(lambda (x) (not x)))
    ("comments" #t ,boolean?)
    ("datatype-normalization" #f ,(lambda (x) (not x)))
    ("element-content-whitespace" #t ,(lambda (x) (eq? #t x)))
    ("entities" #t ,boolean?)
    ("error-handler" ,default-dom-error-handler ,procedure?)
    ("infoset" () ,boolean? 
     ,(lambda (d x) 
	(if x (begin (sdom:set-dom-config-parameter d "validate-if-schema" #f)
		     (sdom:set-dom-config-parameter d "entities" #f)
		     (sdom:set-dom-config-parameter 
		      d "datatype-normalization" #f)
		     (sdom:set-dom-config-parameter d "cdata-sections" #f)
		     (sdom:set-dom-config-parameter 
		      d "namespace-declarations" #t)
		     (sdom:set-dom-config-parameter d "well-formed" #t)
		     (sdom:set-dom-config-parameter 
		      d "element-content-whitespace" #t)
		     (sdom:set-dom-config-parameter d "comments" #t)
		     (sdom:set-dom-config-parameter d "namespaces")))))
    ("namespaces" #t ,(lambda (x) (eq? #t x)))
    ("namespace-declarations" #t ,boolean?)
    ("normalize-characters" #f ,(lambda (x) (not x)))
    ("split-cdata-sections" #t ,boolean?)
    ("validate" #f ,(lambda (x) (not x)))
    ("validate-if-schema" #f ,(lambda (x) (not x)))
    ("well-formed" #t ,(lambda (x) (eq? #t x)))

    ("sdom:prefer-orig-prefix" #f ,boolean?)
    ("sdom:resolve-new-prefixes" #t ,boolean?)))

(define get-sdom-config-default 
  (lambda (str) 
    (let ((def (find (lambda (x) (equal? (car x) str)) sdom-config-defaults)))
      (if (null? def)
	  (throw 'sdom:exception sdom:exception-code-not-found-err)
	  (cadr def)))))

(define sdom:get-dom-config-parameter
  (lambda (doc str) 
    (if (not (eqv? (sdom:node-type doc) sdom:node-type-document))
	(throw 'sdom:exception sdom:exception-code-type-mismatch-err))
    (let* ((config (whole-annotation doc '*CONFIG*))
	   (match (find (lambda (x) (and (list? x) (equal? (car x) str)))
			config)))
      (if (not match) (get-sdom-config-default str) (cadr match)))))

(define internal-get-dom-config-entry
  (lambda (str)
    (find (lambda (x) (equal? (car x) str)) sdom-config-defaults)))

(define internal-check-dom-config-parameter
  (lambda (doc str val)
    (if (not (eqv? (sdom:node-type doc) sdom:node-type-document))
	(throw 'sdom:exception sdom:exception-code-type-mismatch-err))
    (let ((foo (internal-get-dom-config-entry str)))
      (if (not foo) 
	  (throw 'sdom:exception sdom:exception-code-not-found-err))
      (if (not (apply (caddr foo) val '()))
	  (throw 'sdom:exception sdom:exception-code-not-supported-err)))))

(define sdom:add-dom-config-parameter! 
  (lambda (str . vals) 
    (append! sdom-config-defaults `(,(append `(,str) vals)))))

(define sdom:set-dom-config-parameter!
  (lambda (doc str val)
    (internal-check-dom-config-parameter doc str val)
    (let* ((config (begin (if (null? (annotation doc '*CONFIG*))
			      (annotate! doc '(*CONFIG*)))
			  (whole-annotation doc '*CONFIG*)))
	   (match (find (lambda (x) (and (list? x) (equal? str (car x))))
			config)))
      (if match 
	  (set-car! (cdr match) val)
	  (append! config `((,str ,val))))
      (let ((entry (internal-get-dom-config-entry str)))
	(if (= (length entry) 4)
	    (apply (cadddr entry) doc val '()))))))

(define sdom:can-set-dom-config-parameter?
  (lambda (doc str val)
    (not (false-if-exception (internal-check-dom-config-parameter 
			      doc str val)))))

;;---------------------------------------------------------------------------;;
;;                                                                           ;;
;; Internal functions for managing entities and entity references            ;;
;;                                                                           ;;
;;---------------------------------------------------------------------------;;

(define dereference-entity-reference
  (lambda (node)
    (let ((a (ancestors node))
	  (f (lambda (x) 
	       (let ((g (lambda (y) 
			  (if (and (eqv? (sdom:node-type y)
					 sdom:node-type-entity)
				   (equal? (derive-name y)
					   (derive-name node)))
			      (extract-children y) #f))))
		 (if (find g (extract-children x)) #t #f)))))
      (find f (reverse a)))))

(define update-entity-reference!
  (lambda (node)
    (let ((vals (dereference-entity-reference node)))
      (for-each (lambda (x) (sdom:remove-child! node x)) 
		(extract-children node))
      (if vals (for-each (lambda (x) (sdom:append-child! node x)) vals)))))

;;---------------------------------------------------------------------------;;
;;                                                                           ;;
;; Internal functions for managing internal and external node                ;;
;; representations                                                           ;;
;;                                                                           ;;
;;---------------------------------------------------------------------------;;

;; The motivation for this pair of functions is that according to the SXML
;; grammar, character data, entities, and comments can't have annotations.
;; The solution is to store the annotations in the node's parent and just
;; attach them to the node when we extract it.

(define pos-symbol 
  (lambda (pos) (string->symbol (string-append "@-" (number->string pos)))))

(define ext->int
  (lambda (node)
    (let ((type (sdom:node-type node))
	  (parent (sdom:get-dom-property node 'sdom:parent-node)))
      (if (and (or (eqv? type sdom:node-type-text)
		   (eqv? type sdom:node-type-cdata-section)
		   (eqv? type sdom:node-type-comment)
		   (eqv? type sdom:node-type-entity))
	       (not (null? parent)))
	  (let* ((rep `(sdom:sxml-representation 
			,(sdom:get-dom-property node 
						'sdom:sxml-representation)))
		 (pos (letrec ((f (lambda (x count) 
				    (cond ((null? x) #f)
					  ((and (list? (car x)) 
						(eq? (caar x) '@))
					   (f (cdr x) count))
					  ((eq? (car x) (cadr rep)) count)
					  (else (f (cdr x) (+ count 1)))))))
			(f (cdr parent) 
			   (if (eqv? (sdom:node-type parent) 
				     sdom:node-type-attr) 0 1))))
		 (a-pos (list-index (lambda (x) (and (list? x) 
						     (eq? (car x) '@)))
				    parent))
		 (r-pos (+ (if (< a-pos pos) (+ pos 1) pos)
			   (if (eqv? (sdom:node-type parent)
				     sdom:node-type-attr)
			       1 0)))
		 (val (sdom:get-dom-property node 'sdom:node-value)))
	    (if pos 
		(begin (annotate! parent 
				  (cons (pos-symbol pos)
					`(,(filter (lambda (x)
						     (not (equal? x rep)))
						   (annotations node)))))
		       (cond ((or (eqv? type sdom:node-type-text)
				  (eqv? type sdom:node-type-cdata-section))
			      (list-set! parent r-pos val))))
		(throw 'sdom:exception sdom:exception-code-not-found-err)))))))

(define int->ext
  (lambda (parent child-pos)
    (letrec ((annts (annotation parent (pos-symbol child-pos)))
	     (f (lambda (item count)
		  (cond ((eq? item '()) '())
			((and (list? (car item))
			      (eq? (caar item) '@)) (f (cdr item) count))
			((eqv? count child-pos) 
			 (if (null? annts)
			     (car item)
			     (append (if (list? (car item)) 
					 (car item) 
					 `(,(car item)))
				     `(,(cons '@
					 (append annts
						 `((sdom:sxml-representation 
						    ,(car item)))))))))
			(else (f (cdr item) (+ count 1)))))))
      (if (eqv? (sdom:node-type parent) sdom:node-type-attr)
	  (f (cddr parent) 1)
	  (f (cdr parent) 1)))))

;; Unless the events module is loaded, this is a no-op.

(define sdom:dispatch-event
  (lambda args #f))

(define sdom:dom-implementation-create-document-type
  (lambda (q-name public-id system-id) ()))

(define sdom:dom-implementation-create-document
  (lambda (namespace-uri q-name doc-type) ()))

(define sdom:dom-implementation-get-feature
  (lambda (feature version) ()))

(define sdom:node-type
  (lambda (node) 
    (if (not (list? node)) (throw 'sdom:exception 
				  sdom:exception-code-type-mismatch-err))
    (cond ((eq? (car node) '*TOP*) sdom:node-type-document)
	  ((eq? (car node) '*COMMENT*) sdom:node-type-comment)
	  ((eq? (car node) '*ENTITY*) sdom:node-type-entity)
	  ((eq? (car node) '*ENTITY-REF*) sdom:node-type-entity-reference)
	  ((eq? (car node) '*FRAGMENT*) sdom:node-type-document-fragment)
	  ((eq? (car node) '*DOCTYPE*) sdom:node-type-document-type)
	  ((string? (car node)) 
	   (let ((annts (single-at-finder node)))
	     (if (or (not annts) 
		     (null? annts)
		     (not (find (lambda (x) (equal? x '(sdom:is-cdata #t)))
				(cdr annts))))
		 sdom:node-type-text
		 sdom:node-type-cdata-section)))
	  ((eq? (car node) '*PI*) sdom:node-type-processing-instruction)
	  ((eq? (car node) '*NOTATION*) sdom:node-type-notation)
	  ((eq? (car node) '@) sdom:node-type-attr) 
	  ((symbol? (car node)) sdom:node-type-element)
	  (else throw 'sdom:exception sdom:exception-code-type-mismatch-err))))

(define derive-name
  (lambda (node)
    (let ((type (sdom:node-type node)))
      (cond ((eqv? type sdom:node-type-attr) (symbol->string (cadr node)))
	    ((eqv? type sdom:node-type-cdata-section) "#cdata-section")
	    ((eqv? type sdom:node-type-comment) "#comment")
	    ((eqv? type sdom:node-type-document) "#document")
	    ((eqv? type sdom:node-type-document-fragment) "#document-fragment")
	    ((eqv? type sdom:node-type-document-type) "???")
	    ((eqv? type sdom:node-type-element) (symbol->string (car node)))
	    ((eqv? type sdom:node-type-entity) "???")
	    ((eqv? type sdom:node-type-entity-reference) "???")
	    ((eqv? type sdom:node-type-notation) "???")
	    ((eqv? type sdom:node-type-processing-instruction)
	     (symbol->string (cadr node)))
	    ((eqv? type sdom:node-type-text) "#text")))))

(define derive-value
  (lambda (node)
    (let ((type (sdom:node-type node)))
      (cond ((eqv? type sdom:node-type-attr)
	     (if (sdom:get-dom-property node 'sdom:specified)
		 (if (sdom:has-child-nodes? node)
		     (sdom:get-dom-property (sdom:get-dom-property 
					     node 'sdom:first-child)
					    'sdom:node-value) "")))
		     
;;		 '())) ;; HAVE TO HANDLE ENTITY-REFS, LOOKUPS, ETC. !!!
	    ((or (eqv? type sdom:node-type-cdata-section)
		 (eqv? type sdom:node-type-text)) (car node))
	    ((eqv? type sdom:node-type-comment) (cadr node))
	    ((eqv? type sdom:node-type-processing-instruction) (caddr node))
	    (else '())))))

(define set-value!
  (lambda (node value)
    (let ((type (sdom:node-type node)))
      (if (not (equal? (sdom:get-dom-property node 'sdom:node-value) value))
	  (begin
	    (cond ((eqv? type sdom:node-type-attr)
		   (let* ((d (sdom:get-dom-property node 'sdom:owner-document))
			  (t (sdom:create-node d sdom:node-type-text value))
			  (x (sdom:get-dom-property node 'sdom:first-child)))
		     (cond ((null? x) (sdom:append-child! node t))
			   ((not (equal? value (derive-value node)))
			    (sdom:replace-child! node t x)))))
		  ((or (eqv? type sdom:node-type-cdata-section)
		       (eqv? type sdom:node-type-text))
		   (set-car! node value)
		   (ext->int node)
		   (sdom:dispatch-event
		    node 'sdom:event-dom-character-data-modified))
		  ((eqv? type sdom:node-type-comment) 
		   (set-car! (cdr node) value)
		   (ext->int node))
		  ((eqv? type sdom:node-type-processing-instruction) 
		   (set-car! (cddr node) value)
		   (sdom:dispatch-event 
		    node 'sdom:event-dom-character-data-modified))))))))

(define qname?
  (lambda (str) ()))

(define get-prefix 
  (lambda (str) 
    (let ((i (string-rindex str #\:))) 
      (if (and i (> i 0)) (substring str 0 i) '()))))

;; This is just a placeholder right now...

(define valid-namespace-combo? (lambda (qname uri) #t))

(define get-local-name 
  (lambda (str) 
    (let ((i (string-rindex str #\:))) 
      (if (and i (not (eqv? i (- (string-length str) 1)))) 
	  (substring str (+ i 1)) 
	  str))))

(define set-prefix!
  (lambda (node p)
    (let ((type (sdom:node-type node))
	  (namespace-uri (sdom:get-dom-property node 'sdom:namespace-uri))
	  (name (sdom:get-dom-property node 'sdom:local-name)))
      (if (or (null? namespace-uri)
	      (and (equal? p "xml")
		   (not (equal? namespace-uri xml-ns-uri)))
	      (and (eqv? type sdom:node-type-attr)
		   (or (and (equal? p "xmlns")
			    (not (equal? namespace-uri xmlns-ns-uri)))
		       (equal? (sdom:get-dom-property
				node 'sdom:qualified-name)
			       "xmlns"))))
	  (throw 'sdom:exception sdom:exception-code-namespace-err)
	  (cond ((eqv? type sdom:node-type-attr)
		 (set-car! (cdr node) 
			   (string->symbol (string-append (p ":" name)))))
		((eqv? type sdom:node-type-element)
		 (set-car! (cdr node)
			   (string->symbol (string-append (p ":" name))))))))))

(define extract-attributes
  (lambda (node)
    (let ((type (sdom:node-type node)))
      (cond ((eqv? type sdom:node-type-element)
	     (let ((node-list (single-at-finder (cdr node))))
	       (if (not (eq? node-list #f))
		   (map (lambda (item) (cons '@ item))
			(filter (lambda (item) (not (eq? (car item) '@)))
				(cdr node-list)))
		   '())))
	    (else '())))))

(define extract-children
  (lambda (node)
    (let ((type (sdom:node-type node))
	  (parent-fn (lambda () node))
	  (not-annt-fn (lambda (item) 
			 (or (and (list? item) (not (eq? (car item) '@)))
			     (string? item)))))
      (cond ((eqv? type sdom:node-type-element)
	     (let ((counter 0)
		   (x (filter not-annt-fn (cdr node))))
	       (map (lambda (item) 
		      (set! counter (+ counter 1))
		      (int->ext node counter))
		    x)))
	    ((eqv? type sdom:node-type-document)
	     (let ((child (find not-annt-fn (cdr node))))
	       (if child `(,(int->ext node 1)) '())))

	    ((eqv? type sdom:node-type-document-fragment)
	     (let ((counter 0))
	       (map (lambda (item)
		      (set! counter (+ counter 1))
		      (int->ext node counter))
		    (filter not-annt-fn (cdr node)))))
	    ((eqv? type sdom:node-type-attr)
	     (let ((child (caddr node))) 
	       (if (not (and (list? child) (eq? (car child) '@))) 
		   `(,(int->ext node 1))
		   '())))
	    (else '())))))

(define first-child 
  (lambda (node)
    (let ((child-list (extract-children node)))
      (if (not (null? child-list))
	  (car child-list)
	  '()))))

(define last-child
  (lambda (node)
    (let ((child-list (extract-children node)))
      (if (not (null? child-list))
	  (car (last-pair child-list))
	  '()))))

(define owner-document
  (lambda (node)
    (letrec 
	((type (sdom:node-type node))
	 (top-finder 
	  (lambda (item) 
	    (if (null? item)
		'()
		(let ((owner (annotation item 'sdom:owner-document))
		      (item-type (sdom:node-type item)))
		  (cond ((not (null? owner)) (owner))
			((eqv? item-type sdom:node-type-document) item)
			((eqv? item-type sdom:node-type-attr)
			 (top-finder 
			  (sdom:get-dom-property item 'sdom:owner-element)))
			((or (eqv? item-type sdom:node-type-element)
			     (eqv? item-type sdom:node-type-entity-reference)
			     (eqv? item-type sdom:node-type-document-type)
			     (eqv? item-type 
				   sdom:node-type-processing-instruction)
			     (eqv? item-type sdom:node-type-text)
			     (eqv? item-type sdom:node-type-cdata-section)
			     (eqv? item-type sdom:node-type-comment))
			 (top-finder 
			  (sdom:get-dom-property item 'sdom:parent-node)))
			(else '())))))))
      (if (eqv? type sdom:node-type-document)
	  '()
	  (top-finder node)))))

(define sdom:dom-structure
  `(,sdom:node-type-node 
    (@ (! sdom:read-only 
	  (,(lambda (x) 
	      (let ((r (find (lambda (y) 
			       (eq? (annotation y 'sdom:read-only) #t))
			     (ancestors x))))
		(if r #t #f)))))
       (! sdom:node-type (,sdom:node-type))
       (! sdom:node-name (,derive-name))
       (sdom:node-value (,derive-value) (,set-value!))
       (! sdom:parent-node (,(lambda (node)
			       (let ((p (annotation node 'sdom:parent-node)))
				 (if (procedure? p) (p) '())))))
       (! sdom:child-nodes (,extract-children))
       (! sdom:first-child (,(lambda (node) 
			       (let ((x (extract-children node)))
				 (if (not (null? x)) (car x) '())))))
       (! sdom:last-child (,(lambda (node)
			      (let ((x (extract-children node)))
				(if (not (null? x)) 
				    (car (last-pair x))
				    '())))))
       (! sdom:previous-sibling 
	  (,(lambda (node) 
	      (let ((p (sdom:get-dom-property node 'sdom:parent)))
		(if (not (null? p)) 
		    (let ((q (memq node (extract-children p)))) 
		      (if (> (length q) 1) (cadr q) '())) '())))))
       (! sdom:next-sibling 
	  (,(lambda (node) 
	      (let ((p (sdom:get-dom-property node 'sdom:parent)))
		(if (not (null? p)) 
		    (let ((q (memq node (reverse (extract-children p)))))
		      (if (> (length q) 1) (cadr q) '())) '())))))
       (! sdom:attributes (,extract-attributes))
       (! sdom:owner-document (,owner-document))
       (! sdom:namespace-uri ())
       (sdom:prefix (,(lambda (x) (get-prefix (derive-name x)))) 
		    (,set-prefix!))
       (! sdom:local-name 
	  (,(lambda (x) 
	      (if (not (null? (sdom:get-dom-property x 'sdom:namespace-uri)))
		  (get-local-name (derive-name x)) '()))))
       (! sdom:base-uri ())
       (sdom:text-content () ())
       (sdom:sxml-representation () ()))
    (,sdom:node-type-character-data 
     (@ (sdom:data (,derive-value) (,set-value!))
	(! sdom:length (,(lambda (node) 
			   (let ((v (derive-value node)))
			     (if (not (null? v))
				 (string-length (derive-value node))
				 0))))))
     (,sdom:node-type-text (@ (! sdom:is-element-content-whitespace ())
			      (! sdom:whole-text ()))
			   (,sdom:node-type-cdata-section))
     (,sdom:node-type-comment))
    (,sdom:node-type-notation (@ (! sdom:public-id ())
				 (! sdom:system-id ())))
    (,sdom:node-type-entity (@ (! sdom:public-id ())
			       (! sdom:system-id ())
			       (! sdom:notation-name ())
			       (! sdom:input-encoding ())
			       (! sdom:xml-encoding ())
			       (! sdom:xml-version ())))
    (,sdom:node-type-entity-reference)
    (,sdom:node-type-processing-instruction 
     (@ (! sdom:target (,cadr))
	(sdom:data (,caddr)
		   (,set-value!))))
    (,sdom:node-type-attr (@ (! sdom:name (,derive-name))
			     (! sdom:specified 
				(,(lambda (node)
				    (if (or (annotation node 'sdom:specified)
					    (find (lambda (item) 
						    (eqv? (sdom:node-type item)
							  sdom:node-type-text))
						  (extract-children node)))
					#t 
					#f))))
			     (sdom:value (,derive-value) (,set-value!))
			     (! sdom:owner-element 
				(,(lambda (x)
				    (let ((p (annotation x 
							 'sdom:owner-element)))
				      (if (procedure? p) (p) '())))))
			     (! sdom:schema-type-info ())
			     (! sdom:is-id 
				(,(lambda (x)
				    (not (null? (annotation 
						 x 'sdom:is-id))))))))
    (,sdom:node-type-element (@ (! sdom:tag-name (,derive-name))
				(! sdom:schema-type-info ())))
    (,sdom:node-type-document-type (@ (! sdom:name ())
				      (! sdom:entities ())
				      (! sdom:notations ())
				      (! sdom:public-id ())
				      (! sdom:system-id ())
				      (! sdom:internal-subset ())))
    (,sdom:node-type-document-fragment)
    (,sdom:node-type-document 
     (@ (! sdom:doc-type ())
	(! sdom:implementation ())
	(! sdom:document-element 
	   (,(lambda (x) 
	       (let ((r (find (lambda (y) 
				(eqv? (sdom:node-type y) 
				      sdom:node-type-element))
			      (extract-children x))))
		 (if r r '())))))
	(! sdom:input-encoding ())
	(! sdom:xml-encoding ())
	(sdom:xml-standalone () ())
	(sdom:xml-version () ())
	(sdom:strict-error-checking () ())
	(sdom:document-uri () ())
	(! sdom:dom-config ())))))


(define get-property-info
  (lambda (node-type name)
    (letrec ((matching-path 
	      (lambda (subtree)
		(cond
		 ((null? subtree) #f)
		 ((eqv? (car subtree) node-type)
		  (let ((r (single-at-finder (cdr subtree))))
		    (if r (cdr r) '())))
		 ((and (list? (car subtree))
		       (not (eq? (caar subtree) '@)))
		  (let ((sub-result (matching-path (car subtree))))
		    (if (not sub-result)
			(matching-path (cdr subtree))
			(append (let ((r (single-at-finder (car subtree))))
				  (if r (cdr r) '()))
				sub-result))))
		 (else (matching-path (cdr subtree)))))))
      
      (find (lambda (item) (or (and (eq? (car item) '!)
				    (eq? (cadr item) name))
			       (eq? (car item) name)))
	    (let ((r (matching-path `(,sdom:dom-structure)))) 
	      (if (not r) '() r))))))

(define sdom:get-dom-property
  (lambda (node name)
    (let ((prop-spec (let ((x (get-property-info (sdom:node-type node) name)))
		       (if (and (list? x) (eq? (car x) '!)) (cdr x) x))))
      
      (cond ((eq? prop-spec #f) 
	     (throw 'sdom:exception sdom:exception-code-not-found-err))
	    ((eq? (cadr prop-spec) '()) (annotation node name))
	    (else (apply (caadr prop-spec) (cons node (cdadr prop-spec))))))))

(define sdom:set-dom-property!
  (lambda (node name value) 
    (let ((x (get-property-info (sdom:node-type node) name)))
      (cond ((eq? x #f) 
	     (throw 'sdom:exception sdom:exception-code-not-found-err))
	    ((eq? (car x) '!)
	     (throw 'sdom:exception 
		    sdom:exception-code-no-modification-allowed-err))
	    (else (let ((f (caddr x)))
		    (if (null? f)
			(annotate! node `(,name ,value))
			(apply (car f) `(,node ,value)))))))))

(define ancestors
  (lambda (node)
    (letrec 
	((f (lambda (item)
	      (let ((type (sdom:node-type item)))
		(cons item
		      (cond ((or (eqv? type sdom:node-type-document)
				 (eqv? type 
				       sdom:node-type-document-fragment))
			     '())
			    ((eqv? type sdom:node-type-attr)
			     (let ((g (sdom:get-dom-property 
				       item 'sdom:owner-element)))
			       (if (not (null? g)) (f g) '())))
			    (else
			     (let ((g (sdom:get-dom-property
				       item 'sdom:parent-node)))
			       (if (not (null? g)) (f g) '())))))))))
      (cdr (f node)))))

(define allowed-child-types
  (lambda (node-type)
    (cond ((eqv? node-type sdom:node-type-attr)
	   `(,sdom:node-type-text
	     ,sdom:node-type-entity-reference))
	  ((or (eqv? node-type sdom:node-type-cdata-section)
	       (eqv? node-type sdom:node-type-comment)
	       (eqv? node-type sdom:node-type-document-type)
	       (eqv? node-type sdom:node-type-notation)
	       (eqv? node-type sdom:node-type-processing-instruction)) '())
	  ((eqv? node-type sdom:node-type-document)
	   `(,sdom:node-type-element
	     ,sdom:node-type-processing-instruction
	     ,sdom:node-type-comment
	     ,sdom:node-type-document-type))
	  ((or (eqv? node-type sdom:node-type-document-fragment)
	       (eqv? node-type sdom:node-type-element)
	       (eqv? node-type sdom:node-type-entity)
	       (eqv? node-type sdom:node-type-entity-reference))
	   `(,sdom:node-type-element
	     ,sdom:node-type-processing-instruction
	     ,sdom:node-type-comment
	     ,sdom:node-type-text
	     ,sdom:node-type-cdata-section
	     ,sdom:node-type-entity-reference))
	  (else (throw 'sdom:exception 
		       sdom:exception-code-type-mismatch-err)))))

(define type-allowed-as-child
  (lambda (parent-type child-type) 
    (not (eq? (memv child-type (allowed-child-types parent-type)) #f))))

(define remove-child!
  (lambda (node pos)
    (letrec ((type (sdom:node-type node))
	     (pos-sym (pos-symbol pos))
	     (f (lambda (item count)
		  (cond ((null? item) '())
			((and (list? (car item)) (eq? (caar item) '@)) 
			 (f (cdr item) count))
			((eqv? count pos) 
			 (sdom:dispatch-event 
			  `(,(car item) sdom:event-dom-node-removed))
			 (delq! (car item) node))
			(else (f (cdr item) (+ count 1)))))))
      (f (cdr node) 1)
      (remove-annotation! node pos-sym)
      (for-each 
       (lambda (item) 
	 (let ((str (symbol->string (car item))))
	   (if (and (equal? (substring str 0 1) "@")
		    (>= (string->number (substring str 2)) pos))
	       (set-car! item (pos-symbol (- (string->number (substring str 2))
					     1))))))
       (annotations node)))))

(define insert-child!
  (lambda (node child pos)
    (let ((type (sdom:node-type node))
	  (new-type (sdom:node-type child))
	  (children (extract-children node)))
      (if (and (not (eqv? type sdom:node-type-document))
	       (not (eq? (owner-document node) (owner-document child))))
	  (throw 'sdom:exception sdom:exception-code-wrong-document-err))
      (if (or (not (type-allowed-as-child type new-type))
	      (or (eq? node child)
		  (not (eq? (memv child (ancestors node)) #f)))
	      (and (eqv? type sdom:node-type-document)
		   (not (eq? children '()))))
	  (throw 'sdom:exception sdom:exception-code-hierarchy-request-err))

      (let ((old-parent (sdom:get-dom-property child 'sdom:parent-node)))
	(if (not (null? old-parent))
	    (sdom:remove-child! old-parent child)
	    (remove-annotation! child 'sdom:owner-document)))
      
      (annotate! child `(sdom:parent-node ,(lambda () node)))

      ;; Here we need to actually insert the sxml representation of the node,
      ;; plus change the special annotations for any requisite nodes that
      ;; fall after the insertion.  First, push all numbered annotations up by
      ;; one.

      (for-each 
       (lambda (item) 
	 (let ((str (symbol->string (car item))))
	   (if (and (equal? (substring str 0 1) "@")
		    (>= (string->number (substring str 2)) pos))
	       (set-car! item (pos-symbol (+ (string->number (substring str 2))
					     1))))))
       (annotations node))

      (let* ((base-pos (cond ((eqv? type sdom:node-type-attr) (+ pos 1))
			     (else pos)))
	     (a-pos (list-index (lambda (x) (and (list? x) (eq? (car x) '@)))
				node))
	     (real-pos (if (< a-pos base-pos) (+ base-pos 1) base-pos))

	     ;; This is how we decide whether or not the child needs to be
	     ;; represented by its sxml-representation.

	     (rep (let ((annt (annotation child 'sdom:sxml-representation)))
		    (if (not (null? annt)) annt child))))
	(if (< (length node) real-pos) 
	    (append! node rep)
	    (set-cdr! node 
		      (let ((a (take (cdr node) (- real-pos 1)))
			    (b (take-right (cdr node) 
					   (- (length node) real-pos))))
			(append a `(,rep) b))))))
    (ext->int child)))
      ;; (sdom:dispatch-event (child 'sdom:event-dom-node-inserted)))))

(define list-pos
  (lambda (lst item) 
    (letrec ((f (lambda (x y z)
		  (cond ((null? x) #f) 
			((equal? (car x) y) z) 
			(else (f (cdr x) y (+ z 1)))))))
      (f lst item 1))))

(define sdom:insert-before!
  (lambda (node new-node . ref-node)
    (if (and (not (null? ref-node))
	     (not (null? (car ref-node))))
	(let ((pos (list-pos (extract-children node) (car ref-node))))
	  (if pos 
	      (insert-child! node new-node pos)
	      (throw 'sdom:exception sdom:exception-code-not-found-err)))
	(insert-child! node new-node (+ (length (extract-children node)) 1)))))

(define sdom:insert-after!
  (lambda (node new-node . ref-node)
    (if (and (not (null? ref-node))
	     (not (null? (car ref-node))))
	(let ((pos (list-pos (extract-children node) (car ref-node))))
	  (if pos
	      (insert-child! node new-node (+ pos 1))
	      (throw 'sdom:exception sdom:exception-code-not-found-err)))
	(insert-child! node new-node (+ (length (extract-children node)) 1)))))

(define sdom:remove-child!
  (lambda (node old-child)
    (let ((parent (sdom:get-dom-property old-child 'sdom:parent-node)))
      (if (eq? node parent)
	  (remove-child! node (list-pos (extract-children node) old-child))
	  (throw 'sdom:exception sdom:exception-code-not-found-err)))))

;; FOR MORE-THAN-ONE-STEP MODIFICATIONS, NEED TO CHECK TO SEE WHETHER BOTH
;; STEPS CAN COMPLETE BEFORE ACTUALLY PERFORMING MODIFICATIONS.  NO, YOU
;; HAVEN'T DONE THIS YET!

(define sdom:replace-child!
  (lambda (node new-child old-child)
    (let ((parent (sdom:get-dom-property old-child 'sdom:parent-node)))
      (if (eq? node parent)
	  (let ((pos (list-pos (extract-children node) old-child)))
	    (remove-child! node pos)
	    (insert-child! node new-child pos))))))

(define sdom:append-child!
  (lambda (node new-child)
    (insert-child! node new-child (+ (length (extract-children node)) 1))))

;; I THINK THIS IS A STUPID PIECE OF THE API.  SHOULDN'T STORE THE DATA IN THE
;; TREE ITSELF, SINCE THE SPEC DOESN'T SAY IT'S A DOM PROPERTY.  EXTERNAL HASH
;; MAYBE?

(define handle-user-data-event
  (lambda (node op src dst)
    (let ((node-hash (hashq-ref user-data-hash node)))
      (if node-hash
	  (hash-fold (lambda (key val foo) 
		       (if (and (pair? val) (procedure? (cdr val)))
			   (apply (cdr val) op key (car val) src dst '()))
		       '())
		     '()
		     node-hash)))))
    
(define sdom:set-user-data!
  (lambda (node key data . handler)
    (let ((node-hash (hashq-ref user-data-hash node)))
      (if node-hash
	  (hashq-set! node-hash key (if (and (not (null? handler))
					     (procedure? (car handler)))
					(cons data (car handler))
					(cons data '())))
	  (let ((new-hash-table (make-hash-table initial-user-data-hash-size)))
	    (hashq-set! user-data-hash node new-hash-table)
	    (hashq-set! new-hash-table key (if (and (not (null? handler))
						    (procedure? (car handler)))
					       (cons data (car handler))
					       (cons data '()))))))))

(define sdom:get-user-data
  (lambda (node key) 
    (let ((node-hash (hashq-ref user-data-hash node)))
      (if node-hash 
	  (let ((data (hashq-ref node-hash key)))
	    (if data (car data) '()))
	  '()))))

(define sdom:equal-node? 
  (lambda (node1 node2)
    (let ((node1-type (sdom:node-type node1))
	  (node2-type (sdom:node-type node2))
	  (rec-equal? (lambda (list1 list2)
			(if (and (null? list1) (null? list2))
			    #t
			    (if (sdom:equal-node? (car list1) (car list2))
				(rec-equal (cdr list1) (cdr list2))
				#f)))))
      (cond ((not (eqv? node1-type node2-type)) #f)
	    ((not (and (equal? (sdom:get-dom-property node1 'sdom:node-name)
			       (sdom:get-dom-property node2 'sdom:node-name))
		       (equal? (sdom:get-dom-property node1 'sdom:local-name)
			       (sdom:get-dom-property node2 'sdom:local-name))
		       (equal? (sdom:get-dom-property node1 
						      'sdom:namespace-uri)
			       (sdom:get-dom-property node2
						      'sdom:namespace-uri))
		       (equal? (sdom:get-dom-property node1 'sdom:prefix)
			       (sdom:get-dom-property node2 'sdom:prefix))
		       (equal? (sdom:get-dom-property node1 'sdom:node-value)
			       (sdom:get-dom-property node2 
						      'sdom:node-value)))) #f)
	    ((not (rec-equal? (extract-attributes node1)
			      (extract-attributes node2))) #f)
	    ((not (rec-equal? (extract-children node1)
			      (extract-children node2)) #f))
	    (else #t)))))

(define sdom:has-child-nodes? 
  (lambda (node) (> (length (extract-children node)) 0)))
(define sdom:same-node? (lambda (node1 node2) (eq? node1 node2)))
(define sdom:supported? (lambda (node feature version) ()))

(define internal-ns-lookup
  (lambda (node str sym)
    (let ((type (sdom:node-type node))
	  (aef (lambda (x) 
		 (find (lambda (y) 
			 (eqv? (sdom:node-type y) sdom:node-type-element))
		       (ancestors x)))))
      (cond ((eqv? type sdom:node-type-element)
	     (let ((ns (sdom:get-dom-property node 'sdom:namespace-uri))
		   (prefix (sdom:get-dom-property node 'sdom:prefix))
		   (decls (whole-annotation node '*NAMESPACES*)))

	       ;; We're doing a prefix lookup.
	       
	       (cond ((eq? sym 'prefix)
		      (let ((f (lambda (elt uri orig)
				 (let ((eltns (sdom:get-dom-property 
					       elt 'sdom:namespace-uri))
				       (eltprefix (sdom:get-dom-property
						   elt 'sdom:prefix)))
				   (if (and (not (null? eltns))
					    (equal? eltns uri)
					    (not (null? eltprefix))
					    (let ((r (internal-ns-lookup 
						      orig eltprefix 'ns)))
					      (and r (equal? r uri))))
				       eltprefix
				       (let ((ae (aef elt)))
					 (if ae 
					     (internal-ns-lookup 
					      ae eltprefix sym)
					     '())))))))
			(f node str node)))

		     ;; We're doing a namespaceURI lookup.

		     ((eq? sym 'ns)
		      (cond ((and (not (null? ns)) (null? prefix)) ns)
			    ((not (null? decls)) 
			     (lookup-namespace-at-node node ns))
			    (else (let ((ae (aef node)))
				    (if ae (internal-ns-lookup 
					    ae str sym) #f)))))

		     ;; We're doing a default namespace lookup.

		     ((eq? sym 'default)
		      (if (null? prefix)
			  (equal? str ns)
			  (let ((ae (aef node)))
			    (if ae (internal-ns-lookup ae str sym) #f)))))))
	    ((eqv? type sdom:node-type-document)	     
	     (let ((de (sdom:get-dom-property node 'sdom:document-element)))
	       (if de (internal-ns-lookup de str sym) #f)))
	    ((or (eqv? type sdom:node-type-entity)
		 (eqv? type sdom:node-type-notation)
		 (eqv? type sdom:node-type-document-type)
		 (eqv? type sdom:node-type-document-fragment)) #f)
	    ((eqv? type sdom:node-type-attr)
	     (let ((p (sdom:get-dom-property node 'sdom:owner-element)))
	       (if (not (null? p)) (internal-ns-lookup p str sym) #f)))
	    (else (let ((ae (aef node)))
		    (if ae (internal-ns-lookup ae str sym) #f)))))))

(define sdom:default-namespace?
  (lambda (node namespace-uri)
    (let ((ns (internal-ns-lookup node namespace-uri 'default)))
      (not (eq? ns #f)))))

;; I don't think we support DOM level 2 namespace-declarations as attributes

(define sdom:lookup-namespace-uri
  (lambda (node prefix)
    (let ((ns (internal-ns-lookup node prefix 'ns)))
      (if ns ns '()))))

(define sdom:lookup-prefix
  (lambda (node namespace-uri)
    (let ((prefix (internal-ns-lookup node namespace-uri 'prefix)))
      (if prefix prefix '()))))

(define sdom:clone-node
  (lambda (node deep)
    (let* ((type (sdom:node-type node))
	   (doc (sdom:get-dom-property node 'sdom:owner-document))
	   (val (sdom:get-dom-property node 'sdom:node-value))
	   (name (sdom:get-dom-property node 'sdom:node-name))
	   (new-node (cond ((eqv? type sdom:node-type-element)
			    (let ((x (sdom:create-node 
				      doc type name)))
			      (for-each (lambda (y) 
					  (let ((z (sdom:clone-node y #t)))
					    (sdom:set-attribute-node! x z)))
					(extract-attributes node))
			      (if deep (for-each (lambda (y) 
						   (let ((z (sdom:clone-node 
							     y #t)))
						     (sdom:append-child! x z)))
						 (extract-children node)))
			      x))
			   ((or (eqv? type sdom:node-type-cdata-section)
				(eqv? type sdom:node-type-text))
			    (sdom:create-node 
			     doc type (string-copy val)))
			   ((eqv? type sdom:node-type-attr)
			    (let ((new-node (sdom:create-node 
					     doc type name)))
			      (if (sdom:has-child-nodes? node)
				  (sdom:append-child! 
				   new-node (sdom:clone-node (first-child node)
							     #t)))
			      new-node)))))
      (handle-user-data-event 
       node sdom:user-data-event-node-cloned node new-node)
      new-node)))
    
;;---------------------------------------------------------------------------;;
;;                                                                           ;;
;; Normalization functions for nodes and documents                           ;;
;;                                                                           ;;
;;---------------------------------------------------------------------------;;

(define internal-normalize!
  (lambda (node doc)
    (letrec 
	((f (lambda (w x)
	      (if (not (null? x))
		  (let ((type (sdom:node-type (car x))))
		    (cond ((eqv? type sdom:node-type-entity-reference)
			   (let ((c (dereference-entity-reference node)))
			     (if c (begin 
				     (update-entity-reference! node)
				     (if (not (sdom:get-dom-config-parameter
					       doc "entities"))
					 (begin
					   (sdom:remove-child! w (car x))
					   (for-each (lambda (y)
						       (sdom:append-child! 
							w y))
						     c))))))
			   (f w (extract-children w)))
			  
			  ((and (eqv? type sdom:node-type-cdata-section)
				(not (sdom:get-dom-config-parameter 
				      doc "cdata-sections")))
			   (sdom:replace-child! 
			    w (sdom:create-node 
			       doc sdom:node-type-text 
			       (sdom:get-dom-property (car x) 
						      'sdom:node-value)) 
			    (car x))
			   (f w (extract-children w)))
			  
			  ((and (eqv? type sdom:node-type-comment)
				(not (sdom:get-dom-config-parameter
				      doc "comments")))
			   (sdom:remove-child! w (car x)) 
			   (f w (extract-children w))))))))
	 
	 ;; Need to add support for namespace normalization in here...
	 
	 (g (lambda (w x)
	      (if (not (null? x))
		  (begin 
		    (if (and (eqv? (sdom:node-type (car x)) 
				   sdom:node-type-text)
			     (not (null? (cdr x)))
			     (eqv? (sdom:node-type (cadr x)) 
				   sdom:node-type-text))
			(begin
			  (sdom:set-dom-property! 
			   (car x) 'sdom:node-value
			   (string-append 
			    (sdom:get-dom-property (car x) 'sdom:node-value)
			    (sdom:get-dom-property (cadr x) 'sdom:node-value)))
			  (sdom:remove-child! w (cadr x))
			  (g w (extract-children w))))
		    (g w (cdr x)))))))
      (if (not (null? doc)) (f node (extract-children node)))
      (g node (extract-children node))
      (for-each (lambda (x) (internal-normalize! x doc)) 
		(extract-children node)))))
	     
(define sdom:normalize!
  (lambda (node) (internal-normalize! node '())))

(define sdom:normalize-document!
  (lambda (node) 
    (if (not (eqv? (sdom:node-type node) sdom:node-type-document))
	(throw 'sdom:exception sdom:exception-code-type-mismatch-err)
	(internal-normalize! node node))))

(define sdom:compare-document-position
  (lambda (node1 node2)
    (let ((type1 (sdom:node-type node1))
	  (type2 (sdom:node-type node2))
	  (parent1 (sdom:get-dom-property node1 
					  (if (eqv? type1 sdom:node-type-attr)
					      'sdom:owner-element 
					      'sdom:parent-node)))
	  (parent2 (sdom:get-dom-property node2
					  (if (eqv? type2 sdom:node-type-attr)
					      'sdom:owner-element
					      'sdom:parent-node))))
      (cond ((sdom:same-node? node1 node2) 0)
	    ((or (null? parent1)
		 (null? parent2)
		 (not (eq? (sdom:get-dom-property node1 'sdom:owner-document)
			   (sdom:get-dom-property node2 
						  'sdom:owner-document))))
	     (+ sdom:document-position-disconnected
		sdom:document-position-implementation-specific))
	    ((find (lambda (x) (eq? x node1)) (ancestors node2)) 
	     (+ sdom:document-position-contained-by
		sdom:document-position-following))
	    ((find (lambda (x) (eq? x node2)) (ancestors node1))
	     (+ sdom:document-position-contains
		sdom:document-position-preceding))))))
	    ;; FINISH TK -- NEED TO FIND COMMON CONTAINERS

(define sdom:create-node
  (lambda (document type . args)
    (let ((newnode 
	   (cond ((eqv? type sdom:node-type-attr)
		  `(@ ,(string->symbol (car args))))
		 ((eqv? type sdom:node-type-cdata-section)
		  `(,(car args) (@ (sdom:is-cdata #t))))		 
		 ((eqv? type sdom:node-type-comment)
		  `(*COMMENT* ,(car args)))
		 ((eqv? type sdom:node-type-document-fragment)
		  '(*FRAGMENT*))
		 ((eqv? type sdom:node-type-element)
		  (list (string->symbol (car args))))		 
		 ((eqv? type sdom:node-type-entity-reference))
		 ((eqv? type sdom:node-type-processing-instruction)
		  `(*PI* ,(car args) ,(cadr args)))		 
		 ((eqv? type sdom:node-type-text)
		  `(,(car args)))
		 (else (throw 'sdom:exception 
			      sdom:exception-code-type-mismatch-err)))))
      (cond ((and (= (length args) 2)
		  (or (eqv? type sdom:node-type-attr)
		      (eqv? type sdom:node-type-element)))

	     ;; Need to check the qname / namespace combo here!!!

	     (annotate! newnode `(sdom:namespace-uri ,(cadr args))))
	    ((eqv? type sdom:node-type-comment)
	     (annotate! newnode 
			`(sdom:sxml-representation ,(list-copy newnode))))
	    ((or (eqv? type sdom:node-type-text)
		 (eqv? type sdom:node-type-cdata-section))
	     (annotate! newnode `(sdom:sxml-representation ,(car args)))))
      (if (not (eqv? type sdom:node-type-document-type))
	  (annotate! newnode `(sdom:owner-document ,(lambda () document))))
      newnode)))

(define sdom:create-document
  (lambda (root-name doctype . namespace-uri)
    (letrec ((newdoc `(*TOP* (,(string->symbol root-name)
			      (@ (@ (sdom:parent-node 
				     ,(lambda () newdoc))))))))
      (if (not (null? doctype))
	  (begin
	    (if (not (eqv? (sdom:node-type doctype) 
			   sdom:node-type-document-type))
		(throw 'sdom:exception sdom:exception-code-type-mismatch-err))
	    (if (not (null? (sdom:get-dom-property doctype 
						   'sdom:owner-document)))
		(throw 'sdom:exception sdom:exception-code-wrong-document-err))
	    (annotate! doctype `(sdom:owner-document `(lambda () ,newdoc)))
	    (annotate! newdoc `(sdom:doctype ,doctype))))
      (if (and (not (null? namespace-uri))
	       (not (null? (car namespace-uri))))
	  (annotate! (sdom:get-dom-property newdoc 'sdom:document-element)
		     `(sdom:namespace-uri ,(car namespace-uri))))
      newdoc)))

(define sdom:create-document-type
  (lambda (qname public-id system-id)
    (let ((new-dtd `(*DOCTYPE* ,qname)))
      (if (not (null? public-id))
	  (annotate! new-dtd `(sdom:public-id ,public-id)))
      (if (not (null? system-id))
	  (annotate! new-dtd `(sdom:system-id ,system-id)))
      new-dtd)))

;; I *think* this is correct -- DOM core isn't formally precise on the
;; definition of document order for non-element/attr nodes             - julian

(define document-order
  (lambda (start)
    (if (not (null? start))
	(let ((type (sdom:node-type start))
	      (f (lambda (x y) (append (document-order x) y))))
	  (cond ((eqv? type sdom:node-type-document)
		 (document-order (sdom:get-dom-property 
				  start 'sdom:document-element)))
		((eqv? type sdom:node-type-element)
		 (cons start (fold f '() (append (extract-attributes start)
						 (extract-children start)))))
		((eqv? type sdom:node-type-attr)
		 (cons start (fold f '() (extract-children start))))
		((or (eqv? type sdom:node-type-document-fragment)
		     (eqv? type sdom:node-type-entity)
		     (eqv? type sdom:node-type-entity-reference))
		 (fold f '() (extract-children start)))
		(else `(,start))))
	'())))

(define sdom:get-elements-by-tag-name
  (lambda (doc name . args)
    (let ((type (sdom:node-type doc)))
      (if (not (eqv? type sdom:node-type-document))
	  (throw 'sdom:exception sdom:exception-code-type-mismatch-err))
      (filter (lambda (x) 
		(and (eqv? (sdom:node-type x) sdom:node-type-element)
		     (or (equal? name "*")
			 (equal? (get-local-name (symbol->string (car x))) 
				 name))
		     (if (not (null? args))
			 (let ((ns (car args)))
			   (if (not (string? ns))
			       (throw 'sdom:exception 
				      sdom:exception-code-type-mismatch-err))
			   (or (equal? ns "*")
			       (let ((y (sdom:lookup-prefix 
					 x (get-prefix (car x)))))
				 (if (null? y) #f (equal? ns y)))))
			 #t)))
	      (document-order doc)))))

;; NOT DONE, OBVIOUSLY

(define sdom:import-node
  (lambda (doc node deep)
    (let* ((type (sdom:node-type node))
	   (new-node (cond ((memv type `(,sdom:node-type-document 
					 ,sdom:node-type-document-type))
			    (throw 'sdom:exception 
				   sdom:exception-code-not-supported-err))
			   ((eqv? type sdom:node-type-notation))
			   ((memv type `(,sdom:node-type-attr 
					 ,sdom:node-type-element))
			    (sdom:clone-node node deep)))))
      (annotate! new-node `(sdom:owner-document ,(lambda () doc)))
      (handle-user-data-event 
       node sdom:user-data-event-node-imported node new-node)
      new-node)))

;; INCOMPLETE -- NEED TO ADD NEW NAMESPACE DECL IF NECESSARY!

(define sdom:adopt-node!
  (lambda (doc node)
    (if (not (eqv? (sdom:node-type doc) sdom:node-type-document))
	(throw 'sdom:exception sdom:exception-code-type-mismatch-err))
    (let ((type (sdom:node-type node)))
      (if (memv type `(,sdom:node-type-document
		       ,sdom:node-type-document-type))
	  (throw 'sdom:exception sdom:exception-code-not-supported-err))
      (if (memv type `(,sdom:node-type-notation
		       ,sdom:node-type-entity))
	  (throw 'sdom:exception sdom:exception-code-type-mismatch-err))
      (if (sdom:get-dom-property node 'sdom:read-only)
	  (throw 'sdom:exception 
		 sdom:exception-code-no-modification-allowed-err))

      (if (eqv? type sdom:node-type-attr)
	  (begin 
	    (remove-annotation! node 'sdom:owner-element)
	    (if (not (sdom:get-dom-property node 'sdom:specified))
		(begin
		  (annotate! node `(sdom:value 
				    ,(sdom:get-dom-property node 'sdom:value)))
		  (annotate! node '(sdom:specified #t))))))
      (let ((parent (sdom:get-dom-property node 'sdom:parent-node)))
	(if (not (null? parent)) (sdom:remove-child! parent node))
	(annotate! node `(sdom:owner-document ,(lambda () doc)))))
    node))

(define sdom:rename-node!
  (lambda (node qname ns)
    (let ((type (sdom:node-type)))
      (if (not (memv? type `(,sdom:node-type-attr ,sdom:node-type-element)))
	  (throw 'sdom:exception sdom:exception-code-not-supported-err))
      (if (or (and (not (equal? (get-prefix qname) ""))
		   (null? ns))
	      (and (or (equal? qname "xml")
		       (equal? (get-prefix qname) "xml"))
		   (not (equal? ns ns-xml-uri)))
	      (and (eqv? type sdom:node-type-attr)
		   (or (equal? qname "xmlns")
		       (equal? (get-prefix qname) "xmlns"))
		   (not (equal? ns ns-xmlns-uri))))
	  (throw 'sdom:exception sdom:exception-code-namespace-err))
      (if (eqv? type sdom:node-type-attr)
	  (let ((n (sdom:clone-node node #f)))
	    (set-car! (cdr node) qname)
	    (for-each (lambda (x) (sdom:append-child! n x)) 
		      (extract-children node))
	    (sdom:replace-child! node n))
	  (set-car! node qname))
      (handler-user-data-event 
       node sdom:user-data-event-node-renamed node '()))))
		   
;; Here are some attribute-mangling functions -- the ones that don't deal with
;; nodes explicitly will call into the ones that do.

(define sdom:get-attribute-node
  (lambda (elt name . namespace-uri) 
    (if (not (eqv? (sdom:node-type elt) sdom:node-type-element))
	(throw 'sdom:exception sdom:exception-code-type-mismatch-err))
    (let* ((finder (if (null? namespace-uri) 
		       (lambda (x) (equal? name (sdom:get-dom-property 
						 x 'sdom:name)))
		       (lambda (x) (and (equal? name 
						(sdom:get-dom-property 
						 x 'sdom:local-name))
					(equal? (sdom:get-dom-property 
						 x 'sdom:namespace-uri)
						(car namespace-uri))))))
	   (match (find finder (extract-attributes elt))))
      (if match match '()))))

(define sdom:set-attribute-node!
  (lambda (elt node) 
    (if (not (and (eqv? (sdom:node-type elt) sdom:node-type-element)
		  (eqv? (sdom:node-type node) sdom:node-type-attr)))
	(throw 'sdom:exception sdom:exception-code-type-mismatch-err))
    (if (not (eq? (sdom:get-dom-property elt 'sdom:owner-document)
		  (sdom:get-dom-property node 'sdom:owner-document)))
	(throw 'sdom:exception sdom:exception-code-wrong-document-err))
    (if (not (null? (sdom:get-dom-property node 'sdom:owner-element)))
	(throw 'sdom:exception sdom:exception-code-inuse-attribute-err))
    (let ((x (find (lambda (y) (equal? (sdom:get-dom-property node 'sdom:name)
				       (sdom:get-dom-property y 'sdom:name)))
		   (extract-attributes elt))))
      (annotate! node `(sdom:owner-element ,(lambda () elt)))
      (remove-annotation! node 'sdom:owner-document)
      (if (not x)
	  (append! (single-at-finder elt) `(,(cdr node)))
	  (sdom:replace-child! x node)))))

(define sdom:remove-attribute-node!
  (lambda (elt node) 
    (if (not (and (eqv? (sdom:node-type elt) sdom:node-type-element)
		  (eqv? (sdom:node-type node) sdom:node-type-attr)))
	(throw 'sdom:exception sdom:exception-code-type-mismatch-err))
    (if (not (eq? elt (sdom:get-dom-property node 'sdom:owner-element)))
	(throw 'sdom:exception sdom:exception-code-not-found-err))
    (delq! (cdr node) (single-at-finder elt))
    (let ((doc (sdom:get-dom-property node 'sdom:owner-document)))
      (annotate! node `(sdom:owner-document ,(lambda () doc))))
    (remove-annotation! node 'sdom:owner-element)))

(define sdom:get-attribute
  (lambda (elt name . namespace-uri)
    (let ((node (if (not (null? namespace-uri))
		    (sdom:get-attribute-node elt name (car namespace-uri))
		    (sdom:get-attribute-node elt name))))
      (derive-value node))))

(define sdom:set-attribute!
  (lambda (elt name value . namespace-uri)
    (let ((ns (if (not (null? namespace-uri)) (car namespace-uri) #f)))
      (if (not (valid-namespace-combo? name (if ns ns '())))
	  (throw 'sdom:exception sdom:exception-code-namespace-err))
      (let* ((doc (sdom:get-dom-property elt 'sdom:owner-document))
	     (attr (sdom:create-node doc sdom:node-type-attr name)))
	(if ns (annotate! attr `(sdom:namespace-uri ,ns)))
	(sdom:set-dom-property! attr 'sdom:value value)
	(sdom:set-attribute-node! elt attr)))))

(define sdom:remove-attribute!
  (lambda (elt name . namespace-uri) 
    (let ((attr (if (not (null? namespace-uri))
		    (sdom:get-attribute-node elt name (car namespace-uri))
		    (sdom:get-attribute-node elt name))))
      (if (not (null? attr)) (sdom:remove-attribute-node! elt attr)))))

(define sdom:get-element-by-id
  (lambda (doc id)
    (let ((pred (lambda (x) 
		  (and (eqv? (sdom:node-type x)
			     sdom:node-type-element)
		       (find (lambda (y) 
			       (and (sdom:get-dom-property y 'sdom:is-id)
				    (equal? id (derive-value y))))
			     (extract-attributes x))))))
      (if (not (eqv? (sdom:node-type doc) sdom:node-type-document))
	  (throw 'sdom:exception sdom:exception-code-type-mismatch-err))
      (find pred (document-order doc)))))

(define sdom:set-id-attribute!
  (lambda (node name is-id . args)
    (let ((attr (if (null? args)
		    (sdom:get-attribute-node node name)
		    (sdom:get-attribute-node node name (car args)))))
      (if (null? attr)
	  (throw 'sdom:exception sdom:exception-code-not-found-err))
      (if is-id 
	  (annotate! attr '(sdom:is-id #t)) 
	  (remove-annotation! attr 'sdom:id)))))

(define sdom:set-id-attribute-node!
  (lambda (attr is-id)
    (if (not (eqv? (sdom:node-type attr) sdom:node-type-attr))
	(throw 'sdom:exception sdom:exception-code-type-mismatch-err))
    (if (is-id)
	(annotate! attr '(sdom:is-id #t))
	(remove-annotation! attr 'sdom:is-id))))

(define sdom:sxml->sdom
  (lambda (sxml-tree)
    (letrec 
	((sdom-tree (copy-tree sxml-tree))	 
	 (tag-sibs!
	  (lambda (node-head parent-node pos)
	    (if (string? node-head)		     
		(annotate! parent-node 
			   `(,(pos-symbol pos) 
			     ((sdom:parent-node ,(lambda () parent-node)))))
		(let ((type (sdom:node-type node-head))
		      (attr-fn (lambda (attr-item) 
				 (if (not (eq? (car attr-item) '@))
				     (tag-sibs! (append '(@) attr-item) 
						node-head 1))))
		      (parent-fn (lambda () parent-node))
		      (counter 1))
		  (for-each (cond ((eqv? type sdom:node-type-element)
				   (lambda (item)
				     (if (and (list? item) 
					      (eq? (car item) '@))
					 (for-each attr-fn (cdr item))
					 (begin
					   (tag-sibs! item node-head counter)
					   (set! counter (+ counter 1))))))
				  ((eqv? type sdom:node-type-document)
				   (lambda (item)
				     (if (and (list? item)
					      (not (eq? (car item) '@)))
					 (begin 
					   (tag-sibs! item node-head counter)
					   (set! counter (+ counter 1))))))
				  ((eqv? type sdom:node-type-attr)
				   (lambda (item)
				     (if (not (and (list? item)
						   (eq? (car item) '@)))
					 (begin
					   (tag-sibs! item node-head counter)
					   (set! counter (+ counter 1))))))
				  (else (lambda (item) '())))
			    (cond ((eqv? type sdom:node-type-attr)
				   (cddr node-head))
				  (else (cdr node-head))))
		  (cond ((eqv? type sdom:node-type-attr)
			 (annotate! node-head 
				    `(sdom:owner-element ,parent-fn)))
			((or (eqv? type sdom:node-type-comment)
			     (eqv? type sdom:node-type-entity))
			 (annotate! parent-node 
				    `(,(pos-symbol pos) 
				      ((sdom:parent-node ,parent-fn)))))
			(else (annotate! node-head `(sdom:parent-node 
						     ,parent-fn)))))))))
      (tag-sibs! sdom-tree #f 1)
      sdom-tree)))
