;; ls.scm: Load/Save exports and implementations for SDOM
;; Copyright (C) 2011 Julian Graham

;; SDOM free software: you can redistribute it and/or modify
;; it under the terms of the GNU General Public License as published by
;; the Free Software Foundation, either version 3 of the License, or
;; (at your option) any later version.

;; SDOM is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;; GNU General Public License for more details.

;; You should have received a copy of the GNU General Public License
;; along with SDOM.  If not, see <http://www.gnu.org/licenses/>.

#!r6rs

(library (sdom ls (0 5 1))
  (export sdom:exception-code-parse-err
	  sdom:exception-code-serialize-err
	  
	  sdom:serialize
	  sdom:serialize-filter
	  sdom:xml->sdom)
  (import (rnrs)
	  (srfi :1)
	  (srfi :13)
	  (sxml ssax)
	  (sdom core))

  (define sdom:exception-code-parse-err		81)
  (define sdom:exception-code-serialize-err	82)

  (sdom:register-feature! "LS" "3.0")

  ;; Add these options for the parser (which doesn't exist yet, 
  ;; I don't think...

  (sdom:add-dom-config-parameter! "charset-overrides-xml-encoding" #t boolean?)
  (sdom:add-dom-config-parameter! "disallow-doctype" #f boolean?)
  (sdom:add-dom-config-parameter! "ignore-unknown-character-denormalizations"
				  #t boolean?)
  (sdom:add-dom-config-parameter! "namespaces" #t boolean?)
  (sdom:add-dom-config-parameter! "resource-resolver" '() procedure?)
  (sdom:add-dom-config-parameter! "supported-media-types-only" #f boolean?)
  
  ;; Add these options for the serializer
  
  (sdom:add-dom-config-parameter! "canonical-form" #f boolean?)
  (sdom:add-dom-config-parameter! "discard-default-content" #t boolean?)
  (sdom:add-dom-config-parameter! "format-pretty-print" #f boolean?)
  (sdom:add-dom-config-parameter! "xml-declaration" #t boolean?)
  
  (define (serialize-error-internal node severity msg type excep data loc)
    (let ((doc (sdom:document-element node)))
      (if (or (not (sdom:signal-error doc severity msg type excep data loc))
	      (eqv? severity sdom:error-severity-fatal-error))
	  (throw 'sdom:exception sdom:exception-code-serialize-err))))
  
  (define (s-elt elt indent f fpp ddc xmld)
    (let ((n (sdom:tag-name elt))
	  (spaces (if fpp (make-string indent #\space) "")))
      (string-append spaces
		     "<" 
		     n
		     (let ((a (sdom:attributes elt))
			   (g (lambda (x) (serialize x 1 f fpp ddc xmld))))
		       (fold-right string-append "" (map g a)))
		     (let ((c (sdom:child-nodes elt)))
		       (if (not (null? c))
			   (string-append ">" 
					  (string #\nl)
					  (let ((g (lambda (x)
						     (serialize 
						      x (+ 2 indent) f
						      fpp ddc xmld))))
					    (fold-right string-append 
							"" (map g c)))
					  spaces
					  "</" 
					  n 
					  ">")
			   "/>"))
		     (if fpp (string #\nl) ""))))
  
  (define (s-attr attr indent f fpp ddc xmld)
    (let ((n (sdom:name attr)))
      (string-append (if fpp (make-string indent #\space) " ")
		     n "=\"" (sdom:value attr) "\"")))
  
  (define (serialize n indent f fpp ddc xmld)
    (cond ((sdom:element? n) 
	   (if (f n) (s-elt n indent f fpp ddc xmld) ""))
	  ((sdom:attr? n)
	   (if (and (or (not ddc) (sdom:specified n)) (f n))
	       (s-attr n indent f fpp ddc xmld) ""))
	  ((sdom:text-node? n) 
	   (if (f n) 
	       (string-append (if fpp (make-string indent #\space) "")
			      (sdom:node-value n)
			      (if fpp (string #\nl) ""))
	       ""))
	  ((sdom:cdata-section? n)
	   (if (f n) 
	       (string-append (if fpp (make-string indent #\space) "")
			      "<![CDATA[" 
			      (sdom:node-value n)
			      "]]>"
			      (if fpp (string #\nl) ""))
	       ""))
	  ((sdom:entity-reference? n) (if (f n) "" ""))
	  ((sdom:entity? n) (if (f n) "" ""))
	  ((sdom:processing-instruction? n) 
	   (if (f n) 
	       (let ((target (sdom:target n))
		     (data (sdom:data n)))
		 (if (string-contains target "?>") 
		     (serialize-error-internal
		      n sdom:error-severity-fatal-error 
		      "illegal string in target" "target" '() target '()))
		 (if (string-contains target "?>") 
		     (serialize-error-internal
		      n sdom:error-severity-fatal-error 
		      "illegal string in data" "data" '() data '()))
		 (string-append (make-string indent #\space)
				"<?" target " " data "?>"
				(string #\nl)) "")))
	  ((sdom:comment? n) (if (f n) "" ""))
	  ((sdom:document? n)
	   (if (f n)
	       (s-elt (sdom:document-element n)
		      0
		      f 
		      fpp 
		      ddc
		      xmld)
	       ""))
	  ((sdom:document-type? n) (if (f n) "" ""))
	  ((sdom:document-fragment? n) (if (f n) "" ""))
	  ((sdom:notation? n) (if (f n) "" ""))))
 
  (define (sdom:serialize-filter node filter . port)
    (let* ((doc (if (sdom:document? node) node (sdom:owner-document node)))
	   (r (serialize 
	       node 0 filter
	       (sdom:get-dom-config-parameter doc "format-pretty-print")
	       (sdom:get-dom-config-parameter doc "discard-default-content")
	       (sdom:get-dom-config-parameter doc "xml-declaration"))))
      (if (null? port) r (display r (car port)))))
  
  (define (sdom:serialize node . port)
    (let ((t (lambda (x) #t)))
      (if (null? port) 
	  (sdom:serialize-filter node t)
	  (sdom:serialize-filter node t (car port)))))
  
;; (define internal-dtd-parse-entity
;;   (lambda (port entities)
;;     ()))

;; (define internal-dtd-parse-notation
;;   (lambda (port notations)
;;     ()))

;; (define internal-dtd-parse-attlist
;;   (lambda (port structure)
;;     ()))

;; (define internal-dtd-parse-element
;;   (lambda (port structure)
;;     ()))

;; (define internal-dtd-parse
;;   (lambda (p dn si i-s? seed) 
;;     (let* ((str (let ((f (lambda () (let ((c (read-char p)))
;; 				      (if (eof-object? c)
;; 					  (error #t "gork"))
;; 				      (if (and (eqv? c #\])
;; 					       (eqv? (peek-char p) #\>))
;; 					  (begin (read-char p) ""))
;; 				      (string-append (string c) (f))))))
;; 		  (f)))
;; 	   (entities '())
;; 	   (elements '())
;; 	   (notations '())
;; 	   (g (lambda ()
;; 		(if (not (eof-object? (peek-char sp)))
;; 		    (let ((token (ssax:read-markup-token sp)))
;; 		      (if (eq? (car token) 'DECL)
;; 			  (cond ((eq? (cadr token) 'ENTITY)
;; 				 (set! entities (internal-dtd-parse-entity 
;; 						 sp entities)))
;; 				((eq? (cadr token) 'NOTATION)
;; 				 (set! notations (internal-dtd-parse-notation
;; 						  sp entities)))
;; 				((eq? (cadr token) 'ATTLIST))
;; 				((eq? (cadr token) 'ELEMENT))))
;; 		      (g sp)))))
;;       (begin 
;; 	(g (open-input-string sp))
;; 	(#f entities '() (cons seed (list '@ (list '*DOCTYPE* str)))))))))

;; (define internal-element-parse
;;   (lambda (elem-gi attributes namespaces parent-seed seed)
;;     (let* ((RES-NAME->SXML (lambda (res-name)
;; 			     (string->symbol
;; 			      (string-append
;; 			       (symbol->string (car res-name))
;; 			       ":"
;; 			       (symbol->string (cdr res-name))))))
;; 	   (seed (ssax:reverse-collect-str-drop-ws seed))
;; 	   (attrs (attlist-fold
;; 		   (lambda (attr accum)
;; 		     (cons (list (if (symbol? (car attr)) (car attr)
;; 				     (RES-NAME->SXML (car attr)))
;; 				 (cdr attr)) accum))
;; 		   '() attributes))
;; 	   (ns-id-used (attlist-fold
;; 			(lambda (attr accum)
;; 			  (if (symbol? (car attr)) accum
;; 			      (cons  (caar attr) accum)))
;; 			(if (symbol? elem-gi) '()
;; 			    (list (car elem-gi)))
;; 			attributes))
;; 	   (local-namespaces
;; 	    (map
;; 	     (lambda (ns-id)
;; 	       (let ((ns-elem 
;; 		      (let loop ((namespaces namespaces))
;; 			(cond
;; 			 ((null? namespaces)
;; 			  (assert #f ns-id))  ; can't happen
;; 			 ((eq? (cadar namespaces) ns-id)
;; 			  (car namespaces))
;; 			 (else (loop (cdr namespaces)))))))
;; 		 (list (cadr ns-elem)
;; 		       (symbol->string (cddr ns-elem))
;; 		       (car ns-elem))))
;; 	     ns-id-used))
;; 	   (attrs
;; 	    (if (null? local-namespaces) attrs
;; 		(cons (list '@
;; 		       (cons '*NAMESPACES* local-namespaces))
;; 		      attrs)))
;; 	   (sxml-element
;; 	    (cons 
;; 	     (if (symbol? elem-gi) elem-gi
;; 		 (RES-NAME->SXML elem-gi))
;; 	     (if (null? attrs) seed
;; 		 (cons (cons '@ attrs) seed)))))
;;      (cons sxml-element parent-seed))))

;; (define internal-pi-parse
;;   (lambda (port pi-tag seed)
;;     (cons (list '*PI* pi-tag (ssax:read-pi-body-as-string port)) seed)))

;; (define sdom:xml->sdom
;;   (lambda (port namespace-prefix-assig)
;;     (letrec ((namespaces
;; 	      (map (lambda (el)
;; 		     (cons* #f (car el) (ssax:uri-string->symbol (cdr el))))
;; 		   namespace-prefix-assig))
;; 
;; 	     (parser (ssax:make-parser
;; 		      DOCTYPE 
;; 		      internal-dtd-parse
;; 		      
;; 		      CHAR-DATA-HANDLER
;; 		      (lambda (s1 s2 seed)
;; 			(if (string-null? s2) (cons s1 seed) (cons* s2 s1 seed)))
;; 		      
;; 		      NEW-LEVEL-SEED (lambda (v w x y z) '())
;; 		      
;; 		      UNDECL-ROOT (lambda (x seed) (values #f '() namespaces seed))
;; 
;; 		      FINISH-ELEMENT internal-element-parse
;; 
;; 		      PI ((*DEFAULT* . internal-pi-parse))))
;; 
;; 	     (result (reverse (parser port '()))))
;;      (cons '*TOP* result))))
)