;; events.scm: DOM events exports and implementation for SDOM
;; Copyright (C) 2010 Julian Graham

;; SDOM free software: you can redistribute it and/or modify
;; it under the terms of the GNU General Public License as published by
;; the Free Software Foundation, either version 3 of the License, or
;; (at your option) any later version.

;; SDOM is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;; GNU General Public License for more details.

;; You should have received a copy of the GNU General Public License
;; along with SDOM.  If not, see <http://www.gnu.org/licenses/>.

(library (sdom events (0 5))
  (export sdom:event?
	  sdom:events-enabled
	  sdom:dispatch-event-internal
	  sdom:add-event-listener!
	  sdom:remove-event-listener!
	  sdom:get-event-property
	  
	  sdom:dom-key-location-standard
	  sdom:dom-key-location-left
	  sdom:dom-key-location-right
	  sdom:dom-key-location-numpad)
  (import (rnrs)
	  (srfi :1)
	  (sdom core (0 5)))

  (sdom:register-feature! "Events" "3.0")
  (sdom:register-feature! "UIEvents" "3.0")
  (sdom:register-feature! "TextEvents" "3.0")
  (sdom:register-feature! "MouseEvents" "3.0")
  (sdom:register-feature! "KeyboardEvents" "3.0")
  (sdom:register-feature! "MutationEvents" "3.0")
  (sdom:register-feature! "MutationNameEvents" "3.0")

  (define sdom:dom-key-location-standard 0)
  (define sdom:dom-key-location-left 1)
  (define sdom:dom-key-location-right 2)
  (define sdom:dom-key-location-numpad 3)

  (define sdom:event-exception-code-unspecified-event-type-err	0)
  (define sdom:event-exception-code-dispatch-request-err 1)

  (define sdom:event-phase-capturing 1)
  (define sdom:event-phase-at-target 2)
  (define sdom:event-phase-bubbling 3)

  (define (ancestors node)
    (cond ((not node) '())
	  ((or (sdom:document? node) (sdom:document-fragment? node))
	   (list node))
	  ((sdom:attr? node) 
	   (cons node (ancestors (sdom:owner-element node))))
	  (else (cons node (ancestors (sdom:parent-node node))))))

  (define-record-type (sdom:event-registration 
		       sdom:event-registraton? 
		       sdom:make-event-registration)
    (fields (immutable handler)
	    (immutable namespace-uri)
	    (immutable use-capture)))

  (define (sdom:event-registration-equal? x y)
    (and (sdom:event-registration? x)
	 (sdom:event-registration? y)
         (eq? (sdom:event-registration-handler x)
	      (sdom:event-registration-handler y))
	 (equal? (sdom:event-registration-namespace-uri x)
		 (sdom:event-registration-namespace-uri y))
	 (eqv? (sdom:event-registration-use-capture x)
	       (sdom:event-registration-use-catpure y))))

  (define (sdom:add-event-listener! node type handler capture . uri)
    (if (not (sdom:node? node))
	(raise (sdom:make-exception sdom:exception-code-type-mismatch-err #f)))
    (let* ((listener-registry (sdom:event-listeners node))
	   (l (hashtable-ref listener-registry type #f))
	   (r (sdom:make-event-registration 
	       handler (and (pair? uri) (car uri)) capture)))
      (if l
	  (if (not (find (lambda (x) (sdom:event-registration-equal? x r)) l))
	      (hashtable-set! listener-registry type (cons r l)))
	  (hashtable-set! listener-registry type (list r)))))

  (define (sdom:remove-event-listener! node type handler capture . uri)
    (if (not (sdom:node? node))
	(raise (sdom:make-exception sdom:exception-code-type-mismatch-err #f)))
    (let* ((listener-registry (sdom:event-listeners node))
	   (l (hashtable-ref listener-registry type #f))
	   (r (sdom:make-event-registration
	       handler (and (pair? uri) (car uri)) capture)))
      (if l (hashtable-set!
	     listener-registry
	     type (remp (lambda (x) (sdom:event-registration-equal? x r)) l)))))

  (define-record-type (sdom:event sdom:make-event sdom:event?)
    (fields (immutable type sdom:event-type)
	    (immutable target sdom:event-target)
	    (mutable current-target 
		     sdom:event-current-target 
		     sdom:set-event-current-target!)
	    (mutable phase sdom:event-phase sdom:set-event-phase!)
	    (immutable bubbles sdom:event-bubbles)
	    (immutable cancelable sdom:event-cancelable)
	    (immutable time-stamp sdom:event-time-stamp)
	    (immutable namespace-uri sdom:event-namespace-uri)))

  (define-record-type (sdom:custom-event 
		       sdom:make-custom-event 
		       sdom:custom-event?)
    (parent sdom:event))

  (define-record-type (sdom:ui-event sdom:make-ui-event sdom:ui-event?)
    (parent sdom:event)
    (fields (immutable view sdom:event-view)
	    (immutable detail sdom:event-detail)))

  (define-record-type (sdom:text-event sdom:make-text-event sdom:text-event?)
    (parent sdom:ui-event)
    (fields (immutable data sdom:event-data)))

  (define-record-type (sdom:mouse-event sdom:make-mouse-event sdom:mouse-event?)
    (parent sdom:ui-event)
    (fields (immutable screen-x sdom:event-screen-x)
	    (immutable screen-y sdom:event-screen-y)
	    (immutable client-x sdom:event-client-x)
	    (immutable client-y sdom:event-client-y)
	    (immutable ctrl-key sdom:event-ctrl-key)
	    (immutable shift-key sdom:event-shift-key)
	    (immutable alt-key sdom:event-alt-key)
	    (immutable meta-key sdom:event-meta-key)
	    (immutable button sdom:event-button)
	    (immutable related-target sdom:event-related-target)))

  (define-record-type (sdom:keyboard-event 
		       sdom:make-keyboard-event 
		       sdom:keyboard-event?)
    (parent sdom:ui-event)
    (fields (immutable key-identifier sdom:event-key-identifier)
	    (immutable key-location sdom:event-key-location)
	    (immutable ctrl-key sdom:event-ctrl-key)
	    (immutable shift-key sdom:event-shift-key)
	    (immutable alt-key sdom:event-alt-key)
	    (immutable meta-key sdom:event-meta-key)))

  (define-record-type (sdom:mutation-event 
		       sdom:make-mutation-event
		       sdom:mutation-event?)
    (parent sdom:event)
    (fields (immutable related-node sdom:event-related-node)
	    (immutable prev-value sdom:event-prev-value)
	    (immutable new-value sdom:event-new-value)
	    (immutable attr-name sdom:event-attr-name)
	    (immutable attr-change sdom:event-attr-change)))

  (define-record-type (sdom:mutation-name-event
		       sdom:make-mutation-name-event
		       sdom:mutation-name-event)
    (parent sdom:mutation-event)
    (fields (immutable prev-namespace-uri sdom:event-prev-namespace-uri)
	    (immutable prev-node-name sdom:event-prev-node-name)))

  ;; Dispatch should do some checking to make sure there's enough data...

  (define (make-event type target . data)
    (case (string->symbol type)
      ((load unload abort error select change submit reset)
       (sdom:make-event type target))

      ((blur
	DOMActivate 
	DOMFocusIn 
	DOMFocusOut 
	focus 
	focusin 
	focusout 
	resize 
	scroll)
       (sdom:make-ui-event type target #f))

      ((click
	dblclick
	mousedown
	mouseenter
	mouseleave
	mousemove
	mouseover
	mouseout
	mouseup)
       (let-values
	   (((s-x s-y c-x c-y c-key s-key a-key m-key button)
	     (apply values args)))
	 (sdom:make-mouse-event 
	  type target s-x s-y c-x c-y c-key s-key a-key m-key button)))

      ((mousewheel)
       (let-values
	   (((s-x s-y c-x c-y c-key s-key a-key m-key button wdelta)
	     (apply values args)))
	 (sdom:make-mouse-wheel-event
	  type target s-x s-y c-x c-y c-key s-key a-key m-key button wdelta)))

      ((wheel) 
       (let-values
	   (((s-x s-y c-x c-y c-key s-key a-key m-key button dx dy dz dmode)
	     (apply values args)))
	 (sdom:make-wheel-event 
	  type target s-x s-y c-x c-y c-key s-key a-key m-key button dx dy dz 
	  dmode)))

      ((textInput) (sdom:make-text-event type target (car data) (cadr data)))

      ((keydown keypress keyup) 
       (let-values
	   (((key-id key-loc c-key s-key a-key m-key repeat)
	     (apply values data)))
       (sdom:make-keyboard-event 
	type target key-id key-loc c-key s-key a-key m-key repeat)))

      ((compositionstart compositionupdate compositionend) 
       (sdom:make-composition-event type target (car data)))

      ((DOMAttrModified
	DOMCharacterDataModified
	DOMNodeInserted
	DOMNodeInsertedIntoDocument
	DOMNodeRemoved
	DOMNodeRemovedFromDocument
	DOMSubtreeModified)
       (let-values 
	   (((related-node prev-value new-value attr-name attr-change)
	     (apply values data)))
	 (sdom:make-mutation-event 
	  type target related-node prev-value new-value attr-name attr-change)))

      ((DOMAttributeNameChanged DOMElementNameChanged)
       (let-values 
	   (((related-node prev-value new-value attr-name attr-change
	      prev-namespace-uri prev-node-name)
	     (apply values data)))
       (sdom:make-mutation-name-event 
	type target related-node prev-value new-value attr-name attr-change 
	prev-namespace-uri prev-node-name)))

      (else (raise (sdom:make-exception 
		    sdom:exception-code-type-mismatch-err #f)))))

  (define (sdom:dispatch-event-internal target event . data)
    (define (dispatch-event-at-node node event capturing)
      (let* ((event-listeners (sdom:event-listeners node))
	     (l (hashtable-ref event-listeners (sdom:event-type event) #f)))
	(if l (for-each 
	       (lambda (r)
		 (if (or (not (eqv? (sdom:event-phase event)
				    (sdom:event-phase-capturing)))
			 (sdom:event-registration-use-capture r))
		     ((sdom:event-registration-handler r) event)))
	       l))))

    (if (not (sdom:node? target))
	(raise (sdom:make-exception sdom:exception-code-type-mismatch-err #f)))
    (let ((event (apply make-event (cons event (cons target data))))
	  (a (cons target (ancestors target))))
      (sdom:set-event-phase! event sdom:event-phase-capturing)
      (for-each (lambda (n) (dispatch-event-at-node n event #t)) a)
      (sdom:set-event-phase! event sdom:event-phase-at-target)
      (dispatch-event-at-node target event #f)
      (if (sdom:event-bubbles event)
	  (begin 
	    (sdom:set-event-phase! event sdom:event-phase-bubbling)
	    (for-each (lambda (n) (dispatch-event-at-node n event #f)) 
		      (reverse a))))))

  (sdom:register-event-dispatcher! sdom:dispatch-event-internal)
)
