/*
SDX: Documentary System in XML.
Copyright (C) 2000, 2001, 2002  Ministere de la culture et de la communication (France), AJLSM

Ministere de la culture et de la communication,
Mission de la recherche et de la technologie
3 rue de Valois, 75042 Paris Cedex 01 (France)
mrt@culture.fr, michel.bottin@culture.fr

AJLSM, 17, rue Vital Carles, 33000 Bordeaux (France)
sevigny@ajlsm.com

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
See the GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the
Free Software Foundation, Inc.
59 Temple Place - Suite 330, Boston, MA  02111-1307, USA
or connect to:
http://www.fsf.org/copyleft/gpl.html
*/
package fr.gouv.culture.sdx.search.lucene.analysis;

import fr.gouv.culture.sdx.exception.SDXException;
import fr.gouv.culture.sdx.exception.SDXExceptionCode;
import fr.gouv.culture.sdx.search.lucene.Field;
import fr.gouv.culture.sdx.search.lucene.FieldsDefinition;
import org.apache.lucene.analysis.TokenStream;

import java.io.Reader;

/**
 * A dispatcher for field specific analyzers.
 *
 * <p>
 * This analyzer knows about specific analyzers for the fields
 * of a document base. Hence, when its tokenStream(fieldName, reader) method
 * is called, it attempts to dispatch the task to a specific analyzer for the
 * given field name.
 */
public class MetaAnalyzer extends AbstractAnalyzer {

    /** The fields definition object. */
    private FieldsDefinition fields;

    /** The default analzyer to use. */
    private Analyzer defaultAnalyzer;

    /**Creates a MetaAnalyzer
     *
     *  <p>
     * A logger must be set and then this MetaAnalyzer must be setUp.
     *
     * @see #enableLogging
     * @see #setUp
     */
    public MetaAnalyzer() {
    }

    /**
     * Builds an analyzer with definitions for fields.
     *
     * @param   fields  The fields and their definitions (cannot be null).
     */
    public void setUp(FieldsDefinition fields) throws SDXException {

        if (fields == null) throw new SDXException(logger, SDXExceptionCode.ERROR_FIELDS_DEF_NULL, null, null);
        this.fields = fields;
        this.defaultAnalyzer = fields.getAnalyzer();
    }

    /**
     * Tokenizes contents.
     *
     * <p>
     * The tokenizing is never done by this object. Instead, using the provided
     * field name, it will try to get a specific analyzer from the fields definition.
     * If it can't find one, it will use the default analyzer.
     *
     * @param fieldName The field name for tokenization
     * @param reader    The reader
     */
    public TokenStream tokenStream(String fieldName, Reader reader) {
        // If the fieldname is not specified, then we use the default analyzer
        if (fieldName == null) return defaultAnalyzer.tokenStream(fieldName, reader);

        Field f = fields.getField(fieldName);

        // If we have no definition for this field, we use the default analyzer again
        if (f == null) return defaultAnalyzer.tokenStream(fieldName, reader);

        Analyzer theAnalyzer = f.getAnalyzer();

        // If the field's analyzer is not defined, then we use the default analyzer
        if (theAnalyzer == null) return defaultAnalyzer.tokenStream(fieldName, reader);

        // Finally, we can use the specific analyzer for this field
        return theAnalyzer.tokenStream(fieldName, reader);
    }

    /** Returns a the FieldsDefinition for this MetaAnalyzer (basically a Hashtable of all the Fields)*/
    //TODO?:is this still necessary, as it exists both in LuceneIndex and MetaAnalyzer?-rbp
    public FieldsDefinition getFieldsDefinition() {
        return this.fields;
    }


}
