/*
SDX: Documentary System in XML.
Copyright (C) 2000, 2001, 2002  Ministere de la culture et de la communication (France), AJLSM

Ministere de la culture et de la communication,
Mission de la recherche et de la technologie
3 rue de Valois, 75042 Paris Cedex 01 (France)
mrt@culture.fr, michel.bottin@culture.fr

AJLSM, 17, rue Vital Carles, 33000 Bordeaux (France)
sevigny@ajlsm.com

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
See the GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the
Free Software Foundation, Inc.
59 Temple Place - Suite 330, Boston, MA  02111-1307, USA
or connect to:
http://www.fsf.org/copyleft/gpl.html
*/

package fr.gouv.culture.sdx.pipeline;

import fr.gouv.culture.sdx.exception.SDXException;
import fr.gouv.culture.sdx.exception.SDXExceptionCode;
import fr.gouv.culture.sdx.utils.Utilities;
import org.apache.avalon.framework.component.ComponentException;
import org.apache.avalon.framework.component.ComponentManager;
import org.apache.avalon.framework.configuration.Configuration;
import org.apache.avalon.framework.configuration.ConfigurationException;
import org.apache.avalon.framework.parameters.Parameters;
import org.apache.cocoon.Constants;
import org.apache.cocoon.serialization.XMLSerializer;
import org.apache.cocoon.xml.AbstractXMLPipe;
import org.apache.cocoon.xml.XMLConsumer;
import org.apache.cocoon.xml.XMLMulticaster;

import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.util.Hashtable;
import java.util.LinkedList;
import java.util.Vector;

/**An abstract implementation of a Pipeline for manipulation of SAX events
 *
 */
public abstract class AbstractPipeline extends AbstractXMLPipe implements Pipeline {

    /**The configruation object*/
    protected Configuration configuration = null;

    /**The parameters for this pipeline*/
    private Parameters params;

    /**The id for the pipeline. */
    protected String id = "";

    /**A list of the transformations that make up this pipeline. */
    protected LinkedList transformations = new LinkedList();

    /** Avalon logger to write information. */
    protected org.apache.avalon.framework.logger.Logger logger;

    /**A list of the transformations that make up this pipeline. */
    protected Vector transformsList = new Vector();

    /** The a component manager */
    protected ComponentManager manager;

    /*The transformation step of which the results should be stored*/
    protected String keepStep = null;

    /**A output stream for the event results of a transformation*/
    private ByteArrayOutputStream transformedBytes = null;

    /**A output stream for the event results of a transformation*/
    private File transformedFile = null;

    FileOutputStream fOs = null;

    private XMLSerializer xsFile = null;

    private XMLSerializer xsBytes = null;

    //TODORefactor?:what should these strings be?
    /**prefix for our tempfile created for the fileOuputStream*/
    private String TEMPFILE_PREFIX = "tempTransDoc";

    /*suffix for our tempfile created for the fileOuputStream*/
    private String TEMPFILE_SUFFIX = ".sdx";
    /**The Pipelines properties. */
    protected Hashtable props;

    /** Sets the logger for the pipeline
     *
     * @param logger   The logger created from Cocoon's logkit.
     */
    public void enableLogging(org.apache.avalon.framework.logger.Logger logger) {
        this.logger = logger;
    }

    /**
     * Sets the pipeline's ComponentManager.
     *
     * @param manager  The ComponentManager to use.
     * @throws org.apache.avalon.framework.component.ComponentException
     */
    public void compose(ComponentManager manager) throws ComponentException {
        this.manager = manager;
    }

    /**
     * Configures the pipeline.
     *
     * @param configuration  The configuration object provided from a document base object.
     *
     *<p>Retrieves the children elements named "transformation" and builds a
     * transformation object for each element. Currently the only supported
     * transformation type is "xslt".
     *
     * <p>Sample configuration snippet:
     * <p>&lt;sdx:pipeline sdx:id = "sdxIndexationPipeline">
     *      <p>&lt;sdx:transformation src = "path to stylesheet, can be absolute or relative to the directory containing this file" sdx:id = "step2" sdx:type = "xslt"/>
     *      <p>&lt;sdx:transformation src = "path to stylesheet, can be absolute or relative to the directory containing this file" sdx:id = "step3" sdx:type = "xslt" keep = "true"/>
     * <p>&lt;/sdx:pipeline>
     * @throws org.apache.avalon.framework.configuration.ConfigurationException
     */
    public void configure(Configuration configuration) throws ConfigurationException {
        //Utilities.checkConfiguration(configuration);
        this.configuration = configuration;

        loadBaseConfiguration(configuration);
        try {
            configureTransformations(configuration);
        } catch (SDXException sdxE) {
            throw new ConfigurationException(sdxE.getMessage(), sdxE);
        }
        if (configuration != null) {
            this.params = Parameters.fromConfiguration(configuration);
            verifyConfigurationParameters(this.params);
        }

    }

    protected void verifyConfigurationParameters(Parameters params) {
        if (params != null) {
            String[] paramNames = params.getNames();
            for (int i = 0; i < paramNames.length; i++) {
                String paramName = paramNames[i];
                if (Utilities.checkString(paramName)) {
                    String paramValue = params.getParameter(paramName, null);
                    if (!Utilities.checkString(paramValue) || paramValue.equals("null"))
                        params.removeParameter(paramName);
                }


            }
        }

    }

    private void loadBaseConfiguration(Configuration configuration) {
        //if we don't already have a good id we find one fromt the configuration
        if (!Utilities.checkString(getId())) {
            if (configuration != null)
                id = configuration.getAttribute(Pipeline.ATTRIBUTE_ID, "");
        }
    }

    private void configureTransformations(Configuration configuration) throws ConfigurationException, SDXException {

        //getting the <sdx:transformation> children elements of the  <sdx:pipeline> element
        Configuration[] transforms = null;
        if (configuration != null)
            transforms = configuration.getChildren(Transformation.ELEMENT_NAME_TRANSFORMATION);
        //testing to see if we have something
        /*      if (transforms == null || transforms.length == 0) {
                  String[] args = new String[1];
                  args[0] = configuration.getLocation();
                  SDXException sdxE = new SDXException(logger, SDXExceptionCode.ERROR_NO_TRANSFORMATIONS_IN_CONFIG, null, null);
                  throw new ConfigurationException(sdxE.getMessage(), sdxE);
              }
         */
        if (transforms != null) {
            /*
            if (transforms.length == 0) {
                String[] args = new String[1];
                args[0] = configuration.getLocation();
                SDXException sdxE = new SDXException(logger, SDXExceptionCode.ERROR_NO_TRANSFORMATIONS_IN_CONFIG, null, null);
                throw new ConfigurationException(sdxE.getMessage(), sdxE);
            }
            */
            //iterating over the configuration elements to create the necessary transformations and add them to the pipeline
            for (int i = 0; i < transforms.length; i++) {
                Transformation trans = null;
                //getting the "type" attribute
                final String transType = transforms[i].getAttribute(Transformation.ATTRIBUTE_TYPE);
                //verifying the attribute
                Utilities.checkConfAttributeValue(Transformation.ATTRIBUTE_TYPE, transType, transforms[i].getLocation());
                //building the fully qualified class name based on the type
                String transClassName = transType;
                //getting the class for the class name
                Class transClass = null;
                try {
                    transClass = Class.forName(transClassName);
                } catch (ClassNotFoundException e) {
                    //logging the first failure
                    String[] args = new String[2];
                    args[0] = transforms[i].getAttribute(ATTRIBUTE_ID, "");
                    args[1] = e.getMessage();
                    //logging first exception
                    SDXException sdxE = new SDXException(null, SDXExceptionCode.ERROR_CONFIGURE_TRANSFORMATION, args, e);
                    Utilities.logWarn(logger, sdxE.getMessage(), null);
                    //trying the attribute value, hopefully the user is providing a fully qualified attribute name
                    transClassName = Transformation.PACKAGE_QUALNAME + transType.substring(0, 1).toUpperCase() + transType.substring(1, transType.length()) + Transformation.CLASS_NAME_SUFFIX;
                    try {
                        transClass = Class.forName(transClassName);
                    } catch (ClassNotFoundException e1) {
                        String[] args2 = new String[2];
                        args2[0] = transforms[i].getAttribute(ATTRIBUTE_ID, "");
                        args2[1] = e1.getMessage();
                        SDXException sdxE2 = new SDXException(logger, SDXExceptionCode.ERROR_CONFIGURE_TRANSFORMATION, args2, e1);
                        throw new ConfigurationException(sdxE2.getMessage(), sdxE2);
                    }
                }
                //building a new instance of the class
                Object obj = null;
                try {
                    obj = transClass.newInstance();
                } catch (InstantiationException e) {
                    String[] args = new String[2];
                    args[0] = transforms[i].getAttribute(ATTRIBUTE_ID, "");
                    args[1] = e.getMessage();
                    throw new SDXException(null, SDXExceptionCode.ERROR_CONFIGURE_TRANSFORMATION, args, e);
                } catch (IllegalAccessException e) {
                    String[] args = new String[2];
                    args[0] = transforms[i].getAttribute(ATTRIBUTE_ID, "");
                    args[1] = e.getMessage();
                    throw new SDXException(null, SDXExceptionCode.ERROR_CONFIGURE_TRANSFORMATION, args, e);
                }
                //we couldn't get an object
                if (obj == null) {
                    String[] args = new String[1];
                    args[0] = transClassName;
                    throw new SDXException(logger, SDXExceptionCode.ERROR_NEW_OBJECT_INSTANCE_NULL, args, null);
                }

                if (!(obj instanceof Transformation)) {
                    //the object doesn't implement our interface
                    String[] args = new String[3];
                    args[0] = "Transformation";
                    args[1] = transClass.getName();
                    args[2] = transType;
                    throw new SDXException(logger, SDXExceptionCode.ERROR_CLASS_NOT_INSTANCEOF_SDX_INTERFACE, args, null);
                }
                //casting into a transformation object
                trans = (Transformation) obj;
                //setting the logger
                trans.enableLogging(this.logger);
                //passing component manager (actually Cocoon's one) to the pipeline object
                try {
                    trans.compose(manager);
                } catch (ComponentException e) {
                    throw new SDXException(null, SDXExceptionCode.ERROR_LOOKUP_FRAMEWORK_COMPONENT, null, e);
                }
                //setting the properties
                trans.setProperties(props);
                //setting up the xslt transformation
                trans.configure(transforms[i]);
                //adding the transformation to the this pipeline
                //addTransformation(trans);
                String transId = trans.getId();
                if (Utilities.checkString(transId)) {
                    if (trans.shouldKeepResult())
                        keepStep = transId;

                    //adding the transformation to the hashtable
                    transformsList.add(trans);
                }

            }
            //we need to build the pipeline after we have all the transformations and have determined the keep step
            for (int k = 0; k < transformsList.size(); k++) {
                Transformation t = (Transformation) transformsList.get(k);
                if (t != null) addTransformation(t);
            }
        }


    }

    /**
     * Adds a transformation to the pipeline and sets the consumer of the last
     * transformation to be the newly added transformation.
     * @param t     The transformation to be added.
     */
    public void addTransformation(Transformation t) throws SDXException {
        if (transformations.size() > 0) {
            //getting the last transformation
            Transformation lastTrans = ((Transformation) transformations.getLast());
            //if we want to retain the events from this step we bridge two consumers
            XMLConsumer consumer = null;
            if (lastTrans != null && lastTrans.getId() != null && lastTrans.getId().equals(keepStep))
            //creating the special consumer to collect the events
                consumer = this.multiCastEvents(t);
            else
                consumer = t;
            //recycling, the content, lexical, and xmlConsumer
            /*this is a necessary step to ensure that each transformation has a new
            consumer for each time this transformation is executed by the pipeline*/
            /*it is not very evident at this time that this step is useful in this method,
            but if we ever have to REconfigure this pipeline, this will be necessary*/
            lastTrans.recycle();

            //this sets the consumer of the last step in the pipeline
            lastTrans.setConsumer(consumer);
        } else {
            //if not we ONLY have the pipeline and need to set it's consumer to be the new transformation
            //recycling, the content, lexical, and xmlConsumer
            /*this is a necessary step to ensure that the pipeline has a new
            consumer*/
            /*it is not very evident at this time that this step is useful in this method,
            but if we ever have to REconfigure this pipeline, this will be necessary*/
            super.recycle();
            //setting consumer and handlers
            super.xmlConsumer = t;
            super.contentHandler = t;
            super.lexicalHandler = t;
        }
        //adding the new transformation to the data structure
        transformations.add(t);
    }

    /**
     * Sets the consumer of the pipeline.
     *
     * @param xmlConsumer   The XMLConsumer for the LAST step of the pipeline
     */
    public void setConsumer(XMLConsumer xmlConsumer) {
        //verifying the consumer
        try {
            Utilities.checkXmlConsumer(logger, xmlConsumer);
        } catch (SDXException e) {
            /*doing nothing here as the exception should have been logged,
            as we are governed by superClasses for the exceceptions signatures*/
        }
        if (transformations.size() > 0) {
            //getting the last transformation
            Transformation lastTrans = ((Transformation) transformations.getLast());
            //recycling, the content, lexical, and xmlConsumer
            /*this is a necessary step to ensure that each transformation has a new
            consumer for each time this transformation is executed by the pipeline*/
            lastTrans.recycle();


            if (Utilities.checkString(keepStep)) {

                //if we want to retain the events from the last step we bridge two consumers
                if (lastTrans != null && lastTrans.getId() != null && lastTrans.getId().equals(keepStep))
                //creating the special consumer to collect the events
                    xmlConsumer = this.multiCastEvents(xmlConsumer);

                /*if we have a valid id for a keep step IN THE PIPELINE(not only the last step)
                *we reset the necessary transformed output event collectors, this is necessary for output
                */
                try {
                    resetTransformedDocumentFields();
                } catch (SDXException e) {
                    /*doing nothing here as the exception should have been logged,
                    as we are governed by superClasses for the exceceptions signatures*/
                }
            }

            //this sets the consumer of the last step in the pipeline
            lastTrans.setConsumer(xmlConsumer);

        } else {
            //TODOLogging:better warn message here, i dont think this should ever arrive as we check on configuration
            //Utilities.logWarn(logger, "no transformations defined", null);
            super.setConsumer(xmlConsumer);
        }


    }

    private void resetTransformedDocumentFields() throws SDXException {
        //resetting the resources output transformation output
        createTempFileResources();
        createTempByteResources();
    }

    /**One should ensure that the createTempFileResources
     *
     */
    //TODO?:should this be in the interface?
    private XMLConsumer multiCastEvents(XMLConsumer xmlConsumer) {

        XMLMulticaster connector = null;
        //buliding the serializers which will supply the output streams
        if (canBuildTempFile() && this.xsFile == null)
            this.xsFile = new XMLSerializer();

        if (this.xsBytes == null)
            this.xsBytes = new XMLSerializer();

        if (this.xsFile != null) {
            XMLMulticaster serializedOutput = new XMLMulticaster(xsBytes, xsFile);
            //building a mutli caster to connect the output serializers and the provided xmlConsumer to the transformation step
            connector = new XMLMulticaster(serializedOutput, xmlConsumer);
        } else {
            connector = new XMLMulticaster(this.xsBytes, xmlConsumer);
        }

        return connector;


    }

    /**Sets the Parameters for each step in the Pipeline
     *
     * @param params    The Parameters to use for each transformation step.
     *
     * <p>Each transformation step in the pipeline can call (getParameters() defined
     * in the Transformation interface) and make use this Parameters object as needed
     * before pipeline processing begins.</p>
     */
    public void setParameters(Parameters params) {
        //setting the class field
        this.params = params;
        setParametersToTransformations(this.params);
    }

    protected void setParametersToTransformations(Parameters parameters) {
        if (transformations != null && !transformations.isEmpty()) {
            Object[] transforms = transformations.toArray();
            for (int i = 0; i < transforms.length; i++) {
                Transformation t = (Transformation) transforms[i];
                t.setParameters(parameters);
            }
        }
    }

    public String getId() {
        return id;
    }

    /**Returns a new instance of this object
     *
     * @return A new instance of this object
     */
    public Pipeline newInstance() throws SDXException {
        try {
            Pipeline pipe = (Pipeline) this.getClass().newInstance();
            //setting the logger
            pipe.enableLogging(this.logger);
            //passing component manager (actually Cocoon's one) to the DB
            pipe.compose(this.manager);
            //setting the properties
            pipe.setProperties(this.props);
            //configuring
            pipe.configure(this.configuration);
            return pipe;
        } catch (ComponentException e) {
            String[] args = new String[2];
            args[0] = id;
            args[1] = e.getMessage();
            throw new SDXException(logger, SDXExceptionCode.ERROR_NEW_INSTANCE, args, null);
        } catch (ConfigurationException e) {
            String[] args = new String[2];
            args[0] = id;
            args[1] = e.getMessage();
            throw new SDXException(logger, SDXExceptionCode.ERROR_NEW_INSTANCE, args, null);
        } catch (InstantiationException e) {
            String[] args = new String[2];
            args[0] = id;
            args[1] = e.getMessage();
            throw new SDXException(logger, SDXExceptionCode.ERROR_NEW_INSTANCE, args, null);
        } catch (IllegalAccessException e) {
            String[] args = new String[2];
            args[0] = id;
            args[1] = e.getMessage();
            throw new SDXException(logger, SDXExceptionCode.ERROR_NEW_INSTANCE, args, null);
        }
    }

    /**Returns the paramters for this pipeline*/
    public Parameters getParameters() {
        return this.params;
    }

    /**Returns a byte array which contains the
     * data of a transformation step. The data will only be present
     * after a pipeline is executed, if no transformation data is retained
     * <code>null</code> will be returned
     *
     * @return
     */
    public byte[] getTransformedBytes() {
        if (this.transformedBytes != null && this.transformedBytes.size() > 0) {
            //TODO:this should be tested with large buffer, ie large xml documents
            return this.transformedBytes.toByteArray();
        } else
            return null;
    }

    /**Returns a file which contains the
     * data of a transformation step. The data will only be present
     * after a pipeline is executed, if no transformation data is retained
     * <code>null</code>  will be returned
     *
     */
    public File getTransformedFile() throws SDXException {
        if (this.fOs != null && this.transformedFile != null && transformedFile.exists() && transformedFile.length() > 0) {
            try {
                this.fOs.flush();
            } catch (IOException e) {
                throw new SDXException(logger, SDXExceptionCode.ERROR_GET_TRANSFORMED_FILE, null, e);
            }
            return this.transformedFile;
        } else
            return null;
    }


    /**Sets Properties.
     *
     * @param props A Properties object containing path information for an instance of an application.
     */
    public void setProperties(Hashtable props) {
        this.props = props;
    }

    private void createTempFileResources() throws SDXException {
        try {

            if (canBuildTempFile() && this.xsFile != null) {
                if (this.fOs != null)
                    this.fOs.close();

                //TODO: test this, if we delete the file, does the mulitcaster keep the reference
                if (this.transformedFile != null && transformedFile.exists()) {
                    if (!this.transformedFile.delete())//if we can't delete the file successfully we will try onExit
                        this.transformedFile.deleteOnExit();
                }
                File tempDir = (File) props.get(Constants.CONTEXT_UPLOAD_DIR);
                if (tempDir.canWrite())
                    tempDir = new File(tempDir, File.separator + "tempTransformedDocs" + File.separator + getId() + File.separator);
                else
                    tempDir = Utilities.getSystemTempDir();

                tempDir.mkdirs();
                this.transformedFile = File.createTempFile(TEMPFILE_PREFIX, TEMPFILE_SUFFIX, tempDir);

                if (this.transformedFile != null && transformedFile.exists())
                    this.fOs = new FileOutputStream(this.transformedFile);
                if (this.fOs != null) {
                    this.xsFile.recycle();
                    this.xsFile.setOutputStream(this.fOs);
                }
            }
        } catch (IOException e) {
            throw new SDXException(logger, SDXExceptionCode.ERROR_BUILD_TRANSFORMED_FILE, null, e);
        }
    }

    private void createTempByteResources() {
        //building the output streams which will be populated during the transformation
        if (this.transformedBytes == null)
            this.transformedBytes = new ByteArrayOutputStream();

        //buliding the serializers which will supply the output streams
        if (this.transformedBytes != null) {
            this.transformedBytes.reset();

            if (this.xsBytes != null) {
                this.xsBytes.recycle();
                this.xsBytes.setOutputStream(this.transformedBytes);
            }
        }
    }

    private boolean canBuildTempFile() {
        try {
            File.createTempFile(TEMPFILE_PREFIX, TEMPFILE_SUFFIX);
            return true;
        } catch (IOException e) {
            return false;
        }
    }

}
