#######################################################################
#  This file is part of GNOWSYS: Gnowledge Networking and
#  Organizing System.
#
#  GNOWSYS is free software; you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as
#  published by the Free Software Foundation; either version 3 of
#  the License, or (at your option) any later version.
#
#  GNOWSYS is distributed in the hope that it will be useful, but
#  WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public
#  License along with GNOWSYS (gpl.txt); if not, write to the
#  Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
#  Boston, MA  02110-1301  USA59 Temple Place, Suite 330,
#
######################################################################
# Contributor: "Dinesh Joshi" <dinesh.joshi@yahoo.com>


import sys
from gnowsysTable import *

class gnowsysDatatypes:
	"""
	Class to manage the datatypes tables
	"""
	def __init__( self, tblName, isPrefixes=0 ):
		"""
		Initializes the following:
		1. tablename
		2. prefixes
		3. list of datatypes
		"""
		# Should we add prefixes to the vid table?
		# 1 = yes
		# other = no
		#
		self.tblName = tblName
		self.isPrefixes = isPrefixes
		self.tablePrefix = "datatypes"
		self.lstTables = []
		
		# format is [ 'datatype name', 'description' ]
		self.datatypes = [ 
			[ 'int8',              '' ],
			[ 'bit',               '' ],
			[ 'varbit',            '' ],
			[ 'boolean',           '' ],
			[ 'box',               '' ],
			[ 'bytea',             '' ],
			[ 'varchar',           '' ],
			[ 'char',              '' ],
			[ 'cidr',              '' ],
			[ 'circle',            '' ],
			[ 'date',              '' ],
			[ 'float8',            '' ],
			[ 'inet',              '' ],
			[ 'int4',              '' ],
			[ 'interval',          '' ],
			[ 'line',              '' ],
			[ 'lseg',              '' ],
			[ 'macaddr',           '' ],
			[ 'money',             '' ],
			[ 'numeric',           '' ],
			[ 'path',              '' ],
			[ 'point',             '' ],
			[ 'polygon',           '' ],
			[ 'float4',            '' ],
			[ 'int2',              '' ],
			[ 'text',              '' ],
			[ 'time',              '' ],
			[ 'timestamptz',       '' ],
			[ 'abstime',           '' ],
			[ 'aclitem',           '' ],
			[ 'bpchar',            '' ],
			[ 'cid',               '' ],
			[ 'oid',               '' ],
			[ 'refcursor',         '' ],
			[ 'regclass',          '' ],
			[ 'regoper',           '' ],
			[ 'regoperator',       '' ],
			[ 'regproc',           '' ],
			[ 'regprocedure',      '' ],
			[ 'regtype',           '' ],
			[ 'reltime',           '' ],
			[ 'smgr',              '' ],
			[ 'tid',               '' ],
			[ 'timetz',            '' ],
			[ 'tinterval',         '' ],
			[ 'unknown',           '' ],
			[ 'xid',               '' ],
			[ 'int2vector',        '' ],
			[ 'name',              '' ],
			[ 'oidvector',         '' ],
			[ 'serial',            '' ],
			[ 'serial8',         '' ],
			]

	def getDataTypeTableName( self, datatype ):
		"""
		Given a datatype, returns its table name in GNOWSYS database
		"""
		if datatype[ len( datatype ) - 2: ] != '[]':
			dt = datatype
			tmpTblName = self.tablePrefix + "_" + dt
			return tmpTblName
		else:
			# array datatype
			# except smgr and unknown from making array datatypes
			dt = datatype[ :len( datatype ) - 2 ]

			if dt == 'smgr' or dt == 'unknown':
				return -1

			tmpTblName = self.tablePrefix + "_" + dt + "_array"
			return tmpTblName
		

	def createTables( self ):		
		"""
		defines the tables in terms of GNOWSYS' storageSpec
		"""
		exceptionList = [ 'line',
				  'lseg',
				  'circle',
				  'point',
				  'box',
				  'path',
				  'polygon',
				  'aclitem',
				  'cid',
				  'refcursor',
				  'smgr',
				  'tid',
				  'unknown',
				  'xid',
				  ]
		
		
		# generate the regular tables
		for dtl in self.datatypes:
			# simple datatype
			dt = dtl[0]
			tmpTblName = self.tablePrefix + "_" + dt
			
			# check the prefixes preference
			if self.isPrefixes == 1:
				tmpDTFldName = dt + "_" + 'vid'
			else:
				tmpDTFldName = 'vid'
			
			tmpDTFldType = 'serial8'
			
			tmpValueFldName = 'value'
			tmpValueFldType = dt

			# disabling the constraints due to problem described here:
			# http://gnowledge.org/pipermail/gnowsys-dev/2008-March/001175.html
			# 			constraint = 'NOT NULL UNIQUE'
			# 			if dt in exceptionList:
			# 				constraint = ""
			constraint = ""

			tmpTbl = gnowsysTable( tmpTblName )
			tmpTbl.addFields(
				[
					[ tmpDTFldName   ,    tmpDTFldType    , 'PRIMARY KEY' ],
					[ tmpValueFldName,    tmpValueFldType , constraint ]
					]
				)
			
			self.lstTables.append( tmpTbl )

			# array datatype
			
			# except smgr and unknown from making array datatypes
			if dt == 'smgr' or dt == 'unknown':
				continue
			
			tmpTblName = self.tablePrefix + "_" + dt + "_array"
			
			# check the prefixes preference
			if self.isPrefixes == 1:
				tmpDTFldName = dt + "_array_" + 'vid'
			else:
				tmpDTFldName = 'vid'
				
			tmpDTFldType = 'serial8'
			
			tmpValueFldName = 'value'
			tmpValueFldType = dt + "[]"

			tmpTbl = gnowsysTable( tmpTblName )
			tmpTbl.addFields( 
				[ 
					[ tmpDTFldName   ,    tmpDTFldType    , 'PRIMARY KEY' ],
					[ tmpValueFldName,    tmpValueFldType , constraint ]
					]
				)
			
			self.lstTables.append( tmpTbl )
	
	def populateMainTable( self ):
		"""
		populates the gbdatatypes table with the available datatype names
		"""
		strSQL = ""
		for dt in self.datatypes:
			query1 = "INSERT INTO %s ( datatypeid, datatypename, description ) VALUES( DEFAULT, '%s', '%s' );" % ( self.tblName, dt[0], dt[1] )
			query2 = "INSERT INTO %s ( datatypeid, datatypename, description ) VALUES( DEFAULT, '%s', '%s' );" % ( self.tblName, dt[0]+ "[]", "" )
			if dt[0] == 'smgr' or dt[0] == 'unknown':
				query2 = ""

			strSQL = strSQL + "\n" + query1 + "\n" + query2
		
		return strSQL

	
	def genSQL( self ):
		"""
		generates the SQL using the table definitions generated by createTables function
		"""
		strSQL = ""
		for t in self.lstTables:
			strSQL = strSQL + t.genSQL()
		
		strSQL = strSQL + self.populateMainTable()
		return strSQL
	
	def genHTML( self ):
		"""
		Document the tables in HTML
		"""
		strHTML = ""
		
		for ft in self.lstTables:
			strHTML = strHTML + ft.genHTML()
		
		return( strHTML )



if __name__ == '__main__':
	# Some test code
	# Generates both HTML as well as SQL code
	d = gnowsysDatatypes(1)
	d.createTables()
	
	print d.genHTML()
	print d.populateMainTable()
	print d.genSQL()
	
