/************************************************************************
 * SGA - A C++ library to help develop Simple Genetic Algorithms        *
 * Copyright (C) 2005 Dorival M. Pedroso                                *
 *                                                                      *
 * This program is free software: you can redistribute it and/or modify *
 * it under the terms of the GNU General Public License as published by *
 * the Free Software Foundation, either version 3 of the License, or    *
 * any later version.                                                   *
 *                                                                      *
 * This program is distributed in the hope that it will be useful,      *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of       *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the         *
 * GNU General Public License for more details.                         *
 *                                                                      *
 * You should have received a copy of the GNU General Public License    *
 * along with this program. If not, see <http://www.gnu.org/licenses/>  *
 ************************************************************************/

#ifndef SGA_ISLAND_H
#define SGA_ISLAND_H

// STL
#include <algorithm> // for sort, min_element and max_element
#include <sstream>   // for ostringstream
#include <vector>    // for ostringstream

// SGA
#include "random.h"
#include "numstreams.h"


/* NOTE: The following functions must be already defined somewhere else:
 *   
 *   template<typename Gene_T> inline void   MutationFun  (Gene_T & OldGene);
 *   template<typename Gene_T> inline double ObjectiveFun (Gene_T const Genes[], uint32_t nGenes);
 *
 * For example:
 *
 *   inline void   MutationFun  (uint32_t & OldGene)
 *   inline double ObjectiveFun (uint32_t const Genes[], uint32_t nGenes)
 */


/** \namespace SGA Simple Genetic Algorithms. */
namespace SGA
{

// {{{ Doxygen - Population of individuals.
/** %Population of individuals.
 * Gene_T is the type of genes, for example: uint32_t, double, etc.
 * nIndiv is the number of individuals.
 * nGenes is the number of genes.
 *
 * Example:
   \verbatim
   The population may be viewed as a matrix where the rows correspond to the individuals
   and the columns to the genes:
             Gene #     0  1  2  3  4
                      ----------------
     Individual # 0  |  0  1  2  3  4
     Individual # 1  |  5  6  7  8  9
     Individual # 2  | 10 11 12 13 14 
   \endverbatim

 * Another example (Circles):
   \verbatim
   Each individual corresponds to a set with all circles.
   The DNA of an individual contains the circle information,
   organized as follows (row-major):
   
    NBASES = 1:
      x0 y0 r0   x1 y1 r1   x2 y2 r2  ==
     _          _
    |  x0 y0 r0  |
    |  x1 y1 r1  |
    |    ....
    |_ xN yN rN _|   N => N circles
   
    NBASES = 2:
      x00 x01 y00 y01 r00 r01   x10 x11 y10 y11 r10 r11   x20 x21 y20 y21 r20 r21  ==
     _                           _
    |  x00 x01  y00 y01  r00 r01  |
    |  x10 y11  y10 y11  r10 r11  |
    |            ....
    |_ xN0 xN1  yN0 yN1  rN0 rN1 _|   N => N circles
   
    NBASES = 4
   
    DNA of ONE individual:
     0 : x00 x01 x02 x03   y00 y01 y02 y03   r00 r01 r02 r03 : circle# 0
    12 : x10 x11 x12 x13   y10 y11 y12 y13   r10 r11 r12 r13 : circle# 1
    24 : x20 x21 x22 x23   y20 y21 y22 y23   r20 r21 r22 r23 : circle# 2
     ^     0   1   2   3     4   5   6   7     8   9  10  11
     |
    iCircle*3*NBASES, where 3 is due to x, y and r values
   
    circle# 0:             circle# 1:             circle# 2:
     x0 = x00+x01+x02+x03   x1 = x10+x11+x12+x13   x2 = x20+x21+x22+x23
     y0 = y00+y01+y02+y03   y1 = y10+y11+y12+y13   y2 = y20+y21+y22+y23
     r0 = r00+r01+r02+r03   r1 = r10+r11+r12+r13   r2 = r20+r21+r22+r23
   \endverbatim */ // }}}
template<typename Gene_T>
class Population
{
public:
	/** Default constructor (must call Allocate later). */
	 Population () : _nindiv(0), _ngenes(0), _genes(NULL) {}

	 /** Destructor. */
	~Population () { if (_genes!=NULL) delete [] _genes; }

	/** Allocate space for genes */
	void Allocate (uint32_t nIndiv, uint32_t nGenes)
	{ // {{{

		_nindiv = nIndiv;
		_ngenes = nGenes;
		_genes  = new Gene_T [_nindiv*_ngenes];

		/* Internally, these values are considered to be in a row-major format,
		 * thus a gene can be accessed using:
		 *
		 *       _genes [i*nGenes+j]
		 *
		 *       0 1 2 3 4   5 6 7 8 9   10 11 12 13 14
		 *
		 * where i is the index of an individual, and j is the index of a gene.
		 */

	} // }}}

	// Access methods
	Gene_T const * operator() (int i)        const { return &_genes[i*_ngenes]; }  ///< Access and individual (read). \param i is the index of the individual
	Gene_T       * operator() (int i)              { return &_genes[i*_ngenes]; }  ///< Access and individual (write). \param i is the index of the individual
	Gene_T const & operator() (int i, int j) const { return _genes[i*_ngenes+j]; } ///< Access a Gene (read). \param i is the index of an individual \param j is the index of a gene
	Gene_T       & operator() (int i, int j)       { return _genes[i*_ngenes+j]; } ///< Access a Gene (write). \param i is the index of an individual \param j is the index of a gene

private:
	uint32_t   _nindiv; ///< Number of individuals
	uint32_t   _ngenes; ///< Number of genes
	Gene_T   * _genes;  ///< All individuals (column-major ordering)

}; // class Population

/** Objective value.
 * Structure used to sort (with STL algorithm) individuals during reproduction
 */
struct ObjectiveValue
{
	uint32_t Index; ///< Index of an individual
	double   Value; ///< The objective value of the Index individual
};
bool operator< (ObjectiveValue const & a, ObjectiveValue const & b) { return a.Value < b.Value; } ///< Compares two ObjectiveValues

/** %Island (main) class for a Simple Genetic Algorithm (%SGA).
 * Only one instance per process is required.
 */
template<typename Gene_T>
class Island
{
public:
	/** Constructor.
	 NOTE: The following functions must be already defined somewhere else:
	 \verbatim
	    template<typename Gene_T> inline void   MutationFun  (Gene_T & OldGene);
	    template<typename Gene_T> inline double ObjectiveFun (Gene_T const Genes[], uint32_t nGenes);
	 \endverbatim
	 For example:
	 \verbatim
	    inline void   MutationFun  (uint32_t & OldGene)
	    inline double ObjectiveFun (uint32_t const Genes[], uint32_t nGenes)
	 \endverbatim */
	Island (uint32_t  nIndiv               , ///< Number of individuals
	        uint32_t  nGenes               , ///< Number of genes
	        bool      IsMinimization = true, ///< Is a minimization problem?
	        uint32_t  EliteSize      = 1   , ///< Number of individuals in the elite
	        uint32_t  MaxNCross      = 1   , ///< Maximum number of sections during crossover (must be smaller than nGenes)
	        double    ProbCrossover  = 0.5 , ///< Probability of crossover
	        double    ProbMutation   = 0.01, ///< Probability of mutation
	        double    SelPressure    = 2.0 , ///< Selective pressure for the Ranking selection (inside: Linear:[1,2], Non-linear:[1,nIndiv-2])
	        NumStream NStream        = _6_3) ///< Used to format the output of Genes
		: _nindiv          (nIndiv        ), // {{{
		  _ngenes          (nGenes        ),
		  _is_minimization (IsMinimization),
		  _elite_size      (EliteSize     ),
		  _max_ncross      (MaxNCross     ),
		  _prob_cross      (ProbCrossover ),
		  _prob_mut        (ProbMutation  ),
		  _sel_pressure    (SelPressure   ),
		  _ns              (NStream       )
	{
		// Check number of individuals
		if (_nindiv<4)
			throw "SGA::Island::Constructor: The number of individuals must be greater than or equal to four (4)";

		// Number of selected individuals for reproduction
		// Must be always even and smaller than or equal to the half of the population size
		_num_selected = _nindiv/2;
		if (_num_selected%2>0)
			_num_selected--;

		// Check elite size
		if (_elite_size>=_num_selected)
			throw "SGA::Island::Constructor: The size of the elite must be smaller than nIndiv/2 (or nIndiv/2-1)";

		// Check MaxNCross
		if (_max_ncross<1 || _max_ncross>=nGenes)
			throw "SGA::Island::Constructor: The maximum number of sections during crossover (MaxNCross) must be greater than 0 and smaller than nGenes";

		// Allocate population
		_pop.Allocate(_nindiv, _ngenes);

		// Allocate arrays
		_sel_prob     = new double         [nIndiv];
		_cum_prob     = new double         [nIndiv];
		_selected     = new uint32_t       [nIndiv/2];
		_non_selected = new uint32_t       [nIndiv];
		_aux_selected = new bool           [nIndiv];
		_ovs          = new ObjectiveValue [nIndiv];
		_cross_pos    = new uint32_t       [nGenes+1];

		// Calculate cumulated probability (independent of objective function)j
		_calc_prob_cumsum();

	} // }}}

	/** Destructor. */
	~Island()
	{ // {{{
		delete [] _sel_prob;
		delete [] _cum_prob;
		delete [] _selected;
		delete [] _non_selected;
		delete [] _aux_selected;
		delete [] _ovs;
		delete [] _cross_pos;
	} // }}}

	/** Set whether this problem is minimization of maximization. */
	inline Island & SetMinimization (bool IsMinimization=true)
	{ // {{{
		_is_minimization = IsMinimization;
		return (*this);
	} // }}}

	/** Set the number of individuals in the elite. */
	inline Island & SetEliteSize (uint32_t EliteSize=0)
	{ // {{{
		_elite_size = EliteSize;
		if (_elite_size>=_num_selected)
			throw "SGA::Island::Constructor: The size of the elite must be smaller than nIndiv/2 (or nIndiv/2-1)";
		return (*this);
	} // }}}

	/** Set the probability of crossover. */
	inline Island & SetProbCross (double ProbCrossover=0.5)
	{ // {{{
		_prob_cross = ProbCrossover;
		return (*this);
	} // }}}

	/** Set the probability of mutation. */
	inline Island & SetProbMut (double ProbMutation=0.01)
	{ // {{{
		_prob_mut = ProbMutation;
		return (*this);
	} // }}}

	/** Set structure to format number output. */
	inline Island & SetNumStream (NumStream NStream=_6_3)
	{ // {{{
		_ns = NStream;
		return (*this);
	} // }}}

	/** Fill all individuals with random Genes.
	 * The Gene values will be inside [Min,Max] for integers and [Min,Max) for reals.
	 * Size of Min and Max arrays is equal to nGenes
	 */
	inline void RandomFill (Gene_T const Min[], Gene_T const Max[])
	{ // {{{
		for (uint32_t i=0; i<_nindiv; ++i)
		for (uint32_t j=0; j<_ngenes; ++j)
			_pop(i,j) = Rnd::T<Gene_T>::Gen (Min[j], Max[j]);
	} // }}}

	/** Get the number of individuals. */
	inline uint32_t nIndiv() const { return _nindiv; }

	/** Get the number of genes. */
	inline uint32_t nGenes() const { return _ngenes; }

	/** Crossing over between Dad and Mom from the same %Island.
	 * \param iDad index for the individual (Dad)
	 * \param iMom index for the individual (Mom)
	 * \param iSon index for the individual (Son)
	 * \param iDau index for the individual (Daughter)
	 */
	inline void CrossingOver (uint32_t iDad, uint32_t iMom, uint32_t iSon, uint32_t iDau)
	{ // {{{
		if (Rnd::Flip(_prob_cross))
		{
			uint32_t ncross = Rnd::T<uint32_t>::Gen(1,_max_ncross); // Number of sections
			for (uint32_t i=0; i<_ngenes; ++i) _cross_pos[i]=i; //  {0,1,2,3,4,5} => nGenes == 6
			Rnd::Shuffle (_ngenes-1, _cross_pos+1);             //  {0,5,2,3,1,4}
			std::sort (_cross_pos+1, _cross_pos+1+ncross);      //  {0,2,3,5}     => ncross == 3
			_cross_pos[ncross+1] = _ngenes;                     //  {0,2,3,5,6}
			bool reverse = false;
			for (uint32_t k=0; k<=ncross; ++k)
			{
				if (reverse)
				{
					for (uint32_t j=_cross_pos[k]; j<_cross_pos[k+1]; ++j)
					{
						_pop(iSon,j) = _pop(iMom,j);
						_pop(iDau,j) = _pop(iDad,j);
					}
					reverse = false;
				}
				else
				{
					for (uint32_t j=_cross_pos[k]; j<_cross_pos[k+1]; ++j)
					{
						_pop(iSon,j) = _pop(iDad,j);
						_pop(iDau,j) = _pop(iMom,j);
					}
					reverse = true;
				}
			}
		}
		else // just copy
		{
			for (uint32_t j=0; j<_ngenes; ++j)
			{
				_pop(iSon,j) = _pop(iDad,j);
				_pop(iDau,j) = _pop(iMom,j);
			}
		}
	} // }}}

	/** Mutation.
	 * \param iInd index for the individual
	 */
	inline void Mutation (uint32_t iInd)
	{ // {{{
		if (Rnd::Flip(_prob_mut))
		{
			uint32_t j = Rnd::T<uint32_t>::Gen(0,_ngenes-1);
			MutationFun (_pop(iInd,j));
		}
	} // }}}

	/** Reproduction. */
	inline void Reproduction ()
	{ // {{{

		// Calculate objective values
		for (uint32_t i=0; i<_nindiv; ++i)
		{
			_ovs[i].Index = i;
			_ovs[i].Value = ObjectiveFun (_pop(i), _ngenes);
		}

		// Sort ascending individuals (only indexes) according to objective values
		std::sort (_ovs, _ovs+_nindiv);

		// Select individuals for reproduction
		_select_some ();

		// Crossover and mutation
		uint32_t dad = 0;
		uint32_t mom = 1;
		for (uint32_t i=0; i<_num_selected/2; ++i)
		{
			CrossingOver (_ovs[_selected[dad]].Index, _ovs[_selected[mom]].Index, _ovs[_non_selected[dad]].Index, _ovs[_non_selected[mom]].Index);
			Mutation     (_ovs[_non_selected[dad]].Index);
			Mutation     (_ovs[_non_selected[mom]].Index);
			dad += 2;
			mom += 2;
		}

		// Elitism
		for (uint32_t i=0; i<_elite_size; ++i)
		{
			if (_is_minimization)
			{
				for (uint32_t j=0; j<_ngenes; ++j)
					_pop(i,j) = _pop(_ovs[i].Index, j);
			}
			else
			{
				for (uint32_t j=0; j<_ngenes; ++j)
					_pop(i,j) = _pop(_ovs[_nindiv-1-i].Index, j);
			}
		}

	} // }}}

	/** Access the Objective Values array (read). */
	ObjectiveValue const * const ObjVals() const { return _ovs; }

	/** Output one individual. */
	inline void OutIndividual (uint32_t iIndiv, std::ostringstream & oss)
	{ // {{{
		for (uint32_t j=0; j<_ngenes; ++j)
			oss << _ns << _pop(iIndiv,j);
	} // }}}

	/** Output one Island. */
	inline void OutIsland (std::ostringstream & oss)
	{ // {{{
		for (uint32_t i=0; i<_nindiv; ++i)
		{
			oss << "  Ind #" << i << " ";
			OutIndividual (i,oss);
			oss << std::endl;
		}
	} // }}}

	/** Access the population (read). */
	Population<Gene_T> const & Pop() const { return _pop; }

	/** Access the population (write). */
	Population<Gene_T> & Pop() { return _pop; }

	/** Return the best individual.
	 * \param BestOV Place to return the best objective value
	 * \param ReCalculateOVs Recalculate Objective Values?
	 * If ReCalculateOVs==false, use previous values calculated during Reproduction.
	 */
	Gene_T const * Best (double & BestOV, bool ReCalculateOVs=false) const
	{ // {{{

		// Calculate objective values
		if (ReCalculateOVs)
		{
			for (uint32_t i=0; i<_nindiv; ++i)
			{
				_ovs[i].Index = i;
				_ovs[i].Value = ObjectiveFun (_pop(i), _ngenes);
			}
		}

		// Find best individual
		if (_is_minimization)
		{
			ObjectiveValue const * pbest = std::min_element(_ovs, _ovs+_nindiv);
			BestOV = pbest->Value;
			return _pop(pbest->Index);
		}
		else
		{
			ObjectiveValue const * pbest = std::max_element(_ovs, _ovs+_nindiv);
			BestOV = pbest->Value;
			return _pop(pbest->Index);
		}

	} // }}}

	/** Return the worst individual.
	 * \param WorstOV Place to return the worst objective value
	 * \param ReCalculateOVs Recalculate Objective Values?
	 * If ReCalculateOVs==false, use previous values calculated during Reproduction.
	 */
	Gene_T const * Worst (double & WorstOV, bool ReCalculateOVs=false) const
	{ // {{{

		// Calculate objective values
		if (ReCalculateOVs)
		{
			for (uint32_t i=0; i<_nindiv; ++i)
			{
				_ovs[i].Index = i;
				_ovs[i].Value = ObjectiveFun (_pop(i), _ngenes);
			}
		}

		// Find worst individual
		if (_is_minimization)
		{
			ObjectiveValue const * pworst = std::max_element(_ovs, _ovs+_nindiv);
			WorstOV = pworst->Value;
			return _pop(pworst->Index);
		}
		else
		{
			ObjectiveValue const * pworst = std::min_element(_ovs, _ovs+_nindiv);
			WorstOV = pworst->Value;
			return _pop(pworst->Index);
		}

	} // }}}

private:
	uint32_t                 _nindiv;          ///< Number of individuals
	uint32_t                 _ngenes;          ///< Number of genes
	bool                     _is_minimization; ///< Is a minimization problem?
	uint32_t                 _elite_size;      ///< Number of individuals in the elite
	uint32_t                 _max_ncross;      ///< Maximum number of sections during crossover
	double                   _prob_cross;      ///< Probability of crossover
	double                   _prob_mut;        ///< Probability of mutation
	double                   _sel_pressure;    ///< Selective pressure for the Ranking selection (inside: Linear:[1,2], Non-linear:[1,nIndiv-2])
	NumStream                _ns;              ///< Used to format the output of Genes
	Population<Gene_T>       _pop;             ///< Population data
	uint32_t                 _num_selected;    ///< Number of selected individuals for reproduction == (nIndiv-_elite_size)/2 (must be always even)
	double                 * _sel_prob;        ///< Selection probabilities (size==nIndiv)
	double                 * _cum_prob;        ///< Cumulated probabilities (size==nIndiv)
	uint32_t               * _selected;        ///< Selected individuals for reproduction (size==nIndiv/2)
	uint32_t               * _non_selected;    ///< Non-selected individuals for reproduction (size==nIndiv)
	bool                   * _aux_selected;    ///< Auxiliar array to find non-selected individuals (size==nIndiv)
	mutable ObjectiveValue * _ovs;             ///< Objective values (size==nIndiv)
	uint32_t               * _cross_pos;       ///< Positions for crossover (size==nGenes+1)

	/** Calculate probability cumulated sum (Stochastic Universal method with linear ranking).
	 * Fill an array of cumulated probability, considering the fitness calculated by the Ranking method.
	 * This function depends ONLY on the number of individuals.
	 * This function must be called ONCE in an optimization.
	 */
	inline void _calc_prob_cumsum(bool LinearRanking=true)
	{ // {{{

		// Calculate the fitness - Ranking (totally independent from Objective Values)
		// Selective pressure must be inside:
		//     Linear     : [1,2]
		//     Non-linear : [1,nIndiv-2]
		// For 11 individuals (example with Linear ranking):
		//    fitness =  0.18 0.16 0.15 0.13 0.11 0.09 0.07 0.06 0.03 0.02 0.00
		double * fitness = new double [_nindiv];
		if (_is_minimization)
		{
			if (LinearRanking)
			{
				for (uint32_t i=0; i<_nindiv; ++i)
					fitness[i] = 2.0-_sel_pressure+2.0*(_sel_pressure-1.0)*(_nindiv-1-i)/(_nindiv-1.0);
			}
			else throw "SGA::Island:_calc_prob_cumsum: Non-linear ranking is not yet implemented";
		}
		else
		{
			if (LinearRanking)
			{
				for (uint32_t i=0; i<_nindiv; ++i)
					fitness[i] = 2.0-_sel_pressure+2.0*(_sel_pressure-1.0)*i/(_nindiv-1.0);
			}
			else throw "SGA::Island:_calc_prob_cumsum: Non-linear ranking is not yet implemented";
		}

		// Calculate the sum of fitness
		// For 11 individuals:
		//   sum_fit = 11
		double sum_fit = 0.0;
		for (uint32_t i=0; i<_nindiv; ++i)
			sum_fit += fitness[i];

		// Calculate the cumulated probability
		// For 11 individuals:
		//   _cum_prob =  0.18 0.34 0.49 0.62 0.73 0.82 0.89 0.95 0.98 1.00 1.00
		_cum_prob[0] = fitness[0]/sum_fit;
		for (uint32_t i=1; i<_nindiv; ++i)
			_cum_prob[i] = _cum_prob[i-1] + fitness[i]/sum_fit;

		// Clean up
		delete [] fitness;

	} // }}}

	/** Selection (ranking).
	 * Stochastic Universal sampling (SUS).
	 */
	inline void _select_some()
	{ // {{{

		// Selection probability
		_sel_prob[0] = Rnd::T<double>::Gen(0.0, 1.0/_num_selected);
		for (uint32_t i=1; i<_num_selected; ++i)
			_sel_prob[i] = _sel_prob[i-1] + 1.0/_num_selected;

		// Find indexes of selected and non-selected
		for (uint32_t i=0; i<_num_selected; ++i)
		{
			_aux_selected[i] = false;
			_selected    [i] = 0;
		}
		for (uint32_t i=_num_selected; i<_nindiv; ++i)
			_aux_selected[i] = false;
		for (uint32_t i=0; i<_num_selected; ++i)
		{
			while (_sel_prob[i]>_cum_prob[_selected[i]])
				_selected[i]++;
			_aux_selected[_selected[i]] = true;
		}
		uint32_t j = 0;
		for (uint32_t i=0; i<_nindiv; ++i)
		{
			if (!_aux_selected[i])
			{
				_non_selected[j] = i;
				j++;
			}
		}

		// Shuffle indexes
		Rnd::Shuffle (_num_selected, _selected);

	} // }}}

}; // class Island

}; // namespace SGA

#endif // SGA_ISLAND_H

// vim:fdm=marker
