/************************************************************************
 * SGA - A C++ library to help develop Simple Genetic Algorithms        *
 * Copyright (C) 2005 Dorival M. Pedroso                                *
 *                                                                      *
 * This program is free software: you can redistribute it and/or modify *
 * it under the terms of the GNU General Public License as published by *
 * the Free Software Foundation, either version 3 of the License, or    *
 * any later version.                                                   *
 *                                                                      *
 * This program is distributed in the hope that it will be useful,      *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of       *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the         *
 * GNU General Public License for more details.                         *
 *                                                                      *
 * You should have received a copy of the GNU General Public License    *
 * along with this program. If not, see <http://www.gnu.org/licenses/>  *
 ************************************************************************/

// FLTK
#include <FL/Fl.H>
#include <FL/Fl_Window.H>
#include <FL/Fl_Button.H>

// Callback
void but_cb (Fl_Widget * o, void*)
{
	/* Remember 'but' is an Fl_Button which is derived from Fl_Widget.
	 * So 'but' can use its base classes callback member function.
	 *     void Fl_Widget::callback(Fl_Callback*, void* = 0)
	 * The second parameter (void*) is for userdata
	 * (any data structure you wish to pass that the callback might need)
	 */

	/* Note: label() and value() are the only two widget members that automatically call redraw.
	 * Everything else requires a manual call of redraw(). As you will see in the next two lines.
	 * BTW widgets don't automatically store a copy of their label.
	 * They only store a pointer to a char array.
	 */

	/* By default FLTK does not store a copy of any widget label in the widget object. It ONLY stores the const pointer to the
	 * label. So if the label is a string literal like "Good job" that is not a problem. But if it is a const pointer, the
	 * programmer must ensure that the pointer exists as long as the widget does. Because if the pointer goes out of scope
	 * before the widget does then when the widget gets redrawn the label will be undefined
	 * If we change the line in the callback function from
	 *     o->label(newcap);
	 * to
	 *     o->copy_label(newcap);
	 * The problem disappears since the widget now stores an actual copy of the label.
	 */

	Fl_Button * b = static_cast<Fl_Button*>(o);
	b->label  ("Good job");    //redraw not necessary
	b->resize (10,150,140,30); //redraw needed
	b->redraw ();
}

// Main
int main()
{
	Fl_Window win(/*width*/300, /*height*/200, /*title*/"Testing");
	win.begin(); // -- start adding widgets
		Fl_Button but(/*x*/10, /*y*/150, /*width*/70, /*height*/30, /*label*/"Click me");
	win.end(); // -- end adding widgets
	but.callback (but_cb);
	win.show     ();
	return Fl::run();

	/* You only need to destroy the Fl_Window (the parent). Doing so will automatically
	 * delete all the children of the window. So when 'win' goes out of scope then 'but'
	 * will be deleted automatically
	 */

	/* The function run() returns 0 and ends the program when all windows are closed or hidden.
	 * Another quick and easy way to end the program is to call exit(0).
	 */
}
