/************************************************************************
 * SGA - A C++ library to help develop Simple Genetic Algorithms        *
 * Copyright (C) 2005 Dorival M. Pedroso                                *
 *                                                                      *
 * This program is free software: you can redistribute it and/or modify *
 * it under the terms of the GNU General Public License as published by *
 * the Free Software Foundation, either version 3 of the License, or    *
 * any later version.                                                   *
 *                                                                      *
 * This program is distributed in the hope that it will be useful,      *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of       *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the         *
 * GNU General Public License for more details.                         *
 *                                                                      *
 * You should have received a copy of the GNU General Public License    *
 * along with this program. If not, see <http://www.gnu.org/licenses/>  *
 ************************************************************************/

// STL
#include <iostream>
#include <sstream>
#include <cmath>
#include <ctime>

// FLTK
#include <FL/Fl.H>
#include <FL/Fl_Button.H>
#include <FL/Fl_Input.H>
#include <FL/fl_draw.H>
#include <FL/Fl_Double_Window.H>

// SGA
#include "triangleio.h"
#include "random.h"

// Constants
const double XMIN = 0.0;
const double XMAX = 1.0;
const double YMIN = 0.0;
const double YMAX = 1.0;

using std::cout;
using std::endl;

// MainWindow
class MainWindow : public Fl_Double_Window
{
public:
	// DrawArea structure
	class DrawArea : public Fl_Widget
	{ // {{{
	public:
		// Constructor
		DrawArea (int xmin, int ymin, int width, int height) // Screen coordinates
			: Fl_Widget (xmin,ymin,width,height,0),
			  _t        (NULL)
		{}

		// Set TriangleIO
		void SetTriangleIO (TriangleIO const * T) { _t = T; }

		// Internal methods
		void draw()
		{
			if (_t!=NULL)
			{
				// Bounding box
				_Xmin = _t->R().pointlist[0];
				_Ymin = _t->R().pointlist[1];
				_Xmax = _t->R().pointlist[0];
				_Ymax = _t->R().pointlist[1];
				for (int i=1; i<_t->R().numberofpoints; ++i)
				{
					if (_t->R().pointlist[i*2  ]<_Xmin) _Xmin = _t->R().pointlist[i*2  ];
					if (_t->R().pointlist[i*2+1]<_Ymin) _Ymin = _t->R().pointlist[i*2+1];
					if (_t->R().pointlist[i*2  ]>_Xmax) _Xmax = _t->R().pointlist[i*2  ];
					if (_t->R().pointlist[i*2+1]>_Ymax) _Ymax = _t->R().pointlist[i*2+1];
				}

				// Scale factors
				_sfx = static_cast<double>((w()-2)/(_Xmax-_Xmin));
				_sfy = static_cast<double>((h()-2)/(_Ymax-_Ymin));

				// Clear the background
				fl_color (FL_BACKGROUND_COLOR);
				fl_rectf (x(),y(),w(),h());

				// Draw points
				int r = _l(0.02*(_Xmax-_Xmin));
				for (int i=0; i<_t->R().numberofpoints; ++i)
				{
					int x = _x(_t->R().pointlist[i*2]);
					int y = _y(_t->R().pointlist[i*2+1]);
					fl_color  (FL_RED);
					fl_pie    (x-r, y-r, 2*r+1, 2*r+1, 0,360);
					fl_color  (FL_BLACK);
					fl_circle (x, y, r);
				}

				// Draw triangles
				fl_color (FL_BLUE);
				for (int i=0; i<_t->R().numberoftriangles; ++i)
				{
					int x1 = _x(_t->R().pointlist[_t->R().trianglelist[i*3  ]*2  ]);
					int y1 = _y(_t->R().pointlist[_t->R().trianglelist[i*3  ]*2+1]);
					int x2 = _x(_t->R().pointlist[_t->R().trianglelist[i*3+1]*2  ]);
					int y2 = _y(_t->R().pointlist[_t->R().trianglelist[i*3+1]*2+1]);
					int x3 = _x(_t->R().pointlist[_t->R().trianglelist[i*3+2]*2  ]);
					int y3 = _y(_t->R().pointlist[_t->R().trianglelist[i*3+2]*2+1]);
					//fl_line (x1,y1, x2,y2, x3,y3);
					fl_line (x1,y1, x2,y2);
					fl_line (x2,y2, x3,y3);
					fl_line (x3,y3, x1,y1);
				}

				/*
				// Draw edges
				fl_line_style (0, 2);
				fl_color (FL_MAGENTA);
				for (int i=0; i<_t->R().numberofedges; ++i)
				{
					int x1 = _x(_t->R().pointlist[_t->R().edgelist[i*2  ]*2  ]);
					int y1 = _y(_t->R().pointlist[_t->R().edgelist[i*2  ]*2+1]);
					int x2 = _x(_t->R().pointlist[_t->R().edgelist[i*2+1]*2  ]);
					int y2 = _y(_t->R().pointlist[_t->R().edgelist[i*2+1]*2+1]);
					fl_line (x1,y1, x2,y2);
				}
				fl_line_style (0);
				*/
			}
		}

	private:
		// Data
		double             _Xmin; // Real coordinates
		double             _Ymin; // Real coordinates
		double             _Xmax; // Real coordinates
		double             _Ymax; // Real coordinates
		double             _sfx;  // Scale factor: x=xmin+(X-Xmin)*sf and y=ymin+ymax-(Y-Ymin)*sf, where x and y are screen coordinates
		double             _sfy;  // Scale factor: x=xmin+(X-Xmin)*sf and y=ymin+ymax-(Y-Ymin)*sf, where x and y are screen coordinates
		TriangleIO const * _t;

		// Private methods
		inline int _x (double   X) const { return static_cast<int>((x()+1)        +_sfx*(X-_Xmin)); } // X in screen coordinates
		inline int _y (double   Y) const { return static_cast<int>((y()+1)+(h()-2)-_sfy*(Y-_Ymin)); } // Y in screen coordinates
		inline int _l (double   L) const { return static_cast<int>(_sfx*L)                        ; } // Length in screen coordinates

	}; // class DrawArea }}}

	// Constructor
	MainWindow(int width=1024, int height=824)
		: Fl_Double_Window (width,height,"Triangle"), // {{{
		  _npoints  (4),
		  _continue (false)
	{

		// Add widgets to window
		begin();
			// Inputs
			_wnpoints = new Fl_Input(100, 0, 100, 30, "NPoints = ");
			char buf[256];
			snprintf(buf, 256, "%d", _npoints); _wnpoints->value(buf);
			// Buttons
			_wquit = new Fl_Button(200,   0, 100, 30, "&Quit");
			_wstep = new Fl_Button(300,   0, 100, 30, "&Step");
			_wrun  = new Fl_Button(400,   0, 100, 30, "&Run" );
			_wstop = new Fl_Button(500,   0, 100, 30, "S&top");
			_wquit->callback(_quit_cb, this);
			_wstep->callback(_step_cb, this);
			_wrun ->callback(_run_cb , this);
			_wstop->callback(_stop_cb, this);
			// Draw area
			_wda = new DrawArea(/*xmin*/5, /*ymin*/35 , /*width*/w()-10, /*height*/h()-40);
		end();

		// Set focus
		_wrun->take_focus();

		// Set resizable and show window
		//resizable(this);
		//resizable(_wda);
		show();

	} // }}}

	// Destructor
	~MainWindow()
	{ // {{{
		delete _wnpoints;
		delete _wquit;
		delete _wrun;
		delete _wstop;
		delete _wstep;
		delete _wda;
	} // }}}

private:
	// Widgets
	Fl_Input  * _wnpoints;
	Fl_Button * _wquit; // Quit button
	Fl_Button * _wrun;  // Run Button
	Fl_Button * _wstop; // Stop Button
	Fl_Button * _wstep; // Step Button
	DrawArea  * _wda;   // Draw area
	uint32_t    _npoints;
	bool        _continue;

	// Callbacks (must be static)
	static void _run_cb  (Fl_Widget * o, void * v) { ((MainWindow*)v)->_run (); }
	static void _stop_cb (Fl_Widget * o, void * v) { ((MainWindow*)v)->_stop(); }
	static void _step_cb (Fl_Widget * o, void * v) { ((MainWindow*)v)->_step(); }
	static void _quit_cb (Fl_Widget * o, void * v) { ((MainWindow*)v)->_quit(); }

	// Run callback
	inline void _run ()
	{ // {{{

		_continue = true;
		while (_continue)
		{
			_step ();
			clock_t start_time;
			start_time = clock();
			while((clock()-start_time)<1*CLOCKS_PER_SEC);
		}

	} // }}}

	inline void _stop () { _continue = false; }

	// Step callback
	inline void _step ()
	{ // {{{

		// Input data
		_npoints = atoi(_wnpoints->value());

		TriangleIO in(_npoints);
		TriangleIO out;

		uint32_t   nx  = static_cast<uint32_t>(sqrt(_npoints)+1);
		uint32_t   ny  = static_cast<uint32_t>(sqrt(_npoints)+1);
		double     dx  = (XMAX-XMIN)/nx;
		double     dy  = (YMAX-YMIN)/ny;
		double   * Xc  = new double   [nx*ny];
		double   * Yc  = new double   [ny*ny];
		uint32_t * idx = new uint32_t [ny*ny];

		uint32_t n = 0;
		for (uint32_t j=0; j<ny; ++j)
		{
			for (uint32_t i=0; i<nx; ++i)
			{
				double  bet = Rnd::T<double>::Gen(0.8,1.2);
				Xc [j*nx+i] = XMIN+i*dx*bet;
				Yc [j*nx+i] = YMIN+j*dy*bet;
				idx[j*nx+i] = n;
				n++;
			}
		}

		Rnd::Shuffle(nx*ny, idx);
		uint32_t m = 0;
		for (uint32_t j=0; j<_npoints; ++j)
		{
			in.R().pointlist[j*2  ] = Xc[idx[m]];
			in.R().pointlist[j*2+1] = Yc[idx[m]];
			m++;
		}

		delete [] Xc;
		delete [] Yc;
		delete [] idx;

		triangulate("Qze",in.P(),out.P(),NULL);

		// Draw
		_wda -> SetTriangleIO (&out);
		_wda -> redraw();
		Fl::wait(0);
		_wstep -> redraw();
		_wrun  -> redraw();
		_wstop -> redraw();

	} // }}}

	// Quit callback
	inline void _quit() { hide(); }

}; // class MainWindow

///////////////////////////////////////////////////////////////////////////////////////////////////// main /////

int main(int argc, char **argv) try
{
	// Initialize random numbers generator
	if (argc>1) Rnd::Init(atoi(argv[1]));
	else        Rnd::Init(1234);

	// Window
	MainWindow win;
	Fl::run();
	return 0;
}
catch (char const * m) // {{{
{
	std::cout << "Fatal: " << m << std::endl;
	exit (1);
}
catch (...)
{
	std::cout << "Some exception (...) ocurred\n";
} //}}} 

// vim:fdm=marker
