/*
 * $Id: memory.c,v 1.17 2003/12/01 09:10:15 troth Exp $
 *
 ****************************************************************************
 *
 * simulavr - A simulator for the Atmel AVR family of microcontrollers.
 * Copyright (C) 2001, 2002, 2003  Theodore A. Roth
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 ****************************************************************************
 */

/**
 * \file memory.c
 * \brief Memory access functions.
 *
 * This module provides functions for reading and writing to simulated memory.
 * The Memory class is a subclass of AvrClass.
 */

#include <config.h>

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "avrerror.h"
#include "avrmalloc.h"
#include "avrclass.h"
#include "utils.h"
#include "callback.h"
#include "op_names.h"

#include "storage.h"
#include "flash.h"

#include "vdevs.h"
#include "memory.h"
#include "stack.h"
#include "register.h"
#include "sram.h"
#include "eeprom.h"
#include "timers.h"
#include "ports.h"

#include "avrcore.h"

#include "display.h"

/** \brief Allocates memory for a new memory object. */

Memory *
mem_new (void)
{
    Memory *mem;

    mem = avr_new (Memory, 1);
    mem_construct (mem);
    class_overload_destroy ((AvrClass *)mem, mem_destroy);

    return mem;
}

/** \brief Constructor for the memory object. */

void
mem_construct (Memory *mem)
{
    if (mem == NULL)
        avr_error ("passed null ptr");

    class_construct ((AvrClass *)mem);

    mem->dev = NULL;
}

/** \brief Descructor for the memory object. */

void
mem_destroy (void *mem)
{
    Memory *this = (Memory *)mem;

    if (mem == NULL)
        return;

    dlist_delete_all (this->dev);

    class_destroy (mem);
}

/** \brief Attach a device to the device list.
 
  Devices that are accessed more often should be attached
  last so that they will be at the front of the list.
 
  A default virtual device can be overridden by attaching
  a new device ahead of it in the list.  */

void
mem_attach (Memory *mem, VDevice *dev)
{
    if (mem == NULL)
        avr_error ("passed null ptr");

    if (dev == NULL)
        avr_error ("attempt to attach null device");

    /* push the new dev to front of list to allow overriding */
    mem->dev = dlist_add_head (mem->dev, (AvrClass *)dev);
}

/** \brief Find the VDevice associated with the given address. */

VDevice *
mem_get_vdevice_by_addr (Memory *mem, int addr)
{
    return (VDevice *)dlist_lookup (mem->dev, (AvrClass *)&addr,
                                    vdev_addr_cmp);
}

/** \brief Find the VDevice associated with the given name. */

VDevice *
mem_get_vdevice_by_name (Memory *mem, char *name)
{
    return (VDevice *)dlist_lookup (mem->dev, (AvrClass *)name,
                                    vdev_name_cmp);
}

/** \brief Reads byte from memory and sanity-checks for valid address. 
 * 
 * \param mem A pointer to the memory object
 * \param addr The address to be read 
 * \return The byte found at that address addr
 */

uint8_t
mem_read (Memory *mem, int addr)
{
    VDevice *dev = mem_get_vdevice_by_addr (mem, addr);

    if (dev == NULL)
    {
        avr_warning ("**** Attempt to read invalid addr: 0x%04x\n", addr);
        return 0;
    }

    return vdev_read (dev, addr);
}

/** \brief Writes byte to memory and updates display for io registers. 
 * 
 * \param mem A pointer to a memory object
 * \param addr The address to be written to
 * \param val The value to be written there
 */

void
mem_write (Memory *mem, int addr, uint8_t val)
{
    VDevice *dev = mem_get_vdevice_by_addr (mem, addr);

    if (dev == NULL)
    {
        avr_warning ("**** Attempt to write invalid addr: 0x%04x\n", addr);
        return;
    }

    /* update the display for io registers here */

    if ((addr >= IO_REG_ADDR_BEGIN) && (addr < IO_REG_ADDR_END))
        display_io_reg (addr - IO_REG_ADDR_BEGIN, val);

    vdev_write (dev, addr, val);
}

static int
mem_reset_iter_func (AvrClass *data, void *user_data)
{
    vdev_reset ((VDevice *)data);
    return 0;                   /* don't delete data from list */
}

/** \brief Resets every device in the memory object.
 * \param mem A pointer to the memory object.
 */

void
mem_reset (Memory *mem)
{
    mem->dev = dlist_iterator (mem->dev, mem_reset_iter_func, NULL);
}

static void
mem_reg_dump_core (Memory *mem, FILE * f_core)
{
    int i, j;

    fprintf (f_core, "General Purpose Register Dump:\n");
    for (i = 0; i < 32; i += 8)
    {
        for (j = i; j < (i + 8); j++)
            fprintf (f_core, "r%02d=%02x  ", j, mem_read (mem, j));
        fprintf (f_core, "\n");
    }
    fprintf (f_core, "\n");
}

/** \brief Fetch the name and value of the io register (addr). 
 *
 * \param mem A pointer to the memory object.
 * \param addr The address to fetch from.
 * \param val A pointer where the value of the register is to be copied.
 * \param buf A pointer to where the name of the register should be copied.
 * \param bufsiz The maximum size of the the buf string.
 */

void
mem_io_fetch (Memory *mem, int addr, uint8_t * val, char *buf, int bufsiz)
{
    VDevice *dev;
    int is_port;

    if ((addr < IO_REG_ADDR_BEGIN) || (addr >= IO_REG_ADDR_END))
    {
        *val = 0;
        strncpy (buf, "NOT AN IO REG", bufsiz);
    }
    else
    {
        dev = mem_get_vdevice_by_addr (mem, addr);
        if (dev == NULL)
        {
            strncpy (buf, "Reserved", bufsiz);
            *val = 0;
        }
        else
        {
            strncpy (buf, vdev_get_reg_name (dev, addr), bufsiz);

            is_port = 0;
            if (strncmp (vdev_get_name (dev), "PORT", 4) == 0)
            {
                is_port = 1;
                port_ext_disable ((Port *)dev);
            }
            *val = vdev_read (dev, addr);
            if (is_port)
                port_ext_enable ((Port *)dev);
        }
    }
}

static void
mem_io_reg_dump_core (Memory *mem, FILE * f_core)
{
    int i, j;
    char name[80];
    uint8_t val;

    int begin = IO_REG_ADDR_BEGIN;
    int end = IO_REG_ADDR_END;
    int half = (end - begin) / 2;
    int mid = begin + half;

    fprintf (f_core, "IO Register Dump:\n");
    for (i = begin; i < mid; i++)
    {
        for (j = i; j < end; j += half)
        {
            memset (name, '\0', sizeof (name));
            mem_io_fetch (mem, j, &val, name, sizeof (name) - 1);

            fprintf (f_core, "%02x : %-10s : 0x%02x               ", j - half,
                     name, val);
        }
        fprintf (f_core, "\n");
    }
    fprintf (f_core, "\n");
}

static void
mem_sram_display (VDevice *dev, FILE * f_core, int base, int size)
{
    int i;
    int dup = 0;
    int ndat = 16;
    char line[80];
    char last_line[80];
    char buf[80];
    line[0] = last_line[0] = '\0';

    for (i = base; i < (base + size); i++)
    {
        if (((i % ndat) == 0) && strlen (line))
        {
            if (strncmp (line, last_line, 80) == 0)
            {
                dup++;
            }
            else
            {
                if (dup > 0)
                    fprintf (f_core, "  -- last line repeats --\n");
                fprintf (f_core, "%04x : %s\n", i - ndat, line);
                dup = 0;
            }
            strncpy (last_line, line, 80);
            line[0] = '\0';
        }
        snprintf (buf, 80, "%02x ", vdev_read (dev, i));
        strncat (line, buf, 80);
    }
    if (dup > 0)
    {
        fprintf (f_core, "  -- last line repeats --\n");
        fprintf (f_core, "%04x : %s\n", i - ndat, line);
    }
    fprintf (f_core, "\n");
}

static void
mem_sram_dump_core (Memory *mem, FILE * f_core)
{
    VDevice *dev;
    int size, base;

    /*
     * Dump the internal sram
     */
    dev = mem_get_vdevice_by_name (mem, "SRAM");
    if (dev == NULL)
        return;                 /* device has no sram */

    fprintf (f_core, "Internal SRAM Memory Dump:\n");
    base = vdev_get_base (dev);
    size = vdev_get_size (dev);
    mem_sram_display (dev, f_core, base, size);

    /*
     * If external sram present, dump it too.
     */

    dev = mem_get_vdevice_by_name (mem, "ESRAM");
    if (dev == NULL)
        return;                 /* device has no sram */

    fprintf (f_core, "External SRAM Memory Dump:\n");
    base = vdev_get_base (dev);
    size = vdev_get_size (dev);
    mem_sram_display (dev, f_core, base, size);

}

static void
mem_eeprom_dump_core (Memory *mem, FILE * f_core)
{
    VDevice *dev = mem_get_vdevice_by_name (mem, "EEProm");

    if (dev != NULL)
        eeprom_dump_core ((EEProm *)dev, f_core);
}

/** \brief Dump all the various memory locations to a file descriptor 
 *         in text format.
 *
 *  \param mem A memory object.
 *  \param f_core An open file descriptor.
 */

void
mem_dump_core (Memory *mem, FILE * f_core)
{
    mem_reg_dump_core (mem, f_core);
    mem_io_reg_dump_core (mem, f_core);
    mem_sram_dump_core (mem, f_core);
    mem_eeprom_dump_core (mem, f_core);
}
