/*
 * $Id: uart.c,v 1.1 2004/01/02 04:03:21 troth Exp $
 *
 ****************************************************************************
 *
 * simulavr - A simulator for the Atmel AVR family of microcontrollers.
 * Copyright (C) 2003  Keith Gudger
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 ****************************************************************************
 */

/**
 * \file uart.c
 * \brief Module to simulate the AVR's uart module.
 */

#include <config.h>

#include <stdio.h>
#include <stdlib.h>

#include "avrerror.h"
#include "avrmalloc.h"
#include "avrclass.h"
#include "utils.h"
#include "callback.h"
#include "op_names.h"

#include "storage.h"
#include "flash.h"

#include "vdevs.h"
#include "memory.h"
#include "stack.h"
#include "register.h"
#include "sram.h"
#include "eeprom.h"
#include "timers.h"
#include "ports.h"
#include "uart.h"

#include "avrcore.h"

#include "intvects.h"

/****************************************************************************\
 *
 * uart Interrupts 
 *
\****************************************************************************/

static uint8_t uart_intr_read (VDevice *dev, int addr);
static void uart_intr_write (VDevice *dev, int addr, uint8_t val);
static void uart_intr_reset (VDevice *dev);
static char *uart_intr_reg_name (VDevice *dev, int addr);
static int uart_intr_cb (uint64_t time, AvrClass *data);

int UART_Int_Table[] = {
    irq_vect_table_index (UART_RX), /* uart Rx complete */
    irq_vect_table_index (UART_UDRE), /* uart data register empty */
    irq_vect_table_index (UART_TX) /* uart Tx complete */
};

int UART0_Int_Table[] = {
    irq_vect_table_index (USART0_RX), /* uart Rx complete */
    irq_vect_table_index (USART0_UDRE), /* uart data register empty */
    irq_vect_table_index (USART0_TX) /* uart Tx complete */
};

int UART1_Int_Table[] = {
    irq_vect_table_index (USART1_RX), /* uart Rx complete */
    irq_vect_table_index (USART1_UDRE), /* uart data register empty */
    irq_vect_table_index (USART1_TX) /* uart Tx complete */
};

/** \brief Allocate a new uart interrupt */

UARTIntr_T *
uart0_intr_new (uint8_t uart_num)
{
    UARTIntr_T *uart;

    uart = avr_new (UARTIntr_T, 1);
    uart0_intr_construct (uart);
    class_overload_destroy ((AvrClass *)uart, uart_intr_destroy);

    if (uart_num > ONE_UART)
        uart->Int_Table = &UART0_Int_Table[0];
    else
        uart->Int_Table = &UART_Int_Table[0];

    return uart;
}

UARTIntr_T *
uart1_intr_new (uint8_t uart_num)
{
    UARTIntr_T *uart;

    uart = avr_new (UARTIntr_T, 1);
    uart1_intr_construct (uart, uart_num);
    class_overload_destroy ((AvrClass *)uart, uart_intr_destroy);

    uart->Int_Table = &UART1_Int_Table[0];

    return uart;
}

/** \brief Constructor for uart interrupt object. */

void
uart0_intr_construct (UARTIntr_T *uart)
{
    char *name = "uartIntr0";

    if (uart == NULL)
        avr_error ("passed null ptr");

    vdev_construct ((VDevice *)uart, name, UART_INTR_BASE_0, UART_INTR_SIZE,
                    uart_intr_read, uart_intr_write, uart_intr_reset,
                    uart_intr_reg_name);

    uart_intr_reset ((VDevice *)uart);
}

void
uart1_intr_construct (UARTIntr_T *uart, uint8_t uart_num)
{
    char *name = "uartIntr1";

    if (uart == NULL)
        avr_error ("passed null ptr");

    if (uart_num > TWO_UART)
        vdev_construct ((VDevice *)uart, name, UART_INTR_BASE_128,
                        UART_INTR_SIZE, uart_intr_read, uart_intr_write,
                        uart_intr_reset, uart_intr_reg_name);
    else
        vdev_construct ((VDevice *)uart, name, UART_INTR_BASE_1,
                        UART_INTR_SIZE, uart_intr_read, uart_intr_write,
                        uart_intr_reset, uart_intr_reg_name);

    uart_intr_reset ((VDevice *)uart);
}

/** \brief Destructor for uart interrupt object. */

void
uart_intr_destroy (void *uart)
{
    if (uart == NULL)
        return;

    vdev_destroy (uart);
}

static uint8_t
uart_intr_read (VDevice *dev, int addr)
{
    UARTIntr_T *uart = (UARTIntr_T *)dev;

    switch (addr - vdev_get_base (dev))
    {
        case UART_INTR_UBRR_ADDR:
            return (uart->ubrr);
        case UART_INTR_UCR_ADDR:
            return (uart->ucr);
        case UART_INTR_USR_ADDR:
            return (uart->usr);
        default:
            avr_error ("Bad address: 0x%04x", addr);
    }
    return 0;                   /* will never get here */
}

static void
uart_intr_write (VDevice *dev, int addr, uint8_t val)
{
    UARTIntr_T *uart = (UARTIntr_T *)dev;
    CallBack *cb;

    switch (addr - vdev_get_base (dev))
    {
        case UART_INTR_UBRR_ADDR:
            (uart->ubrr = val);
            break;
        case UART_INTR_USR_ADDR:
            if (val & mask_TXC)
                uart->usr &= ~mask_TXC;
            break;
        case UART_INTR_UCR_ADDR:
            (uart->ucr = val);  /* look for interrupt enables */
            if (((uart->ucr & mask_TXEN) && (uart->ucr & mask_TXCIE))
                || ((uart->ucr & mask_RXEN) && (uart->ucr & mask_RXCIE))
                || (uart->ucr & mask_UDRIE))
            {
                if (uart->intr_cb == NULL)
                {
                    /* we need to install the intr_cb function */
                    cb = callback_new (uart_intr_cb, (AvrClass *)uart);
                    uart->intr_cb = cb;
                    avr_core_async_cb_add ((AvrCore *)vdev_get_core (dev),
                                           cb);
                }
            }
            else
            {
                uart->intr_cb = NULL;
                /* no interrupt are enabled, remove the callback */
            }

            break;
        default:
            avr_error ("Bad address: 0x%04x", addr);
    }
}

static void
uart_intr_reset (VDevice *dev)
{
    UARTIntr_T *uart = (UARTIntr_T *)dev;

    uart->intr_cb = NULL;

    uart->ubrr = 0;
    uart->usr = 0;
    uart->ucr = 0;
    uart->usr_shadow = 0;
}

static char *
uart_intr_reg_name (VDevice *dev, int addr)
{
    switch (addr - vdev_get_base (dev))
    {
        case UART_INTR_UBRR_ADDR:
            return ("UBRR");
        case UART_INTR_UCR_ADDR:
            return ("UCR");
        case UART_INTR_USR_ADDR:
            return ("USR");
        default:
            avr_error ("Bad address: 0x%04x", addr);
    }
    return NULL;                /* will never get here */
}

static int
uart_intr_cb (uint64_t time, AvrClass *data)
{
    UARTIntr_T *uart = (UARTIntr_T *)data;

    if (uart->intr_cb == NULL)
        return CB_RET_REMOVE;

    if ((uart->ucr & mask_RXCIE) && (uart->usr & mask_RXC))
        /* an enabled interrupt occured */
    {
        AvrCore *core = (AvrCore *)vdev_get_core ((VDevice *)uart);
        avr_core_irq_raise (core, (uart->Int_Table[URX]));
    }

    if ((uart->ucr & mask_TXCIE) && (uart->usr & mask_TXC))
        /* an enabled interrupt occured */
    {
        AvrCore *core = (AvrCore *)vdev_get_core ((VDevice *)uart);
        avr_core_irq_raise (core, (uart->Int_Table[UTX]));
        uart->usr &= ~mask_TXC;
    }

    if ((uart->ucr & mask_UDRIE) && (uart->usr & mask_UDRE)
        && (uart->usr_shadow & mask_UDRE))
        /* an enabled interrupt occured */
    {
        AvrCore *core = (AvrCore *)vdev_get_core ((VDevice *)uart);
        avr_core_irq_raise (core, (uart->Int_Table[UUDRE]));
        uart->usr_shadow &= ~mask_UDRE; /* only issue one interrupt / udre */
    }

    return CB_RET_RETAIN;
}

/****************************************************************************\
 *
 * uart  
 *
\****************************************************************************/

static uint8_t uart_read (VDevice *dev, int addr);
static void uart_write (VDevice *dev, int addr, uint8_t val);
static void uart_reset (VDevice *dev);
static char *uart_reg_name (VDevice *dev, int addr);
static int uart_clk_incr_cb (uint64_t ck, AvrClass *data);

/** \brief Allocate a new uart structure. */

UART_T *
uart_new (uint8_t uart_num)
{
    UART_T *uart;

    uart = avr_new (UART_T, 1);
    if (uart_num > ONE_UART)
        uart1_construct (uart, uart_num);
    else
        uart0_construct (uart);
    class_overload_destroy ((AvrClass *)uart, uart_destroy);

    return uart;
}

/** \brief Constructor for uart object. */

void
uart0_construct (UART_T *uart)
{
    char *name = "uart0";

    if (uart == NULL)
        avr_error ("passed null ptr");

    vdev_construct ((VDevice *)uart, name, UART_BASE_0, UART_SIZE, uart_read,
                    uart_write, uart_reset, uart_reg_name);

    uart_reset ((VDevice *)uart);
    uart->int_name = "uartIntr0";
}

void
uart1_construct (UART_T *uart, uint8_t uart_num)
{
    char *name = "uart1";

    if (uart == NULL)
        avr_error ("passed null ptr");

    if (uart_num > TWO_UART)
        vdev_construct ((VDevice *)uart, name, UART_BASE_128, UART_SIZE,
                        uart_read, uart_write, uart_reset, uart_reg_name);
    else
        vdev_construct ((VDevice *)uart, name, UART_BASE_1, UART_SIZE,
                        uart_read, uart_write, uart_reset, uart_reg_name);

    uart_reset ((VDevice *)uart);
    uart->int_name = "uartIntr1";
}

/** \brief Destructor for uart object. */

void
uart_destroy (void *uart)
{
    if (uart == NULL)
        return;

    vdev_destroy (uart);
}

static uint8_t
uart_read (VDevice *dev, int addr)
{
    UART_T *uart = (UART_T *)dev;
    UARTIntr_T *uart_t;
    int offset = addr - vdev_get_base (dev);
    uint16_t udr_temp;

    uart_t =
        (UARTIntr_T *)avr_core_get_vdev_by_name ((AvrCore *)
                                                 vdev_get_core ((VDevice *)
                                                                uart),
                                                 uart->int_name);

    if (offset == UART_UDR_ADDR)
    {
        uart_t->usr &= ~mask_RXC; /* clear RXC bit in USR */
        if (uart->clk_cb)       /* call back already installed */
        {
            udr_temp = uart_port_rd (addr);
            uart->udr_rx = (uint8_t) udr_temp; /* lower 8 bits */
            if ((uart_t->ucr & mask_CHR9) && /* 9 bits rec'd */
                (udr_temp & (1 << 8))) /* hi bit set */
                uart_t->ucr |= mask_RXB8;
            else
                uart_t->ucr &= ~mask_RXB8;
        }
        return uart->udr_rx;
    }
    else
        avr_error ("Bad address: 0x%04x", addr);

    return 0;                   /* will never get here */
}

static void
uart_write (VDevice *dev, int addr, uint8_t val)
{
    UART_T *uart = (UART_T *)dev;
    UARTIntr_T *uart_t;
    int offset = addr - vdev_get_base (dev);
    CallBack *cb;

    uart_t =
        (UARTIntr_T *)avr_core_get_vdev_by_name ((AvrCore *)
                                                 vdev_get_core ((VDevice *)
                                                                uart),
                                                 uart->int_name);

    if (offset == UART_UDR_ADDR)
    {
        if (uart_t->usr & mask_UDRE)
        {
            uart_t->usr &= ~mask_UDRE;
            uart_t->usr_shadow &= ~mask_UDRE;
        }
        else
        {
            uart_t->usr |= mask_UDRE;
            uart_t->usr_shadow |= mask_UDRE;
        }
        uart->udr_tx = val;

        /*
         * When the user writes to UDR, a callback is installed for 
         * clock generated increments. 
         */

        uart->divisor = (uart_t->ubrr + 1) * 16;

        /* install the clock incrementor callback (with flair!) */

        if (uart->clk_cb == NULL)
        {
            cb = callback_new (uart_clk_incr_cb, (AvrClass *)uart);
            uart->clk_cb = cb;
            avr_core_clk_cb_add ((AvrCore *)vdev_get_core ((VDevice *)uart),
                                 cb);
        }

        /* set up timer for 8 or 9 clocks based on ucr 
           (includes start and stop bits) */

        uart->tcnt = (uart_t->ucr & mask_CHR9) ? 11 : 10;
    }
    else
    {
        avr_error ("Bad address: 0x%04x", addr);
    }
}

static void
uart_reset (VDevice *dev)
{
    UART_T *uart = (UART_T *)dev;

    uart->clk_cb = NULL;

    uart->udr_rx = 0;
    uart->udr_tx = 0;
    uart->tcnt = 0;
    uart->divisor = 0;
}

static char *
uart_reg_name (VDevice *dev, int addr)
{
    switch (addr - vdev_get_base (dev))
    {
        case UART_UDR_ADDR:
            return "UDR";
        default:
            avr_error ("Bad address: 0x%04x", addr);
    }

    return NULL;
}

static int
uart_clk_incr_cb (uint64_t ck, AvrClass *data)
{
    UART_T *uart = (UART_T *)data;
    UARTIntr_T *uart_t;
    uint8_t last = uart->tcnt;

    uart_t =
        (UARTIntr_T *)avr_core_get_vdev_by_name ((AvrCore *)
                                                 vdev_get_core ((VDevice *)
                                                                uart),
                                                 uart->int_name);

    if (uart->clk_cb == NULL)
        return CB_RET_REMOVE;

    if (uart->divisor <= 0)
        avr_error ("Bad divisor value: %d", uart->divisor);

    /* decrement clock if ck is a mutliple of divisor */

    uart->tcnt -= ((ck % uart->divisor) == 0);

    if (uart->tcnt != last)     /* we've changed the counter */
    {
        if (uart->tcnt == 0)
        {
            if (uart_t->usr & mask_UDRE) /* data register empty */
            {
                uart_t->usr |= mask_TXC;
                return CB_RET_REMOVE;
            }
            else                /* there's a byte waiting to go */
            {
                uart_t->usr |= mask_UDRE;
                uart_t->usr_shadow |= mask_UDRE; /* also write shadow */

                /* set up timer for 8 or 9 clocks based on ucr,
                   (includes start and stop bits) */

                uart->tcnt = (uart_t->ucr & mask_CHR9) ? 11 : 10;
            }
        }
    }

    return CB_RET_RETAIN;
}

uint16_t
uart_port_rd (int addr)
{
    int data;
    char line[80];

    while (1)
    {
        fprintf (stderr, "\nEnter 9 bits of hex data to read into the uart at "
                 "address 0x%04x: ", addr);

        /* try to read in a line of input */
        if (fgets (line, sizeof (line), stdin) == NULL)
            continue;

        /* try to parse the line for a byte of data */
        if (sscanf (line, "%x\n", &data) != 1)
            continue;

        break;
    }

    return (uint16_t) (data & 0x1ff);
}

void
uart_port_wr (uint8_t val)
{
    fprintf (stderr, "wrote 0x%02x to uart\n", val);
}
