
(* This file is part of Numerix.  Numerix is free software; you can
   redistribute it and/or modify it under the terms of the GNU General
   Public License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.

   This program is distributed in the hope that it will be useful, but
   WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
   USA *)

#! /bin/sh
case $MODULENAME in
clong) prefix=c; longueur=longint; chiffre=word;;
dlong) prefix=d; longueur=longint; chiffre=cardinal;;
slong) prefix=s; longueur=longint; chiffre=cardinal;;
esac
cat <<EOF | sed -e "s/xx_/"$prefix"x_/g" -e "s/xz_/"$prefix"z_/g"
(* +------------------------------------------------------------------------+
   |                                                                        |
   |             Entiers extensibles de longueur arbitraire                 |
   |                                                                        |
   |                      Interface publique Pascal                         |
   |                                                                        |
   +------------------------------------------------------------------------+ *)

(* M. Quercia, 18/08/2001 *)

unit $MODULENAME;

{+----------------------------------------------------------------------------+
 }                               interface                                    {
 +----------------------------------------------------------------------------+}

type longueur = $longueur;
type chiffre  = $chiffre;
   
type _int =  record
  hd   : longueur;      (* longueur et signe *)
  val  : array[0..0] of chiffre; (* chiffres *)
end;	  
type _xint = record     (* entier extensible *)
  lmax : longueur;      (* capacit maximale *)
  e    : _int;          (* valeur            *)
end;
type xint = ^_xint;
	       
(* allocation/dsallocation *)
function  xnew : xint;
procedure xfree(var x : xint);

(* initialisation  partir d'un entier, d'un long ou d'une chaine *)
function  of_int     (a : longint) : xint;        cdecl; external 'cnumx' name 'xx_of_int';
function  of_xint    (a : xint   ) : xint;        cdecl; external 'cnumx' name 'xx_of_xint';
function  of_string  (s : pchar  ) : xint;        cdecl; external 'cnumx' name 'xx_of_string';
procedure copy       (var x : xint; a : xint);    cdecl; external 'cnumx' name 'xx_copy';
procedure copy_int   (var x : xint; a : longint); cdecl; external 'cnumx' name 'xx_copy_int';
procedure copy_string(var x : xint; s : pchar);   cdecl; external 'cnumx' name 'xx_copy_string';

(* conversion en chaine *)
function string_of(x : xint) : ansistring;

(* accs  la structure binaire, lowbits, highbits et int_of retournent
  des mots de 31 bits seulement pour compatibilit avec Ocaml *)
function nbits   (a : xint)              : longint; cdecl; external 'cnumx' name 'xx_nbits';
function lowbits (a : xint)              : longint; cdecl; external 'cnumx' name 'xx_lowbits';
function highbits(a : xint)              : longint; cdecl; external 'cnumx' name 'xx_highbits';
function nth_word(a : xint; n : longint) : longint; cdecl; external 'cnumx' name 'xx_nth_word';
function int_of  (a : xint)              : longint; cdecl; external 'cnumx' name 'xx_int_of';

(* comparaison *)
function eq   (a,b : xint) : boolean;
function neq  (a,b : xint) : boolean;
function inf  (a,b : xint) : boolean;
function infeq(a,b : xint) : boolean;
function sup  (a,b : xint) : boolean;
function supeq(a,b : xint) : boolean;
function cmp  (a,b : xint) : longint;
function sgn  (a   : xint) : longint;

function cmp_1(a : xint; b : longint) : longint; cdecl; external 'cnumx' name 'xx_cmp_1';
function eq_1   (a : xint; b : longint) : boolean;
function neq_1  (a : xint; b : longint) : boolean;
function inf_1  (a : xint; b : longint) : boolean;
function infeq_1(a : xint; b : longint) : boolean;
function sup_1  (a : xint; b : longint) : boolean;
function supeq_1(a : xint; b : longint) : boolean;

(* oprations usuelles *)
procedure abs     (var x   : xint; a     : xint); 	       cdecl; external 'cnumx' name 'xx_abs';
procedure neg     (var x   : xint; a     : xint); 	       cdecl; external 'cnumx' name 'xx_neg';
procedure sqr     (var x   : xint; a     : xint); 	       cdecl; external 'cnumx' name 'xx_sqr';
procedure sqrt    (var x   : xint; a     : xint);              cdecl; external 'cnumx' name 'xx_sqrt';
procedure fact    (var x   : xint; a     : xint);     	       cdecl; external 'cnumx' name 'xx_fact';
procedure add     (var x   : xint; a,b   : xint); 	       cdecl; external 'cnumx' name 'xx_add';
procedure sub     (var x   : xint; a,b   : xint);              cdecl; external 'cnumx' name 'xx_sub';
procedure mul     (var x   : xint; a,b   : xint);              cdecl; external 'cnumx' name 'xx_mul';
procedure quomod  (var x,y : xint; a,b   : xint); 	       cdecl; external 'cnumx' name 'xx_quomod';
procedure quo     (var x   : xint; a,b   : xint); 	       cdecl; external 'cnumx' name 'xx_quo';
procedure powmod  (var x   : xint; a,b,c : xint);              cdecl; external 'cnumx' name 'xx_powmod';
procedure add_1   (var x   : xint; a     : xint; b : longint); cdecl; external 'cnumx' name 'xx_add_1';
procedure sub_1   (var x   : xint; a     : xint; b : longint); cdecl; external 'cnumx' name 'xx_sub_1';
procedure mul_1   (var x   : xint; a     : xint; b : longint); cdecl; external 'cnumx' name 'xx_mul_1';
procedure shiftl  (var x   : xint; a     : xint; b : longint); cdecl; external 'cnumx' name 'xx_shl';
procedure shiftr  (var x   : xint; a     : xint; b : longint); cdecl; external 'cnumx' name 'xx_shr';
procedure split   (var x,y : xint; a     : xint; b : longint); cdecl; external 'cnumx' name 'xx_split';
procedure join    (var x   : xint; a,b   : xint; c : longint); cdecl; external 'cnumx' name 'xx_join';
function  quomod_1(var x   : xint; a     : xint; b : longint): longint;
                                                               cdecl; external 'cnumx' name 'xx_quomod_1';
procedure pow     (var x   : xint; a     : xint; b : longint); cdecl; external 'cnumx' name 'xx_pow';
procedure root    (var x   : xint; a     : xint; b : longint); cdecl; external 'cnumx' name 'xx_root';
procedure gcd     (var z         : xint; a,b : xint);  	       cdecl; external 'cnumx' name 'xx_gcd';
procedure gcd_ex  (var x,y,z     : xint; a,b : xint); 	       cdecl; external 'cnumx' name 'xx_gcd_ex';
procedure cfrac   (var r,s,x,y,z : xint; a,b : xint); 	       cdecl; external 'cnumx' name 'xx_cfrac';

(* chronomtrage *)
procedure chrono(msg : pchar); cdecl; external 'cnumx' name 'chrono';

(* erreurs *)
procedure failwith(msg : pchar);
   
{+----------------------------------------------------------------------------+
 }                            implementation                                  {
 +----------------------------------------------------------------------------+}

uses Linux; (* pour stderr *)

                        {+-------------------+
                         |  Gestion mmoire  |
                         +-------------------+}

procedure free(x :  pointer); cdecl; external 'c';
function  xnew : xint;        begin xnew := NIL; end;
procedure xfree(var x: xint); begin free(x); x := NIL; end;

                          {+------------------------+
                           |  conversion en chaine  |
                           +------------------------+}

function  z_string_of(var x : _int): pchar; cdecl; external 'cnumx' name 'xz_string_of';

function string_of(x: xint) : ansistring;
var p : pchar;
    s : ansistring;
begin
   p := z_string_of(x^.e);
   s := p;
   free(p);
   string_of := s;
end;

(* comparaison *)
function z_cmp(var a,b : _int) : longint; cdecl; external 'cnumx' name 'xz_cmp';
function eq   (a,b : xint) : boolean; begin eq    := z_cmp(a^.e,b^.e) =  0 end;
function neq  (a,b : xint) : boolean; begin neq   := z_cmp(a^.e,b^.e) <> 0 end;
function inf  (a,b : xint) : boolean; begin inf   := z_cmp(a^.e,b^.e) <  0 end;
function infeq(a,b : xint) : boolean; begin infeq := z_cmp(a^.e,b^.e) <= 0 end;
function sup  (a,b : xint) : boolean; begin sup   := z_cmp(a^.e,b^.e) >  0 end;
function supeq(a,b : xint) : boolean; begin supeq := z_cmp(a^.e,b^.e) >= 0 end;
function cmp  (a,b : xint) : longint; begin cmp   := z_cmp(a^.e,b^.e);     end;

function sgn(a : xint) : longint;
begin
   if a^.e.hd = 0 then sgn := 0 else if a^.e.hd > 0 then sgn := 1 else sgn := -1;
end;

function eq_1   (a : xint; b : longint) : boolean; begin eq_1    := cmp_1(a, b) =  0 end;
function neq_1  (a : xint; b : longint) : boolean; begin neq_1   := cmp_1(a, b) <> 0 end;
function inf_1  (a : xint; b : longint) : boolean; begin inf_1   := cmp_1(a, b) <  0 end;
function infeq_1(a : xint; b : longint) : boolean; begin infeq_1 := cmp_1(a, b) <= 0 end;
function sup_1  (a : xint; b : longint) : boolean; begin sup_1   := cmp_1(a, b) >  0 end;
function supeq_1(a : xint; b : longint) : boolean; begin supeq_1 := cmp_1(a, b) >= 0 end;

                                 {+-----------+
                                  |  Erreurs  |
                                  +-----------+}

procedure failwith(msg : pchar); begin writeln(stderr,msg); halt(1); end;

end.  
EOF