/*
 *   $Id: Slag.cpp,v 1.54 2006/04/09 00:28:39 rhizome Exp $
 *
 *      Copyright (C) 2004, 2005, 2006 Alex Marandon
 *
 *  This file is part of slag, a pattern-based audio sequencer.
 *
 *  slag is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  slag is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with slag; see the file COPYING.  If not, write to
 *  the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */
#include <ctime>
#include <iostream>
#include <qsettings.h>
#include <qtextcodec.h>
#include <qtranslator.h>
#include <qdatetime.h>
#include <qmessagebox.h>
#include <qprogressdialog.h>

#include "common.h"
#include "config.h"
#include "constants.h"
#include "Slag.h"
#include "config/Config.h"
#include "audio_engine/Serialize.h"
#include "audio_engine/SongReader.h"
#include "audio_engine/Song.h"
#include "audio_IO/AudioDriver.h"
#include "gui/MainWidget.h"

// Inclusion des drivers audio
#include "audio_IO/NullDriver.h"
#include "audio_IO/WavFileOutput.h"

#ifdef WITH_JACK
  #include "audio_IO/JackDriver.h"
  #include "audio_IO/ChannelJackDriver.h"
#endif

#ifdef WITH_LIBAO
  #include "audio_IO/LibaoDriver.h"
#endif

using std::cerr;
using std::endl;

SongReader* globalSongReader;
int client_buf_size;
const QString not_compiled_str("Not compiled with %1 support.");

inline void process(sample_t * buffer) {
    globalSongReader->read(buffer, client_buf_size);
}

inline void processChannel(Channel* channel, sample_t * buffer) {
    globalSongReader->read(buffer, client_buf_size, channel);
}

void jackErrorCallBack(const char* msg) {
    QMessageBox::critical(NULL, "Slag - JACK error", QString(msg));
    qFatal(QString(msg));
}

PatternChangeEvent::PatternChangeEvent(QString name) : 
    QCustomEvent(PatternChangeEventType),
    pattern_name(name)
{ }

void msgHandler(QtMsgType type, const char *msg) {
    const QString time_str = QTime::currentTime().toString("hh:mm:ss.zzz");
    switch (type) {
        case QtDebugMsg:
            if (Config::instance()->debug()) {
                cerr << time_str << " => " << msg << endl;
            }
            break;
        case QtWarningMsg:
            cerr << time_str << " WARNING: " << msg << endl;
            break;
        case QtFatalMsg:
            cerr << time_str << " FATAL ERROR: " << msg << endl;
            abort();     // deliberately core dump
    }
}

void Slag::restartAudioDriver() {
    initAudioDriver();
#ifdef WITH_JACK
    if (config->audioDriverName() == JACK_LABEL && config->jackTrackOutput()) {
        addChannelsToJackMulti();
    }
#endif
    driver->connect();
}

void Slag::showJackErrorPopup() {
    QMessageBox::critical(NULL, tr("Slag - JACK error"), 
            tr("Unable to connect JACK server. Is jackd running?"
               "\nNow we try to use libao for audio output."));
}

void Slag::showLibaoErrorPopup() {
    QMessageBox::critical(NULL, tr("Slag - libao error"), 
            tr("Unable to use libao for audio output"
               "\nYou won't hear anything..."));
}

void Slag::initAudioDriver() {

    if (driver != NULL) {
        qDebug("Disconnecting previous audio driver.");
        driver->disconnect();
        delete driver;
    }

    qDebug(QString("Initializing audio driver %1.")
                   .arg(config->audioDriverName()));

    if (config->audioDriverName() == LIBAO_LABEL)
        initLibaoDriver();
    else if (config->audioDriverName() == NULL_LABEL)
        initNullDriver();
    else if (config->jackTrackOutput())
        initJackTrackDriver();
    else
        initJackDriver();

    if (songReader != NULL) {
        songReader->setBufferSize(config->buf_size());
    }
}

void Slag::initJackDriver() {
#ifdef WITH_JACK
    try {
        driver = new JackDriver(process, jackErrorCallBack);
        config->setAudioDriverName(JACK_LABEL);
    } catch (audio_IO::JackConnectionException & e) {
        showJackErrorPopup();
        qWarning(e.what());
        initLibaoDriver();
    }
    client_buf_size = config->buf_size();
#else
    qFatal(not_compiled_str.arg("JACK"));
#endif
}

void Slag::initJackTrackDriver() {
#ifdef WITH_JACK
    try {
        driver = new ChannelJackDriver(processChannel, jackErrorCallBack);
        config->setAudioDriverName(JACK_LABEL);
    } catch (audio_IO::JackConnectionException  e) {
        showJackErrorPopup();
        qWarning(e.what());
        initLibaoDriver();
    }
    client_buf_size = config->buf_size();
#else
    qFatal(not_compiled_str.arg("JACK"));
#endif
}

void Slag::initLibaoDriver() {
#if WITH_LIBAO
    try {
        client_buf_size = config->buf_size();
        driver = new LibaoDriver(process, client_buf_size);
        config->setAudioDriverName(LIBAO_LABEL);
    } catch (audio_IO::LibaoConnectionException & e) {
        qWarning(e.what());
        showLibaoErrorPopup();
        initNullDriver();
    }
#else
    qFatal(not_compiled_str.arg("libao"));
#endif
}

void Slag::initNullDriver() {
    driver = new NullDriver(process, client_buf_size);
    config->setAudioDriverName(NULL_LABEL);
    qDebug("Using Null driver.");
}

Slag::Slag(int argc, char**argv)
    : QApplication(argc, argv), 
      driver(NULL), 
      main_widget(NULL), 
      songReader(NULL),
      song_file(NULL)
{
    qInstallMsgHandler(msgHandler);
    installTranslators();

    // Traitement des options de la ligne de commande
    try {
        config = Config::instance(argc, argv);
    } catch (BadOption & bad_opt) {
        cerr << endl << bad_opt.msg() << endl << endl;
        die_usage(argv);
    }
    
    srand(static_cast<unsigned>(time(0)));

    initAudioDriver();

    config->buf_size(client_buf_size); // taille de buffer par dfaut

    QSettings settings;
    settings.setPath("alpage.org", "Slag");
    settings.beginGroup("/Slag");

    QStringList recentFiles = settings.readListEntry("/recentFiles");
    if(settings.readBoolEntry("/openLastSong") && recentFiles.count() > 0) {
        try {
            openSong(recentFiles[0]);
        } catch (audio_engine::SongOpenException e) {
            QMessageBox::critical(NULL, tr("Slag - Song open error"), 
                    tr("Unable to open last song file %1."
                        "\nStarting with a new song.").arg(recentFiles[0]));
            openSong(config->songfile());
        }
    } else {
        openSong(config->songfile());
    }

    settings.endGroup();
}

void Slag::installTranslators() {
    // FIXME il faudra trouver quelque chose de plus portable pour trouver les
    // fichiers de traduction de Qt
    const QString slag_translation_file = 
        QString("%1/share/%2/l10n/").arg(CONFIG_PREFIX, PACKAGE_TARNAME) 
        + QTextCodec::locale();
    QTranslator* slag_translator = new QTranslator;
    if(!slag_translator->load(slag_translation_file)) {
        qWarning(tr("Unable to load Slag translation file %1 for locale %2.")
                .arg(slag_translation_file, QTextCodec::locale()));
    }
    installTranslator(slag_translator);

    const QString qt_translation_file = QString(getenv("QTDIR")) + 
        "/translations/" + QString("qt_") + QTextCodec::locale();

    QTranslator* qt_translator = new QTranslator;
    if(!qt_translator->load(qt_translation_file)) {
        qWarning(tr("Unable to load Qt translation file %1 for locale %2.")
                .arg(qt_translation_file, QTextCodec::locale()));
    }
    installTranslator(qt_translator);
}

int Slag::start() {
    return exec();
}

void Slag::openSong(const QString & filename) {
    QProgressDialog progress_dialog(
            tr("Loading song %1...").arg(filename),
            0, 0, main_widget, 0, TRUE);
    progress_dialog.setMinimumDuration(2000);
    SongFile* old_song_file = NULL;
    if (song_file != NULL) { 
        old_song_file = song_file;
    }
    song_file = new SongFile(filename);
    delete old_song_file;
    // Construction et initialisation de la song
    if (songReader != NULL) {
        driver->disconnect();
        delete songReader;
    }

    Song* new_song;
    if(config->no_gui()) {
        new_song = new Song(song_file->getRootElem());
    } else {
        new_song = new Song(song_file->getRootElem(), &progress_dialog);
    }
    processEvents();

    songReader = new SongReader(new_song, config->buf_size());
    connect(songReader, SIGNAL(stop()), this, SLOT(stop()));

    globalSongReader = songReader;

    setSamplesPerPad(song()->tempo());

    if (! config->no_gui()) {
        initGui(&progress_dialog);
    }

    songReader->init();
    //song_updater->start(QThread::TimeCriticalPriority);
    connect(songReader, SIGNAL(patternAdded(Pattern*)), 
            main_widget, SLOT(addStackedWidget(Pattern*)));
    connect(songReader, SIGNAL(patternRemoved(Pattern*)), 
            main_widget, SLOT(removeStackedWidget(Pattern*)));
#ifdef WITH_JACK
    if (config->audioDriverName() == JACK_LABEL && config->jackTrackOutput()) {
        addChannelsToJackMulti();
    }
#endif
    driver->connect();

    main_widget->rewind();

    if (config->no_gui()) play();
}

#ifdef WITH_JACK
void Slag::addChannelsToJackMulti() {
    ((ChannelJackDriver*)driver)->clear();
    ((ChannelJackDriver*)driver)->activate();
    if (config->jackTrackOutput()) {
        connect(song(), SIGNAL(channelAdded(Channel*)), 
                (ChannelJackDriver*) driver, SLOT(addChannel(Channel*)));
        connect(song(), SIGNAL(channelRemoved(Channel*)), 
                (ChannelJackDriver*) driver, SLOT(deleteChannel(Channel*)));

        QPtrList<Channel>::iterator channels_end = song()->channels().end();
        for (QPtrList<Channel>::iterator it(song()->channels().begin());
                it != channels_end; ++it)
            ((ChannelJackDriver*)driver)->addChannel(*it);
    }
}
#endif

void Slag::initGui(QProgressDialog* progress_dialog) {
    if (main_widget == NULL) {
        main_widget = new MainWidget(this, progress_dialog);
    } else {
        main_widget->songChanged(progress_dialog);
    }

    connect(song(), SIGNAL(modified()), 
            main_widget,           SLOT(songModified()));
    main_widget->setModified(false);

    connect(songReader, SIGNAL(patternChanged(const QString &)),
            this,  SLOT(autoPatternChange(const QString &)));
    connect(songReader, SIGNAL(updated()), this,  SLOT(oneStepForward()));
    main_widget->show();
    setMainWidget(main_widget);
}

void Slag::autoPatternChange(const QString & name) {
    QApplication::postEvent(main_widget, new PatternChangeEvent(name));
}

void Slag::oneStepForward() {
    QApplication::postEvent(main_widget, new QCustomEvent(SongUpdateEventType));
}

bool Slag::patternMode() {
    return songReader->patternMode();
}

void Slag::updatePatternList(const StringItemList& list) {
    songReader->updatePatternList(list);
}

void Slag::play() {
    qDebug("Play");
    if (songReader->isPlaying()) {
        songReader->setPlaying(false);
        main_widget->rewind();
    }
    songReader->init();
    songReader->setPlaying(true);
}

void Slag::stop() {
    qDebug("Stop");
    songReader->setPlaying(false);
    main_widget->rewind();
    driver->disableFileOutput();
}

void Slag::setTempo(int tempo) {
    song()->setTempo(tempo);
    setSamplesPerPad(tempo);
}

void Slag::setSamplesPerPad(int tempo) {
    unsigned int new_val = static_cast<unsigned int>(
            ((static_cast<float>(60) / tempo) / 4)
            * Config::instance()->samplerate() * 2
            );
    if (new_val % 2 != 0) {
        ++new_val; // must be even
    }
    Config::instance()->samplesPerPad(new_val);
}

void Slag::setVolume(int v) {
    song()->setVolume(v);
}

void Slag::setSongMode() {
    songReader->setSongMode();
}

void Slag::setPatternMode() {
    songReader->setPatternMode();
}

void Slag::setLoopMode(bool v) {
    songReader->setLoopMode(v);
}

void Slag::patternChange(const QString& pattern_name) {
    if (not songReader->isPlaying()) {
        main_widget->patternWidgetChange(pattern_name);
    }
    songReader->setNextWantedPattern(pattern_name);
}

void Slag::deleteChannel(Channel* channel) {
    main_widget->deleteChannel(channel);
    songReader->deleteChannel(channel);
}

void Slag::exportWav(const QString &filename) {
    driver->enableFileOutput(filename);
    play();
}

void Slag::save() {
    song_file->save(song()->getDom());
}

Song* Slag::song() { 
    return songReader->song(); 
}

SongFile* Slag::songFile() {
    return song_file;
}

void Slag::die_usage(char**argv) {
    cerr << tr("Usage: %1 [OPTIONS] [file.xml]").arg(argv[0]) 
         << endl
         << tr("Options :")
         << endl
         << " -h\t" << tr("Print this message.")
         << endl
         << " -a\t" << tr("Use libao for audio output.")
         << endl
#ifdef WITH_JACK
         << " -j\t" << tr("Use JACK with two output ports for audio output "
                           "(this is the default).") 
         << endl
         << " -t\t" 
         << tr("Use JACK with separate ports for each tracks for audio output.")
         << endl
#endif
         << " -n\t" << tr("No audio output.")
         << endl 
         << " -p\t" << tr("Player mode : Slag runs without graphical user interface and file supplied as argument is played.")
         << endl
         << " -v\t" << tr("Enable verbose debugging output.") << endl;
    std::exit(1);
}

int main(int argc, char **argv) {
    Slag slag(argc, argv);
    slag.start();
}

//EOF
