-- Copyright (C) 2010-2011 Papavasileiou Dimitris                             
--                                                                      
-- This program is free software: you can redistribute it and/or modify 
-- it under the terms of the GNU General Public License as published by 
-- the Free Software Foundation, either version 3 of the License, or    
-- (at your option) any later version.                                  
--                                                                      
-- This program is distributed in the hope that it will be useful,      
-- but WITHOUT ANY WARRANTY; without even the implied warranty of       
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        
-- GNU General Public License for more details.                         
--                                                                      
-- You should have received a copy of the GNU General Public License    
-- along with this program.  If not, see <http://www.gnu.org/licenses/>.

require "frames"
require "switches"

function slipstream.control (values)
   self = frames.node {
      phase = 1,
      granularity = 10,

      name = "Control",

      step = function (self)
		if self.phase == self.granularity then
		   -- Update each system.

		   for key, child in children (self) do
		      if child.update then
			 child.update(child)
		      end
		   end
		   
		   self.phase = 1
		else
		   self.phase = self.phase + 1
		end
	     end,
   }

   for key, value in pairs(values) do
      self[key] = value
   end

   return self
end

function slipstream.cascade (values)
   self = frames.node {
      input = 0,
      output = 0,

      name = "Cascade",
      
      update = function(self)
		  local i, input

		  -- Gather the results of each system.

		  i, input = 1, self.input

		  while self[i] do
		     self[i].input = input
		     
		     if self[i].update then
			self[i].update(self[i])
		     end

		     input, i = self[i].output, i + 1
		  end
		  
		  self.output = input
	       end
   }

   for key, value in pairs(values) do
      self[key] = value
   end

   return self
end

function slipstream.rubberband (values)
   self = frames.cursor {
      cursor = shapes.points {
	 color = {0.11, 0.26, 0.007},
	 opacity = 0.8,
	 width = 6,
	 
	 [1] = {0, 0},
      },
      
      knob = shapes.points {
	 color = {0.11, 0.26, 0.007},
	 opacity = 0.8,
	 width = 6,
	 
	 [1] = {0, 0},
      },
      
      vector = shapes.lines {
	 stipple = {3, 0x5555},
	 opacity = 0.8,
	 width = 3,
	 
	 [1] = {0, 0},
	 [2] = {0, 0},
      },

      events = frames.event {
	 commands = {0, 0},

	 motion = function (self, button, x, y)
		     if self.engaged then
			self.commands = {self.origin[1] - x,
					 self.origin[2] - y}

			self.parent.knob[1] = self.commands
			self.parent.vector[1] = self.commands
			self.parent.vector.color =
			   self.commands[2] < 0 and
			   {1, 0, 0} or {0.1607, 0.9763, 0}
		     end
		  end,
	 
	 buttonpress = function (self, button, x, y)
			  if button == 2 then
			     self.origin = {x, y}
			     self.engaged = true

			     graphics.grabinput = true
			  end
		       end,
	 
	 unlink = function (self)
		     graphics.grabinput = true
		  end,
      },
      
      update = function(self)
		  for key, child in children (self) do
		     if child.update then
			child.input = child.axis and
			   self.events.commands[child.axis] or 0

			child.update(child)
		     end
		  end   
			
		  self.output = 0
	       end,
   }

   for key, value in pairs(values) do
      self[key] = value
   end

   return self
end

function slipstream.lag (values)
   self = frames.node {
      input = 0,
      output = 0,
      lag = 0,

      name = "Lag",
      
      step = function (self, h, t)
		if self.lag > 0 then
		   self.output = (self.output +
				  (self.input - self.output) *
			          h / self.lag)
		else
		   self.output = self.input
		end
	     end,
   }

   for key, value in pairs(values) do
      self[key] = value
   end

   return self
end

function slipstream.blackbox (values)
   self = frames.node {
      input = 0,
      output = 0,

      name = "Blackbox",
      
      link = function (self)
		if self.entries then
		   self.replay = #self.entries > 0
		else
		   self.entries = {}
		end

		self.index = 1
	     end,

      update = function (self)
		  if not self.replay then
		     self.entries[self.index] = self.input
		  elseif self.index > #self.entries then
		     self.output = 0
		     return
		  end

		  self.output = self.entries[self.index]
		  self.index = self.index + 1
	       end,
   }

   for key, value in pairs(values) do
      self[key] = value
   end

   return self
end
