-- Copyright (C) 2010-2011 Papavasileiou Dimitris                           
--                                                                      
-- This program is free software: you can redistribute it and/or modify 
-- it under the terms of the GNU General Public License as published by 
-- the Free Software Foundation, either version 3 of the License, or    
-- (at your option) any later version.                                  
--                                                                      
-- This program is distributed in the hope that it will be useful,      
-- but WITHOUT ANY WARRANTY; without even the implied warranty of       
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        
-- GNU General Public License for more details.                         
--                                                                      
-- You should have received a copy of the GNU General Public License    
-- along with this program.  If not, see <http://www.gnu.org/licenses/>.

require "dynamics"
require "automotive"
require "shading"
require "units"
require "nature"
require "moremath"
require "transforms"
require "toon"

local modulated = resources.pipeline {resources.modulate(0.1, 0.4, 1),
				      textures.clamped}

-- Setup collision response.

dynamics.collision.track = function (a, b)
				if a.iswheel or b.iswheel then
				   assert (a.iswheel)

				   -- print (a, b)

				   if b.isground then
				      return 1, 0.6, 0.6, 1, 0.6, 0.6
				   else
				      if layout.scaling then
					 return 1, unpack (layout.scaling)
				      end
				   end
				end
			     end

-- Load the imagery and elevation data.

local tiles = {}
local samples, imagery, detail, sparse, grass, dirt, asphalt

samples = resources.dofile ("slipstream/imagery/laguna-elevation.lc")
imagery = resources.dofile ("slipstream/imagery/laguna-imagery.lc")
sparse = resources.dofile ("slipstream/imagery/sparse-grass-detail.lc")
grass = resources.dofile ("slipstream/imagery/grass-detail.lc")
dirt = resources.dofile ("slipstream/imagery/dirt-detail.lc")
asphalt = resources.dofile ("slipstream/imagery/asphalt-detail.lc")

for i = 0, 440 do
   tiles[i] = {
      samples[2 * i + 1],
      samples[2 * i + 2],
      imagery[i + 1],
      {i == 220 and 52.880 or 159.738,
       i == 220 and -13.227 or -42.597}
   }
end

local elevation, ground = nature.elevation {    
   size = {21, 21},
   depth = 11,
   resolution = math.scale({0.625, 0.625}, 1),
}

return nature.atmosphere {
   size = {1024, 512},

   turbidity = 4,

   rayleigh = math.scale({6.95e-06, 1.18e-05, 2.44e-05}, 1),
   mie = 7e-5,

   sun = {1.74, math.pi/4},

   -- The environment.

   land = (options.toon and nature.earth or nature.land) {
      albedo = 2,
      separation = layout.separation,

      palette = {
      	 {grass, math.scale({1, 1}, 0.0025), layout.grass},
      	 {sparse, math.scale({1, 1}, 0.0045), layout.sparse},
      	 {dirt, math.scale({1, 1}, 0.0025), layout.dirt},
      	 {asphalt, math.scale({1, 1}, 0.0015), layout.asphalt},
      },
      
      elevation {
	 position = units.meters {1280/2 - 533.24,
				  1280/2 - 202.28,
				  0},

	 target = 15000,
	 anisotropy = 8,

	 unpack(tiles)
      },
   },

   environment = bodies.environment {
      ground {
      	 position = units.meters {1280/2 - 533.24,
      				  1280/2 - 202.28,
      				  0},

      	 isground = true,
      },

      track = automotive.racetrack {
	 tessellation = {units.feet(12), 0.01 / 3, 0.01},
	 scale = {2, 2},

	 istrack = true,

	 -- Turn 1.

	 {units.feet(269), units.feet(15), units.feet(15), 0, 0, 0},
	 {units.feet(150), units.feet(15), units.feet(15), 0, -0.045, -0.033},
	 {units.feet(182), units.feet(15), units.feet(15), 1 / units.feet(516), -0.045, -0.119},
	 {units.feet(239), units.feet(20), units.feet(20), 0, -0.045, -0.028},

	 -- Turn 2.

	 {units.feet(301), units.feet(20), units.feet(20), 0, -0.02, -0.03},
	 {units.feet(100), units.feet(20), units.feet(25), 1 / units.feet (127), -0.02, -0.037},
	 {units.feet(173), units.feet(20), units.feet(30), 1 / units.feet (127), -0.02, -0.044},
	 {units.feet(174), units.feet(20), units.feet(25), 1 / units.feet (127), -0.02, -0.038},

	 -- Turn 3.

	 {units.feet(75), units.feet(20), units.feet(20), 0, 0, -0.005},
	 {units.feet(25), units.feet(20), units.feet(20), 0, 0, 0},
	 {units.feet(75), units.feet(20), units.feet(20), 1 / units.feet (-900), 0, 0.022},
	 {units.feet(325), units.feet(20), units.feet(20), 1 / units.feet (-900), 0, 0.022},
	 {units.feet(50), units.feet(20), units.feet(20), 1 / units.feet (-145), 0, 0.022},
	 {units.feet(195), units.feet(20), units.feet(20), 1 / units.feet (-145), 0, 0.026},
	 {units.feet(50), units.feet(20), units.feet(20), 0, 0, 0.015},
	 {units.feet(70), units.feet(20), units.feet(20), 0, 0, 0},

	 -- Turn 4.

	 {units.feet(470), units.feet(20), units.feet(20), 0, 0, 0},
	 {units.feet(80), units.feet(20), units.feet(20), 0, 0, 0.022},
	 {units.feet(90), units.feet(20), units.feet(20), 1 / units.feet (-200), 0, 0.026},
	 {units.feet(150), units.feet(20), units.feet(20), 1 / units.feet (-200), 0, 0.026},
	 {units.feet(90), units.feet(20), units.feet(20), 0, 0, 0.015},
	 {units.feet(50), units.feet(20), units.feet(20), 0, 0, 0},

	 -- Turn 5.

	 {units.feet(298), units.feet(20), units.feet(20), 0, 0, 0},
	 {units.feet(60), units.feet(20), units.feet(20), 1 / units.feet (-1000), 0.01, 0.02},
	 {units.feet(255), units.feet(20), units.feet(20), 1 / units.feet (-1000), 0.01, 0.02},
	 {units.feet(60), units.feet(20), units.feet(20), 0, 0.01, 0},
	 {units.feet(480), units.feet(20), units.feet(20), 0, 0.01, 0},
	 {units.feet(100), units.feet(20), units.feet(20), 0, 0.01, -0.021},
	 {units.feet(80), units.feet(20), units.feet(20), 1 / units.feet (170), 0.03, -0.085},
	 {units.feet(234), units.feet(20), units.feet(20), 1 / units.feet (170), 0.03, -0.113},
	 {units.feet(80), units.feet(17), units.feet(17), 0, 0.04, -0.085},
	 {units.feet(60), units.feet(15), units.feet(15), 0, 0.04, -0.068},

	 -- Turn 6.
	 
	 {units.feet(285), units.feet(15), units.feet(15), 0, 0.07, -0.068},
	 {units.feet(180), units.feet(15), units.feet(15),
	  1 / units.feet (-9000), 0.07, -0.057},
	 {units.feet(180), units.feet(15), units.feet(15), 0, 0.07, -0.066},
	 {units.feet(78), units.feet(15), units.feet(15), 1 / units.feet (1755), 0.07, -0.068},
	 {units.feet(200), units.feet(15), units.feet(15), 0, 0.07, -0.068},
	 {units.feet(50), units.feet(15), units.feet(15), 1 / units.feet (80), 0.07, -0.068},
	 {units.feet(42), units.feet(15), units.feet(15), 1 / units.feet (80), 0.07, -0.096},
	 {units.feet(60), units.feet(15), units.feet(15), 0, 0.07, -0.038},

	 -- Turn 7.
	 
	 {units.feet(224), units.feet(15), units.feet(15), 0, 0.05, 0},
	 {units.feet(80), units.feet(15), units.feet(15), 1 / units.feet (500), 0.05, -0.005},
	 {units.feet(80), units.feet(15), units.feet(15), 0, 0.05, -0.005},
	 {units.feet(56), units.feet(15), units.feet(15), 0, 0.05, -0.005},
	 {units.feet(90), units.feet(15), units.feet(15), 1 / units.feet (1125), 0.05, -0.01},
	 {units.feet(95), units.feet(15), units.feet(15), 0, 0.05, -0.01},
	 {units.feet(83), units.feet(15), units.feet(15), 0, 0.05, -0.01},
	 {units.feet(120), units.feet(15), units.feet(15), 1 / units.feet (-3200), 0.05, -0.015},
	 {units.feet(112), units.feet(15), units.feet(15), 0, 0.05, -0.015},
	 {units.feet(150), units.feet(15), units.feet(15), 1 / units.feet (1950), 0.05, -0.015},
	 {units.feet(139), units.feet(15), units.feet(15), 0, 0.05, -0.015},
	 {units.feet(60), units.feet(15), units.feet(15), 1 / units.feet (-450), 0.05, 0},
	 {units.feet(122), units.feet(15), units.feet(15), 1 / units.feet (-450), 0.05, 0.016},
	 {units.feet(90), units.feet(15), units.feet(15), 0, 0.04, 0},

	 -- Turn 8.
	 
	 {units.feet(127), units.feet(20), units.feet(20), 0, -0.11, -0.066},
	 {units.feet(35), units.feet(20), units.feet(20), 1 / units.feet (75), -0.11, -0.126},
	 {units.feet(51), units.feet(20), units.feet(20), 1 / units.feet (75), -0.11, -0.126},
	 {units.feet(51), units.feet(20), units.feet(20), 1 / units.feet (75), -0.11, -0.066},
	 {units.feet(35), units.feet(20), units.feet(20), 0, -0.11, 0},
	 {units.feet(30), units.feet(20), units.feet(20), 1 / units.feet (-90), -0.11, 0.058},
	 {units.feet(45), units.feet(20), units.feet(20), 1 / units.feet (-90), -0.11, 0.114},
	 {units.feet(45), units.feet(20), units.feet(20), 1 / units.feet (-90), -0.11, 0.068},
	 {units.feet(30), units.feet(20), units.feet(20), 0, -0.11, 0.059},
	 {units.feet(163), units.feet(20), units.feet(20), 1 / units.feet (-1170), -0.09, 0},
	 {units.feet(80), units.feet(20), units.feet(20), 1 / units.feet (-369), -0.09, -0.048},
	 {units.feet(50), units.feet(20), units.feet(20), 0, -0.09, -0.081},
	 {units.feet(20), units.feet(20), units.feet(20), 1 / units.feet (310), -0.09, -0.098},
	 {units.feet(43), units.feet(20), units.feet(20), 1 / units.feet (310), -0.09, -0.114},
	 {units.feet(20), units.feet(20), units.feet(20), 0, -0.09, -0.12},
	 {units.feet(27), units.feet(20), units.feet(20), 1 / units.feet (-545), -0.09, -0.112},
	 {units.feet(10), units.feet(20), units.feet(20), 0, -0.09, -0.116},

	 -- Turn 9.
	 
	 {units.feet(60), units.feet(20), units.feet(20), 1 / units.feet (265), -0.09, -0.125},
	 {units.feet(205), units.feet(20), units.feet(20), 1 / units.feet (265), -0.09, -0.125},
	 {units.feet(200), units.feet(20), units.feet(20), 1 / units.feet (265), -0.09, -0.057},
	 {units.feet(100), units.feet(20), units.feet(20), 0, -0.09, -0.021},

	 -- Turn 10.
	 
	 {units.feet(356), units.feet(20), units.feet(20), 0, -0.02, 0},
	 {units.feet(100), units.feet(20), units.feet(20), 1 / units.feet (-150), -0.02, 0.06},
	 {units.feet(121), units.feet(20), units.feet(20), 1 / units.feet (-150), -0.02, 0.1},
	 {units.feet(60), units.feet(20), units.feet(20), 0, -0.01, 0.066},
	 {units.feet(120), units.feet(20), units.feet(20), 0, -0.01, 0},

	 -- Turn 11.
	 
	 {units.feet(578), units.feet(20), units.feet(20), 0, -0, 0},
	 {units.feet(40), units.feet(20), units.feet(20), 1 / units.feet (45), 0, 0.02},
	 {units.feet(45), units.feet(25), units.feet(25), 1 / units.feet (45), 0, 0.02},
	 {units.feet(40), units.feet(25), units.feet(25), 0, 0, 0.02},
	 {units.feet(514.455), units.feet(25), units.feet(25), 0, 0, 0}, 
	 {units.feet(300), units.feet(25), units.feet(25), 0, 0.1, 0},
	 {units.feet(200), units.feet(25), units.feet(25), 0, 0.0273, 0},
	 {units.feet(50), units.feet(20), units.feet(20), 1 / units.feet (137.628), 0.02, 0},
	 {units.feet(50.85), units.feet(15), units.feet(15), 0, 0, 0},
	 {19.715, units.feet(15), units.feet(15), 0, 0, 0},
      },
   },
   
   sunlight = options.toon and toon.lamp {
      position = math.transform (transforms.euler (math.deg(1.09),
						   0,
						   math.deg(5.74)),
				 {0, 1e5, 0}),

      intensity = {[0] = 0.4, [0.04] = 0.6, [0.15] = 0.8, [0.45] = 1},
      orientation = transforms.euler (180, 0, 0),
      ambience = 0.4,

      islamp = true
   } or shading.light {
      position = math.transform (transforms.euler (math.deg(1.09),
						   0,
						   math.deg(5.74)),
				 {0, 1e5, 0}),

      offset = {1, 2},
      volume = {-1000, 1000, -1000, 1000, 0.1, 1000},
      intensity = math.scale({1, 1, 0.8}, 1.2),
      size = {0, 0},
      
      islight = true
   },

   skylight = not options.toon and shading.ambient {
      orientation = transforms.euler (0, 180, 0),
      intensity = modulated "slipstream/imagery/horizon.lc",
   },
   
   haze = (options.toon and toon.haze or shading.fog) {
      link = function (self)
		self.color = self.parent.intensity
		self.linear = 5 * (self.parent.turbidity - 1) *
		self.parent.mie
	  end
   },
}
