# svs_tracking.analysis.entities

#    Copyright (c) 2005 Simon Yuill.
#
#    This file is part of 'Social Versioning System' (SVS).
#
#    'Social Versioning System' is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation; either version 2 of the License, or
#    (at your option) any later version.
#
#    'Social Versioning System' is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with 'Social Versioning System'; if not, write to the Free Software
#    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

"""
This contains classes for the basic entities used in analysing a cvs
repository.

These provide an object-based framework for looking at the cvs data.
The objects provide network serialisation through the use of the
Twisted Perspective Broker system:

U{http://twistedmatrix.com}


@author:	Simon Yuill
@copyright:	2005 Simon Yuill
@license:	GNU GPL version 2 or any later version
@contact:	simon@lipparosa.org
"""
# external imports
import os
from twisted.spread import pb

#########################
# FACTORY METHODS
#########################
def makeCVSServer(url):
	"""
	Factory method for CVSServer, use this instead of CVSServer().
	"""
	server = CVSServer()
	server.url = url
	return server

def makeCVSModule(name):
	"""
	Factory method for CVSModule, use this instead of CVSModule()
	"""
	module = CVSModule()
	module.name = name
	return module

def makeCVSFile(name, parent=None):
	"""
	Factory method for CVSFile, use this instead of CVSFile().
	"""
	cvsFile = CVSFile()
	cvsFile.setName(name)
	cvsFile.parent = parent
	return cvsFile

def makeCVSDirectory(name, parent=None):
	"""
	Factory method for CVSDirectory, use this instead of CVSDirectory().

	This will recursively build all the sub-directories within the specified
	directory.
	"""
	cvsDir = CVSDirectory()
	#cvsDir.name = os.path.split(name)[1]
	if name.startswith('/'):
		cvsDir.name = name[1:]
	else:
		cvsDir.name = name
	cvsDir.parent = parent
	# build file list
	if cvsDir.parent:
		path = cvsDir.parent.getPath()
	else:
		path = os.path.split(name)[0]
	f = open(os.path.join(path, name, 'CVS', 'Entries'))
	for entry in f:
		s=entry.split('/')
		if len(s) > 1:
			if s[0] == 'D':
				cvsDir.addDirectory(makeCVSDirectory(s[1], cvsDir))
		        else:
		        	cvsDir.addFile(makeCVSFile(s[1]))
	f.close()
	return cvsDir



	
#########################
# BASE CLASS
#########################
class CVSEntity(pb.Copyable, pb.RemoteCopy):
	"""
	Generic base class for other entities. Provides serialisation
	capabilities from the Twisted Perspective Broker system.
	"""
	def __init__(self):
		self.id = 0
	
	def setCopyableState(self, state):
		self.__dict__ = state

	def getStateToCopy(self):
		d = self.__dict__.copy()
		return d

# register with perspective broker
pb.setUnjellyableForClass(CVSEntity, CVSEntity)

#########################
# SERVER CLASSES
#########################
class CVSServer(CVSEntity):
	"""
	Holds data about server being analysed.
	"""
	def __init__(self):
		CVSEntity.__init__(self)
		self.url = None

# register with perspective broker
pb.setUnjellyableForClass(CVSServer, CVSServer)

	
class CVSModule(CVSEntity):
	"""
	Represents a module within the cvs.
	"""
	def __init__(self):
		CVSEntity.__init__(self)
		self.name = None

# register with perspective broker
pb.setUnjellyableForClass(CVSModule, CVSModule)


#########################
# FILE CLASSES
#########################
class CVSFile(CVSEntity):
	"""
	Represents a file contained within the cvs.
	"""
	def __init__(self):
		CVSEntity.__init__(self)
		self.name = None
		self.parent = None
		self.filetype = None
		self.abspath = None
	
	def setName(self, name):
		"""
		Sets name and filetype based on filename extension.
		"""
		self.name = name
		self.filetype = self.name.split(".")[-1:]

	def getPath(self):
		self.abspath = '' 
		if self.parent:
			self.abspath = self.parent.getPath()
		self.abspath = "%s/%s" % (self.abspath, self.name)
		return self.abspath

# register with perspective broker
pb.setUnjellyableForClass(CVSFile, CVSFile)


class CVSDirectory(CVSFile):
	"""
	Represents a directory holding files contained within the cvs.
	"""
	def __init__(self):
		CVSFile.__init__(self)
		self.files = {}
		self.dirs = {}

	def addFile(self, cvsFile):
		cvsFile.parent = self
		self.files[cvsFile.name] = cvsFile

	def addDirectory(self, cvsDir):
		cvsDir.parent = self
		self.dirs[cvsDir.name] = cvsDir

	def display(self):
		print "Directory: <%s>" % self.name
		print "__files:"
		for dirFile in self.files.values():
			print "- %s" % dirFile.name
		print "__directories:"
		for subDir in self.dirs.values():
			subDir.display()
		
# register with perspective broker
pb.setUnjellyableForClass(CVSDirectory, CVSDirectory)
		

class CVSFileMap(CVSEntity):
	"""
	Represents a map of all source files within a module.
	"""
	def __init__(self):
		CVSEntity.__init__(self)
		self.root = None
		

# register with perspective broker
pb.setUnjellyableForClass(CVSFileMap, CVSFileMap)


#########################
# METADATA CLASSES
#########################
class CVSFileLog(CVSEntity):
	"""
	Represents the log record for a file contained within the cvs.
	"""
	def __init__(self):
		CVSEntity.__init__(self)
		self.cvsFile = None

# register with perspective broker
pb.setUnjellyableForClass(CVSFileLog, CVSFileLog)


#########################
# PEOPLE CLASSES
#########################
class CVSUser(CVSEntity):
	"""
	Represents an account holder who can log into the cvs.
	"""
	def __init__(self):
		CVSEntity.__init__(self)
		self.name = None

# register with perspective broker
pb.setUnjellyableForClass(CVSUser, CVSUser)


class CVSContributer(CVSUser):
	"""
	Represents a contributer to the cvs.
	"""
	def __init__(self):
		CVSUser.__init__(self)
	
# register with perspective broker
pb.setUnjellyableForClass(CVSContributer, CVSContributer)


class CVSContributerList(CVSEntity):
	"""
	Maintains a list of contributers to a cvs module.
	"""
	def __init__(self):
		CVSEntity.__init__(self)
		self.contribList = {}

	def addContributer(self, contributerName):
		"""
		Checks if contributer already present in the list, 
		returns existing entry if present, creates new
		entry if not and returns this.
		"""
		if self.contribList.has_key(contributerName):
			return self.contribList[contributerName]
		else:
			contribObj = CVSContributer()
			contribObj.name = contributerName
			self.contribList[contributerName] = contribObj
			return contribObj
	
# register with perspective broker
pb.setUnjellyableForClass(CVSContributerList, CVSContributerList)
