# svs_tracking.visualisation.timeutilities

#    Copyright (c) 2005 Simon Yuill.
#
#    This file is part of 'Social Versioning System' (SVS).
#
#    'Social Versioning System' is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation; either version 2 of the License, or
#    (at your option) any later version.
#
#    'Social Versioning System' is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with 'Social Versioning System'; if not, write to the Free Software
#    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

"""
Specisla utlitiy classes and functions for dealing with time in a visualisation.


@author:	Simon Yuill
@copyright:	2005 Simon Yuill
@license:	GNU GPL version 2 or any later version
@contact:	simon@lipparosa.org
"""
# external imports
import calendar

# internal imports
from svs_tracking.visualisation.timeline import *


#########################
# CALENDAR TIMECODE
#########################

def calendarTimecodeFromString(timeStr):
	"""
	Factory method to produce L{Timecode} object from
	specified string.
	
	@raise L{TimecodeException}: If the string cannot be converted.
	"""
	timecode = Timecode()
	timecode.timeFromString(timeStr, True)
	#print "timecode:", timecode
	calTimecode = CalendarTimecode()
	calTimecode.add(timecode)
	return calTimecode

def calendarTimecodeFromList(timeList):
	"""
	Factory method to produce L{Timecode} object from
	specified list.
	
	@raise L{TimecodeException}: If the list cannot be converted.
	"""
	formatLength = len(CalendarTimecode.calFormat)
	if len(timeList) < formatLength:
		raise TimecodeException("list must match calendar format")
	calTimecode = CalendarTimecode()
	for idx in range(formatLength):
		calTimecode.timeValue[idx] = timeList[idx]
	calTimecode.add(timecodeFromString("0:0:0:0:0:0", CalendarTimecode.calFormat))
	return calTimecode


class CalendarTimecode(Timecode):
	"""
	Specialised version of L{Timecode} that provides support 
	for calendar time.
	"""
	monthLengths = [0, 31, 28, 31, 30, 31, 30, 31, 31, 30, 31, 30, 31]
	calFormat = [9999,12,31,24,60,60]

	def __init__(self):
		Timecode.__init__(self)
		self.timeValue = [0,1,0,0,0,0]
		self.format = self.__class__.calFormat
		self.scFormat = None

	def increment(self, timeStep):
		"""
		Increases current
		"""

	def formatFromString(self, timeStr, applyToSelf=False):
		"""
		Overrides the original method in L{Timecode} so 
		that the format cannot be changed.
		"""
		return self.format
	
	def add(self, otherTimecode, multiplier=1):
		"""
		Increases the current timecode values by the amounts
		specified in C{otherTimecode}.

		@todo: allow for float values on C{multiplier}
		"""
		# seconds
		secs = self.timeValue[5] + (otherTimecode.timeValue[5] * multiplier)
		mins = secs / self.format[5]
		secs = secs % self.format[5]
		# minutes
		mins = mins + self.timeValue[4] + (otherTimecode.timeValue[4] * multiplier)
		hrs = mins / self.format[4]
		mins = mins % self.format[4]
		# hours
		hrs = hrs + self.timeValue[3] + (otherTimecode.timeValue[3] * multiplier)
		days = hrs / self.format[3]
		hrs = hrs % self.format[3]
		# days
		#print "self.timeValue:", self.timeValue 
		#print "self.timeValue[1]:", self.timeValue[1]
		# check for leap year effect
		if calendar.isleap(self.timeValue[0]) and self.timeValue[1] == 2:
			days = days + self.timeValue[2] + (otherTimecode.timeValue[2] * multiplier)
			months = days / 29
			days = days % 29
			if not days: days = 29
		else:
			days = days + self.timeValue[2] + (otherTimecode.timeValue[2] * multiplier)
			months = days / self.__class__.monthLengths[self.timeValue[1]]
			days = days % self.__class__.monthLengths[self.timeValue[1]]
			if not days: days = self.__class__.monthLengths[self.timeValue[1]]
		# months
		months = months + self.timeValue[1] + (otherTimecode.timeValue[1] * multiplier)
		years = months / self.format[1]
		months = months % self.format[1]
		if not months: months = self.format[1]
		# years
		years = years + self.timeValue[0] + (otherTimecode.timeValue[0] * multiplier)
		self.timeValue = [years, months, days, hrs, mins, secs]

	def addScalar(self, otherScalar, multiplier=1):
		"""
		Overrides method in parent class, L{Timecode}. Does nothing.
		"""
		pass
		
	def scalarValue(self):
		"""
		Overrides method in parent class, L{Timecode}. Returns 0.
		"""
		return 0


	def formatScalar(self, timeScalar):
		"""
		Overrides method in parent class, L{Timecode}. Returns L{CalendarTimecode.timeValue}.
		"""
		return self.timeValue
				
			
	
	def __scalariseFormat(self):
		"""
		Overrides method in parent class, L{Timecode}. Does nothing.
		"""
		pass
	
	def conformToFormat(self, timeValue=None, applyToSelf=True):
		"""
		Ensures the specified C{timeValue} is within the
		limits of the timecode's format.
		"""
		# adjust for calendar format !!!!!
		if not timeValue:
			timeValue = self.timeValue

		timeValueLength = len(timeValue)
		for idx in range(len(self.format)):
			if idx < timeValueLength:
				timeValue[idx] = timeValue[idx] % self.format[idx]
			else:
				timeValue.append(0)
		if applyToSelf:
			self.timeValue = timeValue
		return timeValue	
		
	def copy(self, otherTimecode):
		"""
		Copies properties of C{otherTimecode} into self.

		Only accepts timecodes with compatible format as self.
		"""
		if len(otherTimecode.timeValue) < len(self.format):
			return
		self.timeValue = []
		for part in otherTimecode.timeValue:
			self.timeValue.append(part)
		self.timeValue = self.conformToFormat(self.timeValue)
