# svs_tracking.workers.worker

#    Copyright (c) 2005 Simon Yuill.
#
#    This file is part of 'Social Versioning System' (SVS).
#
#    'Social Versioning System' is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation; either version 2 of the License, or
#    (at your option) any later version.
#
#    'Social Versioning System' is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with 'Social Versioning System'; if not, write to the Free Software
#    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

"""
Generic methods for worker classes.

Worker classes perform specific jobs within an SVS cluster, and are linked
to a client.

@author:	Simon Yuill
@copyright:	2005 Simon Yuill
@license:	GNU GPL version 2 or any later version
@contact:	simon@lipparosa.org
"""
# external imports
from time import time

# internal imports
from svs_core.commands.scriptcommands import CommandResult
from svs_core.utilities.constants import svs_const


class GenericWorker:
	def __init__(self):
		self.client = None
		self.running = False
		self.data = None
			
	def setClient(self, client):
		"""
		Sets client for worker.
		"""
		self.client = client
	
	def setDataSource(self, source):
		"""
		Sets source from which worker can draw data.
		"""
		self.source = source
	
	def getData(self):
		"""
		Returns data handled by worker.
		
		Note: if this is to be transferred over a network, the data 
		must conform to twisted's serialisation formats:
		U{http://twistedmatrix.com/documents/current/howto/pb-copyable}
		"""
		return self.data

	def update(self, timestamp):
		"""
		Updates time-based activity.
		"""
		if not self.running:
			return

	def start(self):
		"""
		Starts worker process.
		"""
		self.running = True

	def stop(self):
		"""
		Stops worker process.
		"""
		self.running = False
	
	#########################
	# PRIVATE SCRIPT COMMANDS
	#########################
	def cmdprivate_start(self, cmd):
		"""
		Starts worker process.
		"""
		self.start()
		return CommandResult(cmd, message="worker process started", status=svs_const.OK)
	
	def cmdprivate_stop(self, cmd):
		"""
		Stops worker process.
		"""
		self.stop()
		return CommandResult(cmd, message="worker process stopped", status=svs_const.OK)
	
	def cmdprivate_update(self, cmd):
		"""
		Manualy update worker process.
		"""
		timestamp = time()
		self.update(timestamp)
		return CommandResult(cmd, message="update [%f]" % timestamp, status=svs_const.OK)


class ListProducer(GenericWorker):
	"""
	Test class, produces a list of data.
	"""
	def __init__(self):
		GenericWorker.__init__(self)
		self.data = []

	def update(self, timestamp):
		if not self.running:
			return
		self.listData.append("item_%s_%d" % (len(self.data), timestamp))
		print "update:", self.data



class ListConsumer(GenericWorker):
	"""
	Test class, displays list of data.
	"""
	def update(self, timestamp):
		if not self.running:
			return
		self.updateListDisplay()

	def updateListDisplay(self):
		if self.client:
			self.client.getDataFrom(self.source, self.displayList)

	def displayList(self, listData):
		print "displayList:", listData

	


		
