# svs_core.commands.scriptcommands

#    Copyright (c) 2005 Simon Yuill.
#
#    This file is part of 'Social Versioning System' (SVS).
#
#    'Social Versioning System' is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation; either version 2 of the License, or
#    (at your option) any later version.
#
#    'Social Versioning System' is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with 'Social Versioning System'; if not, write to the Free Software
#    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

"""
Parses string-formatted commands, such as clients might receive.

@author:	Simon Yuill
@copyright:	Simon Yuill
@license:	GNU GPL version 2 or any later version
@contact:	simon@lipparosa.org
"""
# external imports
from shlex import shlex
from twisted.spread import pb

# internal imports
from svs_core.utilities.constants import svs_const


#########################
# FACTORY METHODS
#########################

def makeCommandResult(cmd=None, message=None, status=svs_const.OK, result=None, recipient=None):
	cmdResult = CommandResult()
	cmdResult.cmd = cmd
	cmdResult.message = message
	cmdResult.status = status
	cmdResult.result = result
	cmdResult.recipient = recipient
	return cmdResult


#########################
# COMMAND CLASSES
#########################
class Command(pb.Copyable, pb.RemoteCopy):
	def __init__(self):
		self.name = ''
		self.args = [] 
		self.sender = None
		self.recipient = None
		self.callback = None

	def setCopyableState(self, state):
		self.__dict__ = state

	def getStateToCopy(self):
		d = self.__dict__.copy()
		return d

# register with perspective broker
pb.setUnjellyableForClass(Command, Command)


class CommandResult(pb.Copyable, pb.RemoteCopy):
	def __init__(self):
		self.cmd = None
		self.message = None
		self.status = None
		self.result = None
		self.recipient = None

	def getStateToCopy(self):
		d = self.__dict__.copy()
		return d
	
	def setCopyableState(self, state):
		self.__dict__ = state


# register with perspective broker
pb.setUnjellyableForClass(CommandResult, CommandResult)


def copyCommand(otherCmd):
	cmd = Command()
	cmd.name = otherCmd.name
	cmd.args = otherCmd.args
	cmd.sender = otherCmd.sender
	cmd.recipient = otherCmd.recipient
	cmd.callback = otherCmd.callback
	return cmd

def copyCommandResult(otherResult):
	return makeCommandResult(otherResult.cmd,
			otherResult.message,
			otherResult.status,
			otherResult.result)



#########################
# PARSER
#########################
class CommandParser:
	def __init__(self):
		pass

	def parse(self, inputText, sender=None):
		inputText.lower()
		# make tokens
		tokenizer = shlex(inputText)
		tokens = []
		tok = tokenizer.get_token()
		while tok:
			tokens.append(tok)
			tok = tokenizer.get_token()
		if len(tokens) == 0:
			return None
		# check for recipient
		recipient = None
		cmd = Command()
		cmd.sender = sender
		if len(tokens) > 1 and tokens[1] == ">":
			recipient = tokens[0]
		# format for recipient
		if recipient:
			if len(tokens) < 3:
				return None
			cmd.recipient = recipient
			cmd.name = tokens[2]
			if len(tokens) > 3:
				self.args = tokens[3:]
			return cmd
		# format as local command
		cmd.name = tokens[0]
		cmd.args = tokens[1:]
		return cmd


class CommandHandler:
	def __init__(self, host):
		self.host = host
		self.commandlist = {}

	def handleCommand(self, command):
		"""
		Checks if host object responds to received command.
		Executes command if found, ignores if not.
		"""
		if command.args and command.args[0] == '?':return self.getCommandDoc(command)
		mth = None
		if command.sender is not self.host:
			mth = self.commandlist_public.get(command.name, None)
		else:
			mth = self.commandlist_private.get(command.name, None)
			if not mth:
				mth = self.commandlist_public.get(command.name, None)
		if not mth:
			return makeCommandResult(command, "command not found", svs_const.ERROR) 
		return mth(command)
		
	def getCommandDoc(self, command):
		"""
		Returns the doc string for the command.

		This can be used to print help info on a command
		in response to the '?' argument.
		"""
		doc = None
		if command.sender is not self.host:
			doc =  self.commandlist_public_doc.get(command.name, None)
		else:
			doc = self.commandlist_private_doc.get(command.name, None)
			if not doc:doc = self.commandlist_public_doc.get(command.name, None)
		if not doc:return makeCommandResult(command, "command not found", svs_const.ERROR)
		return makeCommandResult(command, message=doc, status=svs_const.OK)
		

	def createCommandList(self):
		"""
		Creates list of scriptable commands on host object.
		Scriptable commands start with "cmdprivate_" or "cmdpublic_".
		"""
		self.commandlist_private = {}
		self.commandlist_public = {}
		self.commandlist_private_doc = {}
		self.commandlist_public_doc = {}
		
		self._addCommandsForObject(self.host)
		
		if self.host.worker:self._addCommandsForObject(self.host.worker)
				
	
	def _addCommandsForObject(self, obj):
		"""
		Adds methods from specified object to command listings.
		
		Note: once a command has been added, it cannot be replaced 
		by one with the same name.
		"""
		for method in dir(obj):
			if callable(getattr(obj, method)):
				if method.startswith("cmdprivate_"):
					mthname = method.split("_", 1)[1]
					if self.commandlist_private.has_key(mthname):
						continue 
					self.commandlist_private[mthname] = getattr(obj, method)
					self.commandlist_private_doc[mthname] = getattr(obj, method).__doc__.strip()
				elif method.startswith("cmdpublic_"):
					mthname = method.split("_", 1)[1]
					if self.commandlist_public.has_key(mthname):
						continue
					self.commandlist_public[mthname] = getattr(obj, method)
					self.commandlist_public_doc[mthname] = getattr(obj, method).__doc__.strip()

	
	def getPublicCommandDoc(self):
		"""
		Returns the names of scriptable commands that can be called by other objects.
		"""
		return self.commandlist_public_doc

	def getPrivateCommandDoc(self):
		"""
		Returns the names of scriptable commands that can be called by the host object.
		"""
		return self.commandlist_private_doc
	
	def getAllCommandDoc(self):
		"""
		Returns the names of all scriptable commands as a 
		dictionary containing two lists, 'public' and 'private'.
		"""
		return {'owner':self.host.getName(),
			'public':self.commandlist_public_doc,
			'private':self.commandlist_private_doc}

