# svs_core.filehandling.dataproviders

#    Copyright (c) 2005 Simon Yuill.
#
#    This file is part of 'Social Versioning System' (SVS).
#
#    'Social Versioning System' is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation; either version 2 of the License, or
#    (at your option) any later version.
#
#    'Social Versioning System' is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with 'Social Versioning System'; if not, write to the Free Software
#    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

"""
Classes for retrieving data from different sources.

@author:	Simon Yuill
@copyright:	2005 Simon Yuill
@license:	GNU GPL version 2 or any later version
@contact:	simon@lipparosa.org
"""

# internal imports
from svs_core.utilities.lib import validateFilePath

class DataProvider:
	"""
	Base class for all data providers.
	"""
	def __init__(self, searchpaths=None, defaultResult=None):
		self.searchpaths = searchpaths
		self.defaultResult = defaultResult

	def getDataForLabel(self, dataRequest):
		"""
		Provides custom handling of data requests.

		This should be overridden by extending classes.
		"""
		if not dataRequest.label:return self.defaultResult
		paths = self.searchpaths.get(dataRequest.label, None)
		if not paths:return self.defaultResult
		for path in paths:
			result = self.getDataOnSearchpath(path, dataRequest.args)
			if result:return result
		return self.defaultResult

	def getDataOnSearchpath(self, path, requestInfo):
		"""
		Finds data matching C{requestInfo} on specified path.

		This method should be overridden by extending classes.

		@type requestInfo: array
		"""
		return None

	def addSearchpath(self, label, searchpath):
		"""
		Adds new searchpath.  Each searchpath is defined by a label which relates
		it to particular categories of data.
		"""
		if not self.searchpaths:self.searchpaths={}
		if not self.searchpaths.has_key(label):self.searchpaths[label] = []
		self.searchpaths[label].append(searchpath)


class FileDataProvider(DataProvider):
	"""
	Provides data from locally stored files.
	"""
	def __init__(self, searchpaths=None):
		DataProvider.__init__(self, searchpaths)

	def getDataOnSearchpath(self, path, requestInfo):
		"""
		Finds data matching C{requestInfo} on specified path.

		This method should be overridden by extending classes.

		@type requestInfo: array
		"""
		path = validateFilePath(path, False)
		filename = requestInfo[0]
		filedata = None
		try:
			filedata = read(filename, basepath=path)# !! check
		except:return None
		return filedata
		

class WikiDataProvider(DataProvider):
	"""
	Provides data from a wiki engine.

	This is tailored for use with the MoinMoin Wiki.
	"""
	def __init__(self, searchpaths=None):
		DataProvider.__init__(self, searchpaths)


class CVSDataProvider(DataProvider):
	"""
	Provides data from a CVS repository.
	"""
	def __init__(self, searchpaths=None):
		DataProvider.__init__(self, searchpaths)
	
