# svs_core.filehandling.sourcecode

#    Copyright (c) 2005 Simon Yuill.
#
#    This file is part of 'Social Versioning System' (SVS).
#
#    'Social Versioning System' is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation; either version 2 of the License, or
#    (at your option) any later version.
#
#    'Social Versioning System' is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with 'Social Versioning System'; if not, write to the Free Software
#    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

"""
Classes for retrieving and storing source code.

@author:	Simon Yuill
@copyright:	2005 Simon Yuill
@license:	GNU GPL version 2 or any later version
@contact:	simon@lipparosa.org
"""
# internal imports
from svs_core.filehandling.dataproviders import *
from svs_core.utilities.lib import validateFilePath


#############################
# EXCEPTIONS
#############################
class SourceRepositoryException(Exception):pass

#############################
# CLASSES
#############################
class SourceReference:
	"""
	Encapsualtes data for refering to a source code file.
	"""
	def __init__(self, name, url, moduleInstance = None):
		self.name = name
		self.url = url
		self.moduleInstance = moduleInstance



class SourceRepository:
	"""
	Provides a wrapper around access to source code files.

	This is currently setup to read files from a local directory.

	Later versions should support remote files, CVS and Wiki access
	through use of L{svs_core.filehandling.dataproviders} classes.
	"""
	def __init__(self): #, dataProvider):
		#self.dataProvider = dataProvider # implement later
		self.sourceRefs = {}

	def loadSourceFile(self, sourceName, url, createInstance=False, addToSystem=False):
		"""
		Loads source file and adds it to repository.

		Returns copy of source reference.

		@rtype: L{SourceReference}
		"""
		code = self.readSourceFile(url)
		if not code:raise SourceRepositoryException("Error loading '%s', unable to open file: %s" % (sourceName, url))
		moduleInstance = None
		if createInstance:
			moduleInstance = self.instantiateModule(sourceName, self.readSourceFile(url), addToSystem=addToSystem)
		sourceRef = SourceReference(sourceName, url, moduleInstance)
		self.addSourceReference(sourceRef)
		return sourceRef
 

	def addSourceReference(self, sourceReference):
		"""
		Adds references for a source file.
		"""
		self.sourceRefs[sourceReference.name] = sourceReference

	def getSourceCode(self, sourceName):
		"""
		Returns source code for specified source name.
		"""
		ref = self.sourceRefs.get(sourceName, None)
		if not ref:return None
		return self.readSourceFile(ref.url)

	def readSourceFile(self, filename):
		"""
		Reads and return source code from file.
		"""
		filename = validateFilePath(filename, False)
		f = open(filename, 'r')
		code = f.read()
		f.close()
		return code

	def getModuleForSource(self, sourceName, caching=True):
		"""
		Returns module instance for specified source name.
		"""
		ref = self.sourceRefs.get(sourceName, None)
		if not ref:return None
		if ref.moduleInstance != None:return ref.moduleInstance
		if caching:
			ref.moduleInstance = self.createModuleForSource(ref)
			return ref.moduleInstance
		else:return self.createModuleForSource(ref)

	def createModuleForSource(self, sourceName, addToSystem=False):
		"""
		Creates a module instance for the source reference.
		"""
		codeFile = self.getSourceCode(sourceName)
		if not codeFile:raise SourceRepositoryException(" Error creating module for '%s', unable to load source code." % sourceName)
		return self.instantiateModule(sourceName, codeFile, addToSystem=addToSystem)

	def instantiateModule(self, moduleName, moduleCode, addToSystem=False):
		"""
		Creates a module instance.
		"""
		import sys,imp
		module = imp.new_module(moduleName)
		exec moduleCode in module.__dict__
		if addToSystem:sys.modules[moduleName] = module
		return module
