# svs_core.network.packets

#    Copyright (c) 2005 Simon Yuill.
#
#    This file is part of 'Social Versioning System' (SVS).
#
#    'Social Versioning System' is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation; either version 2 of the License, or
#    (at your option) any later version.
#
#    'Social Versioning System' is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with 'Social Versioning System'; if not, write to the Free Software
#    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

"""
Encapsulators for information exchange between clients.

The objects provide network serialisation through the use of the
Twisted Perspective Broker system:

U{http://twistedmatrix.com}


@author:	Simon Yuill
@copyright:	2005 Simon Yuill
@license:	GNU GPL version 2 or any later version
@contact:	simon@lipparosa.org
"""
# external imports
from twisted.spread import pb
from time import time
from string import join

# internal imports
from svs_core.utilities.constants import svs_const


#########################
# EXCEPTIONS
#########################
class DataPacketException(Exception):pass
class ProfileRequestException(Exception):pass


#########################
# FACTORY METHODS
#########################
def makeDataPacket(sender, recipient=None, content=None, label=None, timeToLive=None):
	"""
	Factory method to create timestamped data packet.

	@rtype: L{DataPacket}
	"""
	packet = DataPacket()
	packet.created = time()
	packet.sender = sender
	packet.recipient = recipient
	packet.content = content
	packet.label = label
	packet.timeToLive = timeToLive
	return packet

def makeDataRequest(sender, recipient=None, label=None, args=[], retrievalAction=None):
	"""
	Factory method to produce timestamped data request.

	@rtype: L{DataRequest}
	"""
	request = DataRequest()
	request.created = time()
	request.sender = sender
	request.recipient = recipient
	request.label = label
	request.args = args
	request.retrievalAction = retrievalAction
	return request

def makeListenRequest(sender, recipient=None, label=None, action=None):
	"""
	Factory method to produce timestamped listener request.

	@rtype: L{ListenRequest}
	"""
	request = ListenRequest()
	request.created = time()
	request.sender = sender
	request.recipient = recipient
	request.label = label
	request.action = action
	return request

def makeProfileRequest(sender, clientNames=None, services=None, methods=None, commands=None):
	"""
	Factory method to produce timestamped profile request.

	At least one of the search parameters must be speicified, otherwise
	throws a L{ProfileRequestException}.

	@rtype: L{ListenRequest}
	@throws: L{ProfileRequestException}
	"""
	if not clientNames and not services and not methods and not commands:raise ProfileRequestException("no search parameters specified.")
	request = ProfileRequest()
	request.created = time()
	request.sender = sender
	request.clientNames = clientNames
	request.services = services
	request.methods = methods
	request.commands = commands
	return request


def makeMessagePacket(sender, messageInput):
	"""
	Factory method to create L{DataPacket} for chat message.
	
	@type messageInput: Array
	@raise L{DataPacketException}: if incorrect input.
	@rtype: L{DataPacket}
	"""
	if len(messageInput) < 2: raise DataPacketException("incorrect input for message packet")
	packet = DataPacket()
	packet.created = time()
	packet.sender = sender
	packet.recipient, packet.content = __parseMessageInput(messageInput)
	if len(packet.recipient) == 0: raise DataPacketException("no recipients for message packet")
	if len(packet.content) == 0: raise DataPacketException("no content for message packet")
	packet.label = svs_const.CHAT_MESSAGE
	return packet

def __parseMessageInput(messageInput):
	"""
	Extracts recipients  and message from C{messageInput}.

	@type messageInput: Array
	@rtype: Array
	"""
	recipients = []
	# check first token for 'all' or '*'
	if messageInput[0] == 'all' or messageInput[0] == '*':
		msgText = __messageInputString(messageInput[1:])
	 	return (svs_const.RECIPIENTS_ALL, msgText)
	# check for list of recipients - separated by comma
	tokenIdx = 0
	readingRecipients = True
	while(tokenIdx < len(messageInput)):
		if messageInput[tokenIdx + 1] == ',':
			recipients.append(messageInput[tokenIdx])
			tokenIdx += 2
		else:
			recipients.append(messageInput[tokenIdx])
			tokenIdx += 1
			break
	msgText = __messageInputString(messageInput[tokenIdx:])
	return (recipients, msgText)
		

def __messageInputString(messageInput):
	"""
	Convert message back into string.
	
	@rtype: String
	"""
	return join(messageInput)
	
		
	


#########################
# CLASSES
#########################
class DataPacket(pb.Copyable, pb.RemoteCopy):
	"""
	Generic data packet.
	"""
	def __init__(self):
		self.created = None
		self.sender = None
		self.recipient = None
		self.content = None
		self.label = None
		self.timeToLive = None
	
	def setCopyableState(self, state):
		self.__dict__ = state

	def getStateToCopy(self):
		d = self.__dict__.copy()
		return d
	
# register with perspective broker
pb.setUnjellyableForClass(DataPacket, DataPacket)



class DataRequest(pb.Copyable, pb.RemoteCopy):
	"""
	Encapsulates a request for data.
	"""
	def __init__(self):
		self.created = None
		self.sender = None
		self.recipient = None
		self.label = None
		self.args = None
		self.retrievalAction = None
	
	def setCopyableState(self, state):
		self.__dict__ = state

	def getStateToCopy(self):
		d = self.__dict__.copy()
		return d

# register with perspective broker
pb.setUnjellyableForClass(DataRequest, DataRequest)


class ListenRequest(pb.Copyable, pb.RemoteCopy):
	"""
	Encapsulates a request for one client to listen to another.
	"""
	def __init__(self):
		self.created = None
		self.sender = None
		self.recipient = None
		self.action = None
		self.label = None
	
	def setCopyableState(self, state):
		self.__dict__ = state

	def getStateToCopy(self):
		d = self.__dict__.copy()
		return d

# register with perspective broker
pb.setUnjellyableForClass(ListenRequest, ListenRequest)


class ProfileRequest(pb.Copyable, pb.RemoteCopy):
	"""
	Encapsulates a request for searching client profiles.
	"""
	def __init__(self):
		self.created = None
		self.sender = None
		self.recipient = None
		self.action = None
		self.label = None
	
	def setCopyableState(self, state):
		self.__dict__ = state

	def getStateToCopy(self):
		d = self.__dict__.copy()
		return d

# register with perspective broker
pb.setUnjellyableForClass(ListenRequest, ListenRequest)

