# svs_core.time.timeline

#    Copyright (c) 2005 Simon Yuill.
#
#    This file is part of 'Social Versioning System' (SVS).
#
#    'Social Versioning System' is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation; either version 2 of the License, or
#    (at your option) any later version.
#
#    'Social Versioning System' is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with 'Social Versioning System'; if not, write to the Free Software
#    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

"""
Provides temporal representation and management of visualisation.


@author:	Simon Yuill
@copyright:	2005 Simon Yuill
@license:	GNU GPL version 2 or any later version
@contact:	simon@lipparosa.org
"""

# external imports
from time import time
import math
from datetime import datetime, timedelta



#########################
# EXCEPTIONS
#########################
class TimecodeException(Exception):
	"""
	Generic exception class for handling L{Timecode}.
	"""
	pass

class TimelineException(Exception):
	"""
	Generic exception class for handling L{Timeline}.
	"""
	pass

class TimeFrameException(Exception):
	"""
	Generic exception class for handling L{TimeFrame}.
	"""
	pass


#########################
# FACTORY METHODS
#########################
def timecodeFromDateString(dateStr):
	"""
	Factory method to create L{Timecode} object by converting CVS log date entry into visualisation timecode object.
	"""
	#print "dateStringToTimecode:", dateStr
	# parse date string
	# works for date format: "y-m-d h:m:s"
	timeList = dateStr.split()[0].split('-') + dateStr.split()[1].split(':')
	for idx in range(len(timeList)):
		timeList[idx] = int(timeList[idx])

	return timecodeFromList(timeList)

def timecodeFromList(timeList):
	"""
	Factory method to create L{Timecode} object from list representing time units.
	"""
	#print "timecodeFromList:", timeList
	timeListLength = len(timeList)
	if timeListLength < 1: raise TimecodeException("Insufficient values to create timecode: %s" % timeList)
	if timeListLength < 6:
		# pad list to correct length
		for i in range(6 - timeListLength):
			timeList.insert(0, 0)
	elif timeListLength > 6:
		# strip list to correct length
		timeList[0:timeListLength - 6] = []

	return Timecode(timeList[0], timeList[1], timeList[2], 
			timeList[3], timeList[4], timeList[5])



def makeTimeFrame(timecode, label=None):
	"""
	Factory method to produce L{TimeFrame} object.
	"""
	frame = TimeFrame()
	frame.timecode = timecode
	frame.label = label
	return frame


#########################
# UTILITY METHODS
#########################
def mergeTimeFrames(frame1, frame2):
	"""
	Combine events in two different L{TimeFrame} objects, 
	returning C{frame1} with the new data.  If C{frame2}
	is None, returns C{frame1} unaltered.

	@raise TimeFrameException: if C{frame1} is None
	"""
	#print "mergeTimeFrames: %s, %s" % (frame1, frame2)
	if not frame1:
		raise TimeFrameException("first merge frame not specified")
	if not frame2:
		return frame1
	
	### CHANGE!!!!
	for item in frame2.events.items():
		if frame1.events.has_key(item[0]):
			frame1.events[item[0]] += item[1]
		else:
			frame1.events[item[0]] = item[1]
	return frame1



#########################
# TIMECODE
#########################
class Timecode(datetime):
	"""
	Represents a timecode reference for a L{Frame} in the L{Timeline}.
	"""
	pass

class Timecode_OLD:
	"""
	Represents a timecode reference for a L{Frame} in the L{Timeline}.
	"""
	def __init__(self, year, month, day, hour, minutes, seconds):
		self.timeValue = datetime(year, month, day, hour, minutes, seconds)

	def __str__(self):
		"""
		Displays timecode as string in format: "00:00:00".
		"""
		return "[Timecode %s]" % self.timeValue

	def add(self, timeDelta):
		"""
		Increases the current timecode values by the amounts
		specified in C{otherTimecode}.
		"""
		print "Timecode add:", timeDelta
		self.timeValue = self.timeValue + timeDelta
		return self.timeValue
	

#########################
# TIMELINE
#########################
class Timeline:
	"""
	Holds time-based representation of CVS project, enabling it to be
	accessed at different points of time.
	"""
	def __init__(self):
		self.label = None
		self.timeline = {}

	def addFrame(self, frame, timecode=None, merge=False):
		"""
		Adds frame into timeline at specified index.

		The position of the frame is determined by its own timecode, or
		one that is specified with the command.  If a timecode is
		given, this replaces the original timecode of the frame.
	
		@raise TimelineException: If no timecode is specified nor 
		defined by the frame itself.
		"""
		if not timecode and not frame.timecode:
			raise TimelineException("no timecode given to insert frame")
		if timecode:
			frame.timecode = timecode
		
		if frame.timecode not in self.timeline:
			self.timeline[frame.timecode] = frame
		else:
			self.timeline[frame.timecode] = mergeTimeFrames(self.timeline[frame.timecode], frame)
			


	def getFrame(self, timecode, defaultReturn=None):
		"""
		Returns frame at specified timecode on node.
		
		If no frame is found at specified timecode, 
	        returns None or the specified C{defaultReturn} object.
	        
	        @type 	timecode: Timecode
	        @param 	timecode: timecode to search for frame
	        @type 	defaultReturn: object
	        @param 	defaultReturn: returned if nothing found
	        @rtype:	TimeFrame
	        @return: frame at specified path timecode
		"""
		return self.timeline.get(timecode, defaultReturn)
		

	def addEvent(self, newEvent):
		"""
		Adds a new event into the timeline.
		"""
		print "addEvent, newEvent:", newEvent.timecode
		frame = self.getFrame(newEvent.timecode)
		if not frame:
			frame = makeTimeFrame(newEvent.timecode)
			self.addFrame(frame)
		frame.addEvent(newEvent)



#########################
# TIMEFRAME
#########################
class TimeFrame:
	"""
	Holds data for an individual frame within a node.
	"""
	def __init__(self):
		self.timecode = None
		self.label = None
		self.events = {}

	def encode(self):
		"""
		Returns encoded representation of self.
		"""
		return {}

	def decode(self, data):
		"""
		Applies encoded data to self.
		"""
		pass
	
	def addEvent(self, newevent):
		"""
		Adds an event to this frame.
		"""
		eventType = "%s" % newevent.__class__
		if self.events.has_key(eventType):
			if newevent not in self.events[eventType]:
				self.events[eventType].append(newevent)
		else:
			self.events[eventType] = []
			self.events[eventType].append(newevent)

	def display(self, displayContext=None):
		if not self.timecode:
			print "I am a blank frame"
		else:
			print "frame [%s]\n" % self.timecode
			for key, value in self.events.items():
				print "event type: %s\n" % key
				for entry in value:
					entry.display()


