# svs_core.utilities.lib

#    Copyright (c) 2005 Simon Yuill.
#
#    This file is part of 'Social Versioning System' (SVS).
#
#    'Social Versioning System' is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation; either version 2 of the License, or
#    (at your option) any later version.
#
#    'Social Versioning System' is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with 'Social Versioning System'; if not, write to the Free Software
#    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

"""
Library of helper classes for general purpose use.

@author:	Simon Yuill
@copyright:	2005 Simon Yuill
@license:	GNU GPL version 2 or any later version
@contact:	simon@lipparosa.org
"""

# external imports
import types

#######################
# FUNCTIONS
#######################
def parseNumericList(listStr, listSep):
	"""
	Converts comma separated string into a list of numbers.
	
	@type 	listStr: string
	@param 	listStr: string to parse
	@rtype:	list
	@return: list of floats
	"""
	import string
	
	strList = string.split(listStr, listSep)
	numList = []
		
	for str in strList:
		try:
			n = float(str)
		except ValueError:
			continue
		numList.append(n)
			
	return numList

def validateFilePath(filepath, create=True):
	"""
	Converts C{filepath} to absolute form, expanding the user if necessary,
	and tests if it exists.  If the directory, or any sub-directory in the path
	does not exist, can optionally create it (on by default).

	Returns validated path.  If C{create} is set to false, and the path does not
	exist, returns C{None}.
	"""
	import os
	import os.path
	absfilepath = os.path.abspath(os.path.expanduser(filepath))
	if filepath[-1] == '/':absfilepath = absfilepath + '/'
	dirpath = os.path.dirname(absfilepath)
	if os.path.exists(dirpath):return absfilepath
	if create:
		os.makedirs(dirpath)
		return absfilepath
	else:return None


def isEncodable(obj):
	"""
	Tests whether an object has an C{encode} methods,
	and returns C{True} or C{False}.
	"""
	try:func = getattr(obj, 'encode')
	except:return False
	if not func:return False
	if not callable(func):return False
	return True

def isDecodable(obj):
	"""
	Tests whether an object has an C{decode} methods,
	and returns C{True} or C{False}.
	"""
	try:func = getattr(obj, 'decode')
	except:return False
	if not func:return False
	if not callable(func):return False
	return True
		
def encodeValue(value):
	"""
	Returns encodable version of value.
	"""
	if isEncodable(value):return value.encode()
	if type(value) is int:return value
	if type(value) is float:return value
	if type(value) is str:return value
	if type(value) is dict:return value
	if type(value) is list:return value
	if type(value) is tuple:return value
	return None # should be error


	
	
#######################
# CLASSES
#######################
class Bunch:
	"""
	Lightweight data container class.

	Allows data to be stored as attributes of an object when created:

	C{data = Bunch(stuff='foo', morestuff='bar')}

	Originally written by Alex Martelli: U{http://aspn.activestate.com/ASPN/Cookbook/Python/Recipe/52308}
	"""
	def __init__(self, **kwds):
		self.__dict__.update(kwds)

class Constants:
	"""
	Provides support for fixed value constants.  It is derived from
	Alex Martelli's recipe in the Python Cookbook:
	
	U{http://aspn.activestate.com/ASPN/Cookbook/Python/Recipe/65207}
	"""

	class ConstantsError(TypeError):pass
	
	def __setattr__(self, name, value):
		if self.__dict__.has_key(name):
			raise self.ConstError, "Cannot rebind constant(%s)"%name
		self.__dict__[name] = value
		
	def __delattr__(self, name):
		if self.__dict__.has_key(name):
			raise self.ConstError, "Cannot unbind constant(%s)"%name
		raise NameError, name


class EnumException(Exception):pass

class Enumeration:
	"""
	Emulates the C C{Enum} function, providing a set of unique values.

	This is taken from a recipe by Will Ware:
	U{http://aspn.activestate.com/ASPN/Cookbook/Python/Recipe/67107}.
	"""
	def __init__(self, name, enumList):
		self.__doc__ = name
		lookup = { }
		reverseLookup = { }
		i = 0
		uniqueNames = [ ]
		uniqueValues = [ ]
		for x in enumList:
			if type(x) == types.TupleType:
				x, i = x
			if type(x) != types.StringType:
				raise EnumException, "enum name is not a string: " + x
			if type(i) != types.IntType:
				raise EnumException, "enum value is not an integer: " + i
			if x in uniqueNames:
				raise EnumException, "enum name is not unique: " + x
			if i in uniqueValues:
				raise EnumException, "enum value is not unique for " + x
			uniqueNames.append(x)
			uniqueValues.append(i)
			lookup[x] = i
			reverseLookup[i] = x
			i = i + 1
		self.lookup = lookup
		self.reverseLookup = reverseLookup
	def __getattr__(self, attr):
		if not self.lookup.has_key(attr):
			raise AttributeError
		return self.lookup[attr]
	def whatis(self, value):
		return self.reverseLookup[value]
	


class PathBasedCollection:
	"""
	Holds data in a nested structure that can be accessed via pathnames.
	
	The pathnames are defined as list objects which represent a series of nodes.
	"""
	dataKey = "__data"
	
	def __init__(self):
		self.collection = {}
		
	def __str__(self):
		return self.collection.__str__()
		
	def addEntry(self, path, data):
		"""
		Adds entry to collection.
		
		@type 	path: list
		@param 	path: nodes address for data
		@type 	data: object
		@param 	data: data to be stored
		"""
		node = self.collection
		
		for unit in path:
			if not node.has_key(unit):
				node[unit] = {}
				
			node = node[unit]
			
		if not node.has_key(PathBasedCollection.dataKey):
			node[PathBasedCollection.dataKey] = []
			
		node[PathBasedCollection.dataKey].append(data)
		

			
	def get(self, path, defaultReturn=[]):
		"""
		Checks for data at specified path 
		and returns as list if found.
		
		If no data is found at specified path, 
		returns an empty list or the specified
		C{defaulrReturn} object.
		
		@type 	path: list
		@param 	path: path to serach for data
		@type 	defaultReturn: object
		@param 	defaultReturn: returne if nothing found
		@rtype:	list
		@return: data at specified path location
		"""
		node = self.collection
		
		for unit in path:
			nextNode = node.get(unit, None)
			if nextNode:
				node = nextNode
			else:
				return defaultReturn
				
		return node.get(PathBasedCollection.dataKey, defaultReturn)
		
		

class GenericPath:
	"""
	Provides a general purpose path-style locator for data.
	
	Note: implement slicing methods.
	"""
	def __init__(self, *args):
		self.nodes = args
		self.delim = '/'
		
	def __str__(self):
		import join
		return join(self.nodes, self.delim)
		


