# svs_demogame.analysis

#    Copyright (c) 2005 Simon Yuill.
#
#    This file is part of 'Social Versioning System' (SVS).
#
#    'Social Versioning System' is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation; either version 2 of the License, or
#    (at your option) any later version.
#
#    'Social Versioning System' is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with 'Social Versioning System'; if not, write to the Free Software
#    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

"""
Analysis of changes and game play.

@author:	Simon Yuill
@copyright:	2005 Simon Yuill
@license:	GNU GPL version 2 or any later version
@contact:	simon@lipparosa.org
"""
# external imports
from difflib import Differ

# internal imports
from svs_demogame.utils import demo_const
from svs_demogame.players import Player
from svs_core.utilities.notification import Listenable
from svs_core.utilities.constants import svs_const
from svs_core.network.packets import makeDataPacket



class ViewableDataSource(Listenable):
	"""
	Generic class which can provide data to a view component.
	"""
	def __init__(self):
		Listenable.__init__(self)

	def addView(self, view):
		"""
		Adds view to data source.

		This is added as a listener with the default data label of L{svs_const.DATA_UPDATED}.
		"""
		self.addListener(view, listenFor=svs_const.DATA_UPDATED)

	def removeView(self, view):
		"""
		Removes view from data source.
		"""
		self.removeListener(view, listenFor=svs_const.DATA_UPDATED)

	def updateViews(self, displayData=None):
		"""
		Notifies views that the source has been altered and needs redisplayed.
		"""
		self.notifyListeners(svs_const.DATA_UPDATED, makeDataPacket(self, content=displayData, label=svs_const.DATA_UPDATED))

	
#############################
# SCRIPT HISTORY
#############################
class ScriptHistory(ViewableDataSource):
	"""
	Provides analysis data about script changes.
	"""
	def __init__(self):
		ViewableDataSource.__init__(self)
		self.scripts = {}
		self.scriptsOrderedKeys = []

	def addScript(self, script):
		"""
		Adds new script to analysis, checks script ID to see if previous version exists.
		"""
		if not self.scripts.has_key(script.scriptID):
			self.scripts[script.scriptID] = []
			self.scriptsOrderedKeys.append(script.scriptID)
		if len(self.scripts[script.scriptID]) == 0:
			self.scripts[script.scriptID].append(ScriptRevision(script))
		else:
			revision = ScriptRevision(script)
			revision.diff(self.scripts[script.scriptID][-1:][0])
			self.scripts[script.scriptID].append(revision)
		self.updateViews((self.scripts, self.scriptsOrderedKeys))



class ScriptRevision:
	"""
	Encapsulates a script with revision data.
	"""
	def __init__(self, script):
		self.script = script
		self.diffValue = 1.0

	def diff(self, otherRevision):
		"""
		Calculates difference value from specified script.
		"""
		d = Differ()
		result = d.compare(otherRevision.script.code, self.script.code)
		diffAdd = 0
		diffSubtract = 0
		diffNone = 0
		for entry in result:
			if entry.startswith('+'):diffAdd += 1
			elif entry.startswith('-'):diffSubtract += 1
			else:diffNone += 1
		diffTotal = diffAdd + diffSubtract + diffNone * 1.0
		if diffTotal < 1: self.diffValue = 0.0
		else:self.diffValue = (diffAdd + diffSubtract) / diffTotal

	def getRevisionTime(self):
		"""
		Returns time at which revision was created.
		"""
		return self.script.revisionTime

#############################
# MESSAGE HISTORY
#############################
class MessageHistory(ViewableDataSource):
	"""
	Provides analysis data about chat messages.

	Messages are ordered into sets according to the sender.
	"""
	def __init__(self):
		ViewableDataSource.__init__(self)
		self.messages = {}
		self.messagesOrderedKeys = []

	def addMessageClient(self, clientName):
		"""
		Adds new message client to history.

		A message client  can be either a sender or recipient of messages.
		"""
		if not self.messages.has_key(clientName):
			self.messages[clientName] = []
			self.messagesOrderedKeys.append(clientName)

	def addMessage(self, message):
		"""
		Adds new message to history.
		"""
		self.addMessageClient(message.sender)
		self.messages[message.sender].append(message)
		if message.recipient != svs_const.RECIPIENTS_ALL:
			for recClient in message.recipient:self.addMessageClient(recClient)
		self.updateViews((self.messages, self.messagesOrderedKeys))

		
		
		
#############################
# INTERACTION NETWORK
#############################
class InteractionNetwork(ViewableDataSource):
	"""
	Provides analysis of interaction between players, scripts and agents.
	"""
	def __init__(self):
		ViewableDataSource.__init__(self)
		self.nodes = {}
		self.nodesOrderedKeys = []
	
	def addPlayer(self, player, updateView=True):
		"""
		Adds player to network.
		"""
		node = self.addNode(NetworkNode(player.getName(), player))
		if updateView:
			self.updateViews((self.nodes, self.nodesOrderedKeys))
		return node 

	def addAgent(self, agent, updateView=True):
		"""
		Adds agent to network.
		"""
		node = self.addNode(NetworkNode(agent.getName(), agent))
		if updateView:
			self.updateViews((self.nodes, self.nodesOrderedKeys))
		return node

	def addScript(self, script, updateView=True):
		"""
		Adds script to network.
		"""
		scriptNode = self.addNode(NetworkNode(script.scriptID, script))
		scriptNode.linkTo(self.addPlayer(Player(script.author), False))
		if updateView:
			self.updateViews((self.nodes, self.nodesOrderedKeys))
		return scriptNode
		
	def addNode(self, node):
		"""
		Adds new node to network, ensuring that it does not already exist.
		"""
		if self.nodes.has_key(node.label):return self.nodes[node.label]
		self.nodes[node.label] = node
		self.nodesOrderedKeys.append(node.label)
		return node


class NetworkNode:
	"""
	Represents node within an L{InteractionNetwork} object.
	"""
	def __init__(self, label, dataSource=None):
		self.label = label
		self.dataSource = dataSource
		self.edges = {}

	def linkTo(self, node):
		"""
		Creates link to specified node.
		"""
		if not self.edges.has_key(node.label):
			self.edges[node.label] = NetworkEdge(node)
		self.edges[node.label].incrementLinkage()

	def getDataSourceType(self):
		"""
		Returns identifier for class type of data source.
		"""
		#if self.dataSource.__class__.__name__ == 'Script':
			#return "Script:%s" % self.dataSource.hostObject.__class__.__name__
		return self.dataSource.__class__.__name__


class NetworkEdge:
	"""
	Represents a connection between two L{NetworkNode}s.
	"""
	def __init__(self, node):
		self.endNode = node
		self.linkage = 0

	def incrementLinkage(self):
		"""
		Increases linkage value.
		"""
		self.linkage += 1

	def decrementLinkage(self):
		"""
		Decreases linkage value.
		"""
		if self.linkage > 0:self.linkage -= 1

	def getLabel(self):
		"""
		Returns label of end node for this edge.
		"""
		return self.endNode.label

#############################
# SCRIPT NETWORK
#############################
class ScriptNetwork(InteractionNetwork):
	"""
	Provides analysis of interaction between players and scripts.
	"""
	def __init__(self):
		InteractionNetwork.__init__(self)
	
	def addScript(self, script, updateView=True):
		"""
		Adds script to network.
		"""
		scriptNode = self.addNode(NetworkNode(script.scriptID, script))
		scriptNode.linkTo(self.addPlayer(Player(script.author), False))
		if updateView:
			self.updateViews((self.nodes, self.nodesOrderedKeys))
		return scriptNode


#############################
# MESSAGE NETWORK
#############################
class MessageNetwork(InteractionNetwork):
	"""
	Provides analysis of interaction between players through chat messages.
	"""
	def __init__(self):
		InteractionNetwork.__init__(self)
	
	def addMessage(self, message, updateView=True):
		"""
		Adds script to network.
		"""
		messageNode = self.addPlayer(Player(message.sender), False)
		if message.recipient == svs_const.RECIPIENTS_ALL:recipients = self.nodesOrderedKeys
		else:recipients = message.recipient
		for recipient in recipients:
			messageNode.linkTo(self.addPlayer(Player(recipient), False))
		if updateView:
			self.updateViews((self.nodes, self.nodesOrderedKeys))
		return messageNode
