# svs_demogame.clients

#    Copyright (c) 2005 Simon Yuill.
#
#    This file is part of 'Social Versioning System' (SVS).
#
#    'Social Versioning System' is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation; either version 2 of the License, or
#    (at your option) any later version.
#
#    'Social Versioning System' is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with 'Social Versioning System'; if not, write to the Free Software
#    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

"""
Clients for demo game.

@author:	Simon Yuill
@copyright:	2005 Simon Yuill
@license:	GNU GPL version 2 or any later version
@contact:	simon@lipparosa.org
"""
# external imports
from twisted.spread import pb
from twisted.internet import reactor
from twisted.cred import credentials
from os.path import expanduser

# internal imports
from svs_demogame.gui import *
from svs_demogame.gameworld import Gameworld
from svs_demogame.scripts import script_const, makeScriptFromCode, setScriptEnvironmentProperty, loadGameCode, loadTemplateCode
from svs_demogame.utils import demo_const
from svs_core.commands.scriptcommands import *
from svs_core.network.clientuser import ScriptableClient
from svs_core.network.packets import *
from svs_core.utilities.constants import svs_const




#############################
# FACTORY METHODS
#############################
def createClient(name, password, clientType=demo_const.CODE_PLAYER_CLIENT):
	"""
	Factory method to properly create client.

	NOTE: currently deprecated, may be re-introduced at later stage.
	
	@rtype:		L{svs.network.clientuser.GenericClient}
	@returns:	subclass of L{svs.network.clientuser.GenericClient}
	"""
	from svs_demogame.tracking_clients import TrackingClient
	if clientType == demo_const.CODE_PLAYER_CLIENT:
		client = CodePlayerClient(name, password)
	elif clientType == demo_const.GAME_WORLD_CLIENT:
		client = GameWorldClient(name, password)
	elif clientType == demo_const.TRACKING_CLIENT:
		client = TrackingClient(name, password)
	else:return None
	return client


#############################
# PLAYER CLIENT
#############################
class PlayerClient(ScriptableClient):
	"""
	Generic player client.
	"""
	def __init__(self, name, passwd):
		ScriptableClient.__init__(self, name, passwd)
		self.gameWorld = None

	def setGameWorld(self, gameWorld):
		"""
		Sets name of gameworld that client should join.
		"""
		self.gameWorld = gameWorld

	#######################
	# NETWORK
	#######################
	def avatarResult_joinClusterGroup(self, group):
		"""
		Handles result of L{joinClusterGroup} action.
		"""
		self.clusterGroup = group
		self.logMessage("joined cluster group <%s>" % self.clusterGroupName)
		self.getGameWorldModel()

	def getGameWorldModel(self):
		"""
		Retrieves model of game from gameworld client.
		"""
		if not self.gameWorld:return
		dataRequest = makeDataRequest(self, recipient=self.gameWorld, label=demo_const.WORLD_MODEL_LABEL)
		self.getData(dataRequest)
		self.startListeningTo(self.gameWorld, listenFor=demo_const.WORLD_UPDATE_LABEL)

	#######################
	# DATA HANDLING
	#######################
	def handleDataPacket(self, dataPacket):
		"""
		Handles data packet received from network.

		This should be overidden by extending classes.
		""" 
		if not dataPacket:return
		dataLabel = dataPacket.label
		if not dataLabel:return
		# world model
		if dataLabel == demo_const.WORLD_MODEL_LABEL:
			#self.setupGameWorld(dataPacket.content)
			pass
		elif dataLabel == demo_const.WORLD_UPDATE_LABEL:
			#self.updateGameWorld(dataPacket.content)
			pass
		else:self.logMessage("data packet received: <%s>" % dataPacket.content)

	#########################
	# REMOTE METHODS
	#########################
	def remote_update(self, timeElapsed):
		"""
		Handles update sent from client proxy on server.

		This should be overidden by extending classes.
		
		@type 	time: list
		@param 	time: current simulation time
		"""
		#self.gui.update(timeElapsed)
		pass


#############################
# GUI PLAYER CLIENT
#############################
class GUIPlayerClient(PlayerClient):
	"""
	Generic player client with GUI.
	"""
	def __init__(self, name, passwd):
		PlayerClient.__init__(self, name, passwd)
		self.openGUI()

	def openGUI(self):
		self.gui = CodePlayerGUI(self)
		self.gui.build()


	#######################
	# DATA HANDLING
	#######################
	def handleDataPacket(self, dataPacket):
		"""
		Handles data packet received from network.
		""" 
		if not dataPacket:return
		dataLabel = dataPacket.label
		if not dataLabel:return
		# world model
		if dataLabel == demo_const.WORLD_MODEL_LABEL:
			self.gui.setupGameWorld(dataPacket.content)
		elif dataLabel == demo_const.WORLD_UPDATE_LABEL:
			self.gui.updateGameWorld(dataPacket.content)
		else:self.logMessage("data packet received: <%s>" % dataPacket.content)

	#######################
	# UTILITY
	#######################
	def statusMessage(self, text):
		"""
		Handles status messages for client.  This should be overridden
		by implementing classes.
		
		@type 	text: string
		@param 	text: message
		"""
		self.gui.statusMessage(text)

	def logMessage(self, text):
		"""
		Handles log messages for client. Log messages can be turned on and off.

		This should be overridden by implementing classes.
		
		@type 	text: string
		@param 	text: message
		"""
		if self.logging: self.gui.statusMessage(text)
		
	
	def errorMessage(self, text):
		"""
		Handles error messages for client.  This should be overridden
		by implementing classes.
		
		@type 	text: string
		@param 	text: message
		"""
		self.gui.errorMessage(text)

	#########################
	# REMOTE METHODS
	#########################
	def remote_update(self, timeElapsed):
		"""
		Handles update sent from client proxy on server.
		
		@type 	time: list
		@param 	time: current simulation time
		"""
		self.gui.update(timeElapsed)

	#########################
	# PRIVATE SCRIPT COMMANDS
	#########################
	def cmdprivate_quit(self, cmd):
		"""
		Disconnect from server and quit client application.
		"""
		self.gui.destroy()
		return makeCommandResult(cmd, message="quitting", status=svs_const.OK)

	def cmdprivate_clearlog(self, cmd):
		"""
		Clear console log display.
		"""
		self.gui.clearLogDisplay()
		return makeCommandResult(cmd, status=svs_const.OK)

	def cmdprivate_clearcode(self, cmd):
		"""
		Clear script editor display.
		"""
		self.gui.clearScriptArea()
		return makeCommandResult(cmd, status=svs_const.OK)

	def cmdprivate_echo(self, cmd):
		"""
		Turns echo of input in console on and off.
		"""
		try:
			state = cmd.args[0]
		except IndexError:
			state = 'on'
		if state == 'off':
			self.gui.setEchoInput(False)
		else:
			self.gui.setEchoInput(True)
		return makeCommandResult(cmd, status=svs_const.OK)


#############################
# CODE PLAYER CLIENT
#############################
class CodePlayerClient(GUIPlayerClient):
	"""
	Player who can work with scripting code.
	"""
	def __init__(self, name, passwd):
		GUIPlayerClient.__init__(self, name, passwd)

	def openGUI(self):
		self.gui = CodePlayerGUI(self)
		self.gui.build()
		
	#######################
	# AREA REQUESTS
	#######################
	def getScriptForArea(self, areaId, revNum=-1):
		"""
		Retrieves script for specified terrain area.
		"""
		if not self.gameWorld:return
		dataRequest = makeDataRequest(self, recipient=self.gameWorld, label=demo_const.AREA_SCRIPT_LABEL, args={demo_const.AREA_ID_LABEL:areaId, demo_const.REVISION_NUMBER_LABEL:revNum})
		self.getData(dataRequest)

	def getRevisionLogForArea(self, areaId):
		"""
		Retrieves revision log for specified terrain area.
		"""
		if not self.gameWorld:return
		dataRequest = makeDataRequest(self, recipient=self.gameWorld, label=demo_const.AREA_REVISION_LOG_LABEL, args={demo_const.AREA_ID_LABEL:areaId})
		self.getData(dataRequest)

	def sendScriptForArea(self, areaId, script):
		"""
		Sends new script to gameworld for specified terrain area.
		"""
		if not self.gameWorld:return
		dataRequest = makeDataRequest(self, recipient=self.gameWorld, label=demo_const.NEW_AREA_SCRIPT_LABEL, args={demo_const.AREA_ID_LABEL:areaId, demo_const.SCRIPT_CODE_LABEL:script})
		self.logMessage("sending script...")
		self.getData(dataRequest)

	#######################
	# AGENT REQUESTS
	#######################
	def getScriptForAgent(self, agentIdNum, revNum=-1):
		"""
		Retrieves script for specified agent.
		"""
		if not self.gameWorld:return
		dataRequest = makeDataRequest(self, recipient=self.gameWorld, label=demo_const.AGENT_SCRIPT_LABEL, args={demo_const.AGENT_ID_LABEL:agentIdNum, demo_const.REVISION_NUMBER_LABEL:revNum})
		self.getData(dataRequest)

	def getRevisionLogForAgent(self, agentIdNum):
		"""
		Retrieves revision log for specified agent.
		"""
		if not self.gameWorld:return
		dataRequest = makeDataRequest(self, recipient=self.gameWorld, label=demo_const.AGENT_REVISION_LOG_LABEL, args={demo_const.AGENT_ID_LABEL:agentIdNum})
		self.getData(dataRequest)

	def sendScriptForAgent(self, agentIdNum, script):
		"""
		Sends new script to gameworld for specified agent.
		"""
		if not self.gameWorld:return
		dataRequest = makeDataRequest(self, recipient=self.gameWorld, label=demo_const.NEW_AGENT_SCRIPT_LABEL, args={demo_const.AGENT_ID_LABEL:agentIdNum, demo_const.SCRIPT_CODE_LABEL:script})
		self.logMessage("sending script...")
		self.getData(dataRequest)
		
	#######################
	# DATA HANDLING
	#######################
	def handleDataPacket(self, dataPacket):
		"""
		Handles data packet received from network.
		""" 
		if not dataPacket:return
		dataLabel = dataPacket.label
		if not dataLabel:return
		# area scripts
		elif dataLabel == demo_const.AREA_SCRIPT_LABEL:
			self.gui.setScriptAreaText(dataPacket.content)
		elif dataLabel == demo_const.AREA_REVISION_LOG_LABEL:
			self.gui.displayRevisionLog(dataPacket.content)
		# agent scripts
		elif dataLabel == demo_const.AGENT_SCRIPT_LABEL:
			self.gui.setScriptAreaText(dataPacket.content)
		elif dataLabel == demo_const.AGENT_REVISION_LOG_LABEL:
			self.gui.displayRevisionLog(dataPacket.content)
		# gamecode handlers
		elif dataLabel == demo_const.WORLD_MESSAGE_LABEL:
			self.statusMessage("game world: %s" % dataPacket.content)
		else:GUIPlayerClient.handleDataPacket(self, dataPacket)


	#########################
	# PRIVATE SCRIPT COMMANDS
	#########################
	def cmdprivate_loadscript(self, cmd):
		"""
		Loads script from local hard-drive into script editor.
		"""
		if not cmd:return makeCommandResult(cmd, message="loadscript <path-to-script-file>", status=svs_const.ERROR)
		filepath = expanduser(''.join(cmd.args))
		try:self.gui.setScriptAreaText(filepath=filepath)
		except:return makeCommandResult(cmd, message="unable to read file: '%s'" % filepath, status=svs_const.ERROR)
		return makeCommandResult(cmd, status=svs_const.OK)

	def cmdprivate_savescript(self, cmd):
		"""
		Saves script from script editor to local hard-drive.
		"""
		if not cmd:return makeCommandResult(cmd, message="savescript <path-to-script-file>", status=svs_const.ERROR)
		filepath = expanduser(''.join(cmd.args))
		scriptText = self.gui.getScriptAreaText()
		if len(scriptText) == 0:return makeCommandResult(cmd, message="no script to save", status=svs_const.ERROR)
		try:open(filepath, 'w').write(scriptText)
		except:return makeCommandResult(cmd, message="unable to write file: '%s'" % filepath, status=svs_const.ERROR)
		return makeCommandResult(cmd, status=svs_const.OK)
	
	
#############################
# GAME WORLD CLIENT
#############################
class GameWorldClient(ScriptableClient):
	"""
	Client that provides game world.
	"""
	def __init__(self, name, passwd, tickDuration=1000):
		ScriptableClient.__init__(self, name, passwd)
		self.gameworld = None
		self.isRunning = False
		self.tickDelay = tickDuration / 1000.0
		# set up game environment stuff
		setScriptEnvironmentProperty(script_const.GAME_CLIENT_PROPERTY, self)
		self.gamecodeLoaded = False
		self.templateCodeLoaded = False

	def loadWorld(self, filepath):
		"""
		Create the game world from specified file.  

		See L{svs_demogame.gameworld.Gameworld.createFromFile} for more details.
		"""
		self.stopClock()
		self.gameworld = Gameworld()
		self.gameworld.createFromFile(filepath)

	def loadGameCode(self, filepath=None):
		"""
		Load predefined scripting code routines for game entities.  
		"""
		loadGameCode(filepath)
		self.gamecodeLoaded = True

	def loadTemplateCode(self, filepath=None):
		"""
		Load template scripting code for game entities.  
		"""
		loadTemplateCode(filepath)
		self.templateCodeLoaded = True

	def startClock(self):
		"""
		Starts simulation clock running.
		"""
		if not self.gamecodeLoaded:self.loadGameCode()
		if not self.templateCodeLoaded:self.loadTemplateCode()
		if not self.isRunning:self.isRunning=True
		reactor.callLater(self.tickDelay, self.tick)

	def stopClock(self):
		"""
		Stops simulation clock running.
		"""
		self.isRunning = False

	def startGame(self):
		"""
		Starts game world running.
		"""
		if not self.isRunning:self.startClock()
		if self.gameworld:self.gameworld.start()
		changes = self.gameworld.getChanges()
		if changes:self.notifyListeners(makeDataPacket(self.profile.name, content=changes, label=demo_const.WORLD_UPDATE_LABEL))

	def stopGame(self):
		"""
		Stops game world running.
		"""
		self.stopClock()
		if self.gameworld:self.gameworld.stop()
		changes = self.gameworld.getChanges()
		if changes:self.notifyListeners(makeDataPacket(self.profile.name, content=changes, label=demo_const.WORLD_UPDATE_LABEL))

	def tick(self):
		"""
		Called by clock to update gameworld.
		"""
		if self.isRunning and self.gameworld:self.gameworld.update()
		changes = self.gameworld.getChanges()
		if changes:self.notifyListeners(makeDataPacket(self.profile.name, content=changes, label=demo_const.WORLD_UPDATE_LABEL))
		reactor.callLater(self.tickDelay, self.tick)

	def getDataForLabel(self, dataRequest):
		"""
		Provides custom handling of data requests.
		"""
		if not dataRequest.label:return ScriptableClient.getDataForLabel(dataRequest)
		if not self.gameworld:return ScriptableClient.getDataForLabel(dataRequest)
		# get world model
		if dataRequest.label == demo_const.WORLD_MODEL_LABEL:
			return makeDataPacket(self.profile.name, content=self.gameworld.getModel(), label=demo_const.WORLD_MODEL_LABEL)

		# get script for area
		elif dataRequest.label == demo_const.AREA_SCRIPT_LABEL:
			area = self.gameworld.getAreaWithId(dataRequest.args[demo_const.AREA_ID_LABEL])
			if not area:return None
			revNum = dataRequest.args.get(demo_const.REVISION_NUMBER_LABEL, -1)
			return makeDataPacket(self.profile.name, content=area.getScriptCode(revNum), label=dataRequest.label)
		# receives new script for area
		elif dataRequest.label == demo_const.NEW_AREA_SCRIPT_LABEL:
			area = self.gameworld.getAreaWithId(dataRequest.args[demo_const.AREA_ID_LABEL])
			if not area:return None
			script = area.setScript(makeScriptFromCode(dataRequest.args[demo_const.SCRIPT_CODE_LABEL], author=dataRequest.sender))
			self.notifyListeners(makeDataPacket(self.profile.name, content=script, label=demo_const.SCRIPT_REVISION_LABEL))
			return makeDataPacket(self.profile.name, label=dataRequest.label)
		# get script revision log for area
		elif dataRequest.label == demo_const.AREA_REVISION_LOG_LABEL:
			area = self.gameworld.getAreaWithId(dataRequest.args[demo_const.AREA_ID_LABEL])
			if not area:return None
			return makeDataPacket(self.profile.name, content=area.getScriptRevisionLog(), label=dataRequest.label)

		# get script for agent
		elif dataRequest.label == demo_const.AGENT_SCRIPT_LABEL:
			agent = self.gameworld.getAgentWithId(dataRequest.args[demo_const.AGENT_ID_LABEL])
			if not agent:return None
			revNum = dataRequest.args.get(demo_const.REVISION_NUMBER_LABEL, -1)
			return makeDataPacket(self.profile.name, content=agent.getScriptCode(revNum), label=dataRequest.label)
		# receives new script for agent
		elif dataRequest.label == demo_const.NEW_AGENT_SCRIPT_LABEL:
			agent = self.gameworld.getAgentWithId(dataRequest.args[demo_const.AGENT_ID_LABEL])
			if not agent:return None
			script = agent.setScript(makeScriptFromCode(dataRequest.args[demo_const.SCRIPT_CODE_LABEL], author=dataRequest.sender))
			self.notifyListeners(makeDataPacket(self.profile.name, content=script, label=demo_const.SCRIPT_REVISION_LABEL))
			return makeDataPacket(self.profile.name, label=dataRequest.label)
		# get script revision log for agent
		elif dataRequest.label == demo_const.AGENT_REVISION_LOG_LABEL:
			agent = self.gameworld.getAgentWithId(dataRequest.args[demo_const.AGENT_ID_LABEL])
			if not agent:return None
			return makeDataPacket(self.profile.name, content=agent.getScriptRevisionLog(), label=dataRequest.label)

		else: # no special method so pass to superclass
			return ScriptableClient.getDataForLabel(self, dataRequest)

	def reportToClient(self, clientName, data, label=demo_const.WORLD_MESSAGE_LABEL):
		"""
		Used to send information to client in response to C{report}
		command in the game script.
		"""
		packet = makeDataPacket(self.profile.name, recipient=clientName, content=data, label=label)
		self.sendData(packet)


	def remote_receiveChatMessage(self, msgPacket):
		"""
		Receives chat message from other clients.

		Chat meesages are used as a way of handling remote commands from other clients.

		NOTE: command messages should be passed to command handler in future versions.
		"""
		msgText = "message from <%s>:\n%s\n" % (msgPacket.sender, msgPacket.content)
		self.logMessage(msgText)
		if len(msgPacket.recipient) != 1:return
		if msgPacket.recipient[0] == self.profile.name:
			msg = msgPacket.content.lower()
			if msg == 'start':
				self.startGame()
			elif msg == 'stop':
				self.stopGame()


		
			

