# svs_demogame.gui

#    Copyright (c) 2005 Simon Yuill.
#
#    This file is part of 'Social Versioning System' (SVS).
#
#    'Social Versioning System' is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation; either version 2 of the License, or
#    (at your option) any later version.
#
#    'Social Versioning System' is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with 'Social Versioning System'; if not, write to the Free Software
#    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

"""
GUIs for game clients.

The game is displayed using the Tkinter library:
U{http://www.pythonware.com/library/tkinter/introduction/}.

@author:	Simon Yuill
@copyright:	2005 Simon Yuill
@license:	GNU GPL version 2 or any later version
@contact:	simon@lipparosa.org
"""

# external imports
from Tkinter import *
from twisted.internet import tksupport
from time import asctime, localtime

# internal imports
from svs_demogame.utils import demo_const
from svs_demogame.gameviews import GameView
from svs_core.gui.clientgui import ClientGUI



class CodePlayerGUI(ClientGUI):
	"""
	Interface for player with script editor.
	"""
	def __init__(self, client):
		ClientGUI.__init__(self, client)

	def build(self):
		"""
		Creates interface components.
		"""
		# clipboard for cut'n'paste
		self.clipboard = None
		# make root window
		self.root = Tk()
		self.root.title('demogame client: %s' % self.client.profile.name)
		self.root.geometry("%dx%d%+d%+d" % (400, 500, 0, 0))
		self.root.bind('<Destroy>', self.destroy)
		# register with twisted
		tksupport.install(self.root)
		### components ###
		## game view
		self.gameFrame = Frame(self.root)
		self.gameFrame.pack(side=TOP, expand=YES, fill=BOTH)
		self.gameView = GameView(self.gameFrame, self)
		self.gameView.canvas.pack(side=LEFT, expand=YES, fill=BOTH)
		## script editing
		self.scriptFrame = Frame(self.root)
		self.scriptFrame.pack(side=TOP, expand=YES, fill=BOTH)
		self.gameInfoLabelText = StringVar()
		self.gameInfoLabel = Label(self.scriptFrame, fg='#666666', justify=LEFT, anchor=W, font=("Helvetica", 12), textvariable=self.gameInfoLabelText)
		self.gameInfoLabel.pack(side=TOP, expand=NO, fill=X)
		# script buttons
		self.scriptBtnFrame = Frame(self.scriptFrame)
		self.scriptBtnFrame.pack(side=BOTTOM, expand=NO, fill=X)
		self.enterScriptBtn = Button(self.scriptBtnFrame, text='Enter', command=self.onEnterScript).pack(side=LEFT)
		self.histBtn = Button(self.scriptBtnFrame, text='History', command=self.onScriptHistory).pack(side=LEFT)
		self.resetScriptBtn = Button(self.scriptBtnFrame, text='Reset', command=self.onResetScript).pack(side=LEFT)
		self.quitBtn = Button(self.scriptBtnFrame, text='Quit', command=self.onQuitClient).pack(side=RIGHT)
		# script editor
		self.scriptAreaBar = Scrollbar(self.scriptFrame)
		self.scriptArea = Text(self.scriptFrame, height=6, relief=SUNKEN)
		self.scriptAreaBar.config(command=self.scriptArea.yview)
		self.scriptArea.config(yscrollcommand=self.scriptAreaBar.set)
		self.scriptArea.config(bg='white', fg='red')
		self.scriptArea.config(font=('courier', 12, 'normal'))
		self.scriptArea.bind('<Control-c>', self.onCopyScriptArea)
		self.scriptArea.bind('<Control-x>', self.onCutScriptArea)
		self.scriptArea.bind('<Control-v>', self.onPasteScriptArea)
		self.scriptAreaBar.pack(side=RIGHT, fill=Y)
		self.scriptArea.pack(side=TOP, expand=YES, fill=BOTH)
		### client communication
		self.consoleFrame = Frame(self.root)
		self.consoleFrame.pack(side=BOTTOM, expand=NO, fill=X)
		consoleLabel = Label(self.consoleFrame, text='Client terminal:', fg='#333333', justify=LEFT, anchor=W, font=("Helvetica", 12)).pack(side=TOP, expand=YES, fill=X)
		# input field
		self.inputFrame = Frame(self.consoleFrame)
		self.inputFrame.pack(side=TOP, expand=YES, fill=X)
		self.inputField = Entry(self.inputFrame)
		self.inputField.pack(side=LEFT, expand=YES, fill=X)
		self.inputField.config(bg='#EEEEEE', fg='#666666')
		self.inputField.bind('<Return>', (lambda event: self.enterCommand()))
		self.inputField.bind('<Up>', (lambda event: self.stepBackCommand()))
		self.inputField.bind('<Down>', (lambda event: self.stepForwardCommand()))
		self.inputBuffer = []
		self.inputBufferMaxSize = 20
		self.inputBufferIndex = 0
		self.echoInput = True
		# console
		self.sbar = Scrollbar(self.consoleFrame)
		self.console = Text(self.consoleFrame, height=6, relief=SUNKEN)
		self.sbar.config(command=self.console.yview)
		self.console.config(yscrollcommand=self.sbar.set)
		self.console.config(bg='#EEEEEE', fg='#666666')
		self.console.bind('<Control-c>', self.onCopyConsole)
		self.console.bind('<Control-x>', self.onCutConsole)
		self.console.bind('<Control-v>', self.onPasteConsole)
		self.sbar.pack(side=RIGHT, fill=Y)
		self.console.pack(side=LEFT, expand=YES, fill=X)
		### root
		self.root.protocol("WM_DELETE_WINDOW", self.destroy)
		#self.root.mainloop()

	def setupGameWorld(self, gameModel):
		"""
		Creates gameworld views from specified model.

		The model is received as a dictionary in the following form:

			- areas_x: number of areas on x axis
			- areas_y: number of areas on y axis
			- area_data: array of information about specific areas
		"""
		if not gameModel:return # make error
		terrainModel = gameModel.get(demo_const.TERRAIN_LABEL)
		self.gameView.setWorldDimensions(terrainModel[demo_const.DIM_X_LABEL], terrainModel[demo_const.DIM_Y_LABEL])
		self.gameView.setupTerrainAreas(terrainModel[demo_const.AREAS_X_LABEL], terrainModel[demo_const.AREAS_Y_LABEL], terrainModel[demo_const.AREA_DATA_LABEL])
		self.gameView.setupAgents(gameModel.get(demo_const.AGENTS_LABEL))
		self.gameView.canvasAdjusted()

	def updateGameWorld(self, updateData):
		"""
		Updates gameworld views from specified data.
		"""
		if not updateData:return # don't bail
		terrainData = updateData.get(demo_const.TERRAIN_LABEL, None)
		if not terrainData:return # don't bail
		self.gameView.updateTerrainAreas(terrainData.get(demo_const.AREAS_LABEL, None))
		self.gameView.updateAgents(terrainData.get(demo_const.AGENTS_LABEL, None))

	#########################
	# BUTTON COMMANDS
	#########################
	def onScriptHistory(self):
		"""
		Responds to 'History' button being clicked, get revision log for script.
		"""
		areaId = self.getSelectedAreaId()
		if areaId:
			self.client.getRevisionLogForArea(areaId)
			return
		agentId = self.getSelectedAgentId()
		if agentId:self.client.getRevisionLogForAgent(agentId)

	def onEnterScript(self):
		"""
		Responds to 'Enter' button being clicked, send script to gameworld.
		"""
		areaId = self.getSelectedAreaId()
		if areaId:
			self.client.sendScriptForArea(areaId, self.getScriptAreaText())
			return
		agentId = self.getSelectedAgentId()
		if agentId:self.client.sendScriptForAgent(agentId, self.getScriptAreaText())
		

	def onResetScript(self):
		"""
		Responds to 'Reset' button being clicked, clears script editor.
		"""
		self.clearScriptArea()
	
	def onQuitClient(self):
		"""
		Quits game client.
		"""
		self.destroy()
	
	#########################
	# SELECTION METHODS
	#########################
	def areaSelected(self, areaIdNum):
		"""
		Called by area view compoents when selected, passes info onto client.
		"""
		self.gameInfoLabelText.set('Area: %d, current script' % areaIdNum)
		self.client.logMessage("[area <%d> selected]" % areaIdNum);
		self.client.getScriptForArea(areaIdNum)

	def getSelectedAreaId(self):
		"""
		Returns id for terrain area selected in L{self.gameView}.
		"""
		if not self.gameView.selectedArea:return None
		return self.gameView.selectedArea.idNum

	def getSelectedAgentId(self):
		"""
		Returns id for agent selected in L{self.gameView}.
		"""
		if not self.gameView.selectedAgent:return None
		return self.gameView.selectedAgent.idNum
 
	def agentSelected(self, agentIdNum):
		"""
		Called by agent view compoents when selected, passes info onto client.
		"""
		self.gameInfoLabelText.set('Agent: %d' % agentIdNum)

	def getScriptForAgent(self, agentIdNum):
		"""
		Passes on request to client for script for agant.
		"""
		self.client.getScriptForAgent(agentIdNum)

	def displayPosition(self, viewX, viewY):
		"""
		Displays gameworld coordinates for specified view location.
		"""
		self.gameInfoLabelText.set('Location: %d, %d' % (viewX, viewY))

	#########################
	# MAIN METHODS
	#########################
	def destroy(self, args=None):
		"""
		Closes and destroys interface components.
		"""
		self.client.disconnect()
		self.root.quit()

	def showHandCursor(self, event):
		self.scriptArea.config(cursor="hand2")

	def showDefaultCursor(self, event):
		self.scriptArea.config(cursor=self.defaultCursor)
		
	def update(self, timestamp):
		"""
		Updates view components.

		@todo: does nothing at present.
		"""
		pass
		#print "update:", timestamp

	#########################
	# SCRIPT AREA
	#########################
	def clearScriptArea(self):
		"""
		Clear script editing area.
		"""
		self.scriptArea.delete('1.0', END)
	
	def setScriptAreaText(self, text='', filepath=None):
		"""
		Clear script editing area and display new text.
		"""
		self.scriptArea.config(bg='white', fg='red')
		if filepath:
			text = open(filepath, 'r').read()
		self.scriptArea.delete('1.0', END)
		self.scriptArea.insert('1.0', text)
		self.scriptArea.mark_set(INSERT, '1.0')
		self.scriptArea.see(INSERT)
		self.scriptArea.config(cursor="xterm")

	def getScriptAreaText(self):
		"""
		Retrieves text contents from script editing area.
		"""
		return self.scriptArea.get('1.0', END+'-1c')

	def onCopyScriptArea(self, event):
		"""
		Copies selected text from script area.
		"""
		self.clipboard = self.scriptArea.get(SEL_FIRST, SEL_LAST)

	def onCutScriptArea(self, event):
		"""
		Cuts and copies selected text from script area.
		"""
		self.clipboard = self.scriptArea.get(SEL_FIRST, SEL_LAST)
		self.scriptArea.delete(SEL_FIRST, SEL_LAST)

	def onPasteScriptArea(self, event):
		"""
		Pastes copied text into script area.
		"""
		try:self.scriptArea.delete(SEL_FIRST, SEL_LAST)
		except TclError:pass
		try:self.scriptArea.insert(INSERT, self.clipboard)
		except TclError:pass
	
	def displayRevisionLog(self, logData):
		"""
		Displays revision log.
		"""
		self.scriptArea.delete('1.0', END)
		self.scriptArea.config(bg='white', fg='black')
		if self.getSelectedAreaId():
			self.gameInfoLabelText.set('Area: %d, script history' % self.getSelectedAreaId())
		elif self.getSelectedAgentId():
			self.gameInfoLabelText.set('Agent: %d, script history' % self.getSelectedAgentId())
		lineNum = 1
		for entry in logData['revisions']:
			author = entry.get('author', None)
			if not author:author = 'no author'
			revNum = entry['revision']
			revTag = 'rev_link_%d' % revNum
			revText = "revision: %d" % revNum
			text = "%s\ndate: %s\nauthor: %s\n\n" % (revText, asctime(localtime(entry['date'])), author)
			self.scriptArea.insert('%d.0' % lineNum, text)
			self.scriptArea.tag_add(revTag, '%d.0' % lineNum, '%d.%d' % (lineNum, len(revText)))
			# handler for revision links
			def linkHandler ( event, self=self, i=revNum):
                		self.revisionSelected ( event, i )
			# end handler
			self.scriptArea.tag_bind(revTag, '<Button-1>', linkHandler)
			self.scriptArea.tag_bind(revTag, "<Enter>", self.showHandCursor)
			self.scriptArea.tag_bind(revTag, "<Leave>", self.showDefaultCursor)
			self.scriptArea.tag_config(revTag, foreground='red')
			self.scriptArea.tag_config(revTag, font=('courier', 12, 'underline'))
			lineNum += 4
		self.scriptArea.config(cursor="left_ptr")
		self.defaultCursor="left_ptr"
		self.scriptArea.mark_set(INSERT, '1.0')
		self.scriptArea.see(INSERT)

	def revisionSelected(self, event, revNumber):
		"""
		Called when a script revision is selected from the revision log.
		"""
		self.defaultCursor="xterm"
		entityId = self.getSelectedAreaId()
		if entityId:
			self.gameInfoLabelText.set('Area: %d, revision %d' % (entityId, revNumber))
			self.client.getScriptForArea(entityId, revNumber)
			return
		entityId = self.getSelectedAgentId()
		if entityId:
			self.gameInfoLabelText.set('Agent: %d, revision %d' % (entityId, revNumber))
			self.client.getScriptForAgent(entityId, revNumber)

	#########################
	# CONSOLE
	#########################
	def onCopyConsole(self, event):
		"""
		Copies selected text from console.
		"""
		self.clipboard = self.console.get(SEL_FIRST, SEL_LAST)

	def onCutConsole(self, event):
		"""
		Cuts and copies selected text from console.
		"""
		self.clipboard = self.console.get(SEL_FIRST, SEL_LAST)
		self.console.delete(SEL_FIRST, SEL_LAST)

	def onPasteConsole(self, event):
		"""
		Pastes copied text into console.
		"""
		try:self.console.delete(SEL_FIRST, SEL_LAST)
		except TclError:pass
		try:self.console.insert(INSERT, self.clipboard)
		except TclError:pass

