# svs_demogame.networkviews

#    Copyright (c) 2005 Simon Yuill.
#
#    This file is part of 'Social Versioning System' (SVS).
#
#    'Social Versioning System' is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation; either version 2 of the License, or
#    (at your option) any later version.
#
#    'Social Versioning System' is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with 'Social Versioning System'; if not, write to the Free Software
#    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

"""
Graphical representation of interaction network.

The game is displayed using the Tkinter library:
U{http://www.pythonware.com/library/tkinter/introduction/}.

@author:	Simon Yuill
@copyright:	2005 Simon Yuill
@license:	GNU GPL version 2 or any later version
@contact:	simon@lipparosa.org
"""
# external imports
from Tkinter import *
import math

# internal imports
from svs_demogame.views import GenericDataView,GenericComponentView, encodeSelectedItem
from svs_demogame.utils import demo_const
from svs_core.utilities.constants import svs_const
from svs_core.geometry.geomlib import geom_const
from svs_core.utilities.lib import Constants


#############################
# CONSTANTS
#############################
networkviews_const = Constants()
# tkinter tags
networkviews_const.NODE_TAG = 'node'
networkviews_const.NODE_EDGE_TAG = 'node_edge'



#############################
# NETWORK
#############################
class NetworkView(GenericDataView):
	"""
	Displays interaction networks within game.
	"""
	def __init__(self, parent, context):
		GenericDataView.__init__(self)
		self.context = context
		self.width = 0
		self.height = 0
		self.canvas = Canvas(parent, bg=demo_const.BG_COLOUR, borderwidth=0, highlightthickness=0)
		self.canvas.bind('<Configure>', self.canvasAdjusted)
		self.horizontalMargin = 10
		self.verticalMargin = 10
		self.maxRadius = 0
		self.spriteRadius = 5
		self.centreX = 0
		self.centreY = 0
		self.nodes = {}
		self.orderedNodeNames = []
		self.selectedNode = None
		self.canvasAdjusted()

	def clearCanvas(self):
		"""
		Clears entire canvas.
		"""
		self.canvas.delete(networkviews_const.NODE_TAG)
		self.canvas.delete(networkviews_const.NODE_EDGE_TAG)

	def canvasAdjusted(self, args=None):
		"""
		Responds to main window being adjusted in size.
		"""
		self.width = self.canvas.winfo_width()
		self.height = self.canvas.winfo_height()
		self.centreX = self.width / 2.0
		self.centreY = self.height / 2.0
		if self.width < self.height:self.maxRadius = (self.width / 2.0) - self.horizontalMargin
		else:self.maxRadius = (self.height / 2.0) - self.verticalMargin
		self.spriteRadius = self.maxRadius / 20.0
		self.clearCanvas()
		self.drawNodesFresh()

	def updateView(self, data):
		"""
		Performs update of view in response to change in data source.
		"""
		nodeData = data[0]
		self.orderedNodeNames = data[1]
		for nodeName in self.orderedNodeNames:
			self.addNode(nodeData[nodeName])
		self.clearCanvas()
		self.drawNodesFresh()

	def addNode(self, node):
		"""
		Adds node to view and create appropriate sprites.
		"""
		if self.nodes.has_key(node.label):return
		self.nodes[node.label] = NetworkNodeSprite(self, node)

	def getNode(self, nodeLabel):
		"""
		Returns node specified by label, if not found returns C{None}.
		"""
		return self.nodes.get(nodeLabel, None)

	def drawNodesFresh(self):
		"""
		Draws new view graphics for nodes.
		"""
		if len(self.orderedNodeNames) == 0: return
		theta = 360.0 / len(self.orderedNodeNames)
		nodeAngle = 0.0
		# NOTE: need to set radius to reflect connectivity
		for nodeName in self.orderedNodeNames:
			spriteX = self.centreX + (self.maxRadius * math.sin(geom_const.deg2rad * nodeAngle))
			spriteY = self.centreY + (self.maxRadius * math.cos(geom_const.deg2rad * nodeAngle))
			nodeAngle += theta
			self.nodes[nodeName].drawFresh(spriteX, spriteY, self.spriteRadius)
		for nodeName in self.orderedNodeNames:
			self.nodes[nodeName].drawEdgesFresh()
		self.canvas.lift(networkviews_const.NODE_TAG)

	def nodeSelected(self, node):
		"""
		Called by node view components when selected, passes info onto client.
		"""
		if self.selectedNode:self.selectedNode.deselect()
		self.selectedNode = node
		self.selectedNode.select()
		self.context.nodeSelected(node)

	def selectItem(self, selectedItem):
		"""
		Selects specified item if present within view.

		This method is used to synchronise selesctions 
		between the different tracker views.
		"""
		if self.nodes.has_key(selectedItem.itemName):self.nodeSelected(self.nodes[selectedItem.itemName])
		else:self.clearSelection()
			
	def clearSelection(self):
		"""
		Clears any selected items in view.
		"""
		if self.selectedNode:
			self.selectedNode.deselect()
			self.selectedNode = None

	def makeSelection(self, selectedItem):
		"""
		Called by component sprites when selected.

		Sends message to parent for synchronisation.
		"""
		self.context.syncSelection(encodeSelectedItem(selectedItem.dataSource.label, selectedItem.dataSource, self))
		
			

		

class NetworkNodeSprite(GenericComponentView):
	"""
	Displays individual node in L{NetworkView}.
	"""
	def __init__(self, context, dataSource):
		GenericComponentView.__init__(self, context)
		self.colour = demo_const.DFT_COLOUR
		self.dataSource = dataSource
		self.edgeFromSprites = []
		self.edgeToSprites = []
		self.x = 0
		self.y = 0

	def drawFresh(self, locX, locY, drawRadius):
		"""
		Draws node onto canvas.
		"""
		self.x = locX
		self.y = locY
		self.xMin = locX - drawRadius
		self.yMin = locY - drawRadius
		self.xMax = self.xMin + (drawRadius * 2)
		self.yMax = self.yMin + (drawRadius * 2)
		dataSourceType = self.dataSource.getDataSourceType()
		#print "dataSourceType:", dataSourceType
		if dataSourceType == 'Player':
			self.sprite = self.canvas.create_oval(self.xMin, self.yMin, self.xMax, self.yMax, fill=self.colour, outline=self.colour, width=1, tag=networkviews_const.NODE_TAG)	
		elif dataSourceType == 'Script':
			self.sprite = self.canvas.create_rectangle(self.xMin, self.yMin, self.xMax, self.yMax, fill=self.colour, outline=self.colour, width=1, tag=networkviews_const.NODE_TAG)
		#elif dataSourceType == 'Script:Agent':
			#self.sprite = self.canvas.create_rectangle(self.xMin, self.yMin, self.xMax, self.yMax, fill='white', outline=self.colour, width=1, tag=networkviews_const.NODE_TAG)
		elif dataSourceType == 'Agent':
			self.sprite = self.canvas.create_oval(self.xMin, self.yMin, self.xMax, self.yMax, fill='white', outline=self.colour, width=1, tag=networkviews_const.NODE_TAG)
		self.canvas.tag_bind(self.sprite, '<Double-1>', self.onDoubleClick)
		self.canvas.tag_bind(self.sprite, '<Button-1>', self.onMouseClick)
		self.canvas.tag_bind(self.sprite, "<Enter>", self.hilight)
		self.canvas.tag_bind(self.sprite, "<Leave>", self.unHighlight)
		if self.selected:self.context.canvas.itemconfig(self.sprite, outline=demo_const.SPRITE_SELECTED_COLOUR, fill=demo_const.SPRITE_SELECTED_COLOUR)

	def drawEdgesFresh(self):
		"""
		Draws edges from node.
		"""
		self.edgeFromSprites = []
		for nodeLabel, edge in self.dataSource.edges.items():
			#print "edge:", edge.linkage
			nodeSprite = self.context.getNode(nodeLabel)
			edgeSprite = NetworkEdgeSprite(self.context)
			edgeSprite.drawFresh(self.x, self.y, nodeSprite.x, nodeSprite.y)
			self.edgeFromSprites.append(edgeSprite)
			nodeSprite.addEdgeTo(edgeSprite)
		if self.selected:
			for edge in self.edgeFromSprites:edge.select()
			for edge in self.edgeToSprites:edge.select()

	def addEdgeTo(self, edge):
		"""
		Adds edge from another node.
		"""
		self.edgeToSprites.append(edge)
		if self.selected:edge.select()

	def onDoubleClick(self, event):
		"""
		Responds to double-click from mouse.
		"""
		#self.context.nodeSelected(self)

	def onMouseClick(self, event):
		"""
		Responds to double-click from mouse.
		"""
		#self.context.nodeSelected(self)
		self.context.makeSelection(self)

	def hilight(self, event):
		"""
		Highlights sprite.
		"""
		if self.selected:return
		self.canvas.itemconfig(self.sprite, outline=demo_const.SPRITE_HILIGHT_COLOUR, fill=demo_const.SPRITE_HILIGHT_COLOUR)
		for edge in self.edgeFromSprites:edge.hilight()
		for edge in self.edgeToSprites:edge.hilight()

	def unHighlight(self, event):
		"""
		Highlights sprite.
		"""
		if self.selected:return
		self.canvas.itemconfig(self.sprite, outline=self.colour, fill=self.colour)
		for edge in self.edgeFromSprites:edge.unHighlight()
		for edge in self.edgeToSprites:edge.unHighlight()
	
	def select(self):
		"""
		Selects sprite.
		"""
		self.selected = True
		self.context.canvas.itemconfig(self.sprite, outline=demo_const.SPRITE_SELECTED_COLOUR, fill=demo_const.SPRITE_SELECTED_COLOUR)
		for edge in self.edgeFromSprites:edge.select()
		for edge in self.edgeToSprites:edge.select()

	def deselect(self):
		"""
		Deselects sprite.
		"""
		if not self.selected:return
		self.selected = False
		self.context.canvas.itemconfig(self.sprite, outline=self.colour, fill=self.colour)
		for edge in self.edgeFromSprites:edge.deselect()
		for edge in self.edgeToSprites:edge.deselect()
			

class NetworkEdgeSprite(GenericComponentView):
	"""
	Displays individual edge in L{NetworkView}.
	"""
	def __init__(self, context):
		GenericComponentView.__init__(self, context)
		self.colour = demo_const.DFT_COLOUR
		self.startX = 0
		self.startY = 0
		self.endX = 0
		self.endY = 0

	def drawFresh(self, startX, startY, endX, endY):
		"""
		Draw edges.
		"""
		self.startX = startX
		self.startY = startY
		self.endX = endX
		self.endY = endY
		self.sprite = self.canvas.create_line(self.startX, self.startY, self.endX, self.endY, fill=self.colour, width=1, tag=networkviews_const.NODE_EDGE_TAG)
		if self.selected:self.context.canvas.itemconfig(self.sprite, fill=demo_const.SPRITE_SELECTED_COLOUR)


	def hilight(self):
		"""
		Highlights sprite.
		"""
		if self.selected:return
		self.canvas.itemconfig(self.sprite, fill=demo_const.SPRITE_HILIGHT_COLOUR)

	def unHighlight(self):
		"""
		Highlights sprite.
		"""
		if self.selected:return
		self.canvas.itemconfig(self.sprite, fill=self.colour)
	
	def select(self):
		"""
		Selects sprite.
		"""
		self.selected = True
		self.context.canvas.itemconfig(self.sprite, fill=demo_const.SPRITE_SELECTED_COLOUR)

	def deselect(self):
		"""
		Deselects sprite.
		"""
		if not self.selected:return
		self.selected = False
		self.context.canvas.itemconfig(self.sprite, fill=self.colour)
