# svs_demogame.terrain

#    Copyright (c) 2005 Simon Yuill.
#
#    This file is part of 'Social Versioning System' (SVS).
#
#    'Social Versioning System' is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation; either version 2 of the License, or
#    (at your option) any later version.
#
#    'Social Versioning System' is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with 'Social Versioning System'; if not, write to the Free Software
#    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

"""
Classes representing terrain in demo game.

@author:	Simon Yuill
@copyright:	2005 Simon Yuill
@license:	GNU GPL version 2 or any later version
@contact:	simon@lipparosa.org
"""
# internal imports
from svs_demogame.utils import demo_const
from svs_demogame.scripts import ScriptHandler, Script
from svs_demogame.base_entities import ScriptableEntity
from svs_simulation.numdata.geomlib import Rect2D


class Terrain:
	"""
	Container class for terrain components.
	"""
	def __init__(self, dimX=1, dimY=1, areasX=0, areasY=0):
		self.bounds = Rect2D(0.0, 0.0, dimX, dimY)
		self.createAreas(areasX, areasY)
		self.changes = None
		self.boundaryArea = BoundaryArea(self, -1, -1)

	def createAreas(self, areasX, areasY):
		"""
		Creates areas for terrain in given dimensions.
		"""
		self.areasX = areasX
		self.areasY = areasY
		self.areaDimX = self.bounds.size.dimX / (self.areasX * 1.0)
		self.areaDimY = self.bounds.size.dimY / (self.areasY * 1.0)
		self.areas = []
		self.areasById = {}
		for x in range(self.areasX):
			self.areas.append([])
			for y in range(self.areasY):
				area = TerrainArea(self, x, y)
				area.setLocation(self.areaDimX * x, self.areaDimY * y)
				area.setDimensions(self.areaDimX, self.areaDimY)
				self.areas[x].append(area)
				self.areasById[area.getId()] = area
				
	def setAreaDefaults(self, density=None, script=None):
		"""
		Sets default values for all areas.
		"""
		for area in self.areasById.values():
			if density: area.setDensity(density)
			if script: area.setScript(script)
			
	def getAreaAtIndices(self, x, y):
		"""
		Returns area at specified index coordinates.  If not
		found returns None.
		"""
		if x < 0 or y < 0:return None
		if x >= self.areasX or y >= self.areasY:return None
		return self.areas[x][y]

	def getAreaAtLocation(self, xPoint, yPoint):
		"""
		Returns area containing specified terrain coordinates. If not
		found returns None.
		"""
		return self.getAreaAtIndices(int(xPoint/self.areaDimX), int(yPoint/self.areaDimY))
		

	def getAreaWithId(self, areaId):
		"""
		Returns area with specified id number.
		"""
		return self.areasById.get(areaId)

	def getAllAreas(self):
		"""
		Returns all areas in a flat list.
		"""
		return self.areasById.values()

	def getAdjacentAreas(self, area, directions=(), markBoundary=False):
		"""
		Returns a list of areas which are adjacent to the specified
		area.  

		If the directions are set these can be used to return
		only the neighbours in the specified directions. The directions
		are defined by compass names, 'N', 'S', 'E', 'W', 'NE', etc.

		@rtype: Array
		"""
		neighbours = {}
		nullArea = None
		if markBoundary: nullArea = self.boundaryArea
		if len(directions) == 0:
			areaNE = self.getAreaAtIndices(area.indexX + 1, area.indexY - 1)
			if areaNE:neighbours['NE'] = areaNE
			else:neighbours['NE'] = nullArea
			areaN = self.getAreaAtIndices(area.indexX, area.indexY - 1)
			if areaN:neighbours['N'] = areaN
			else:neighbours['N'] = nullArea
			areaNW = self.getAreaAtIndices(area.indexX - 1, area.indexY - 1)
			if areaNW:neighbours['NW'] = areaNW
			else:neighbours['NW'] = nullArea
			areaE = self.getAreaAtIndices(area.indexX + 1, area.indexY)
			if areaE:neighbours['E'] = areaE
			else:neighbours['E'] = nullArea
			areaW = self.getAreaAtIndices(area.indexX - 1, area.indexY)
			if areaW:neighbours['W'] = areaW
			else:neighbours['W'] = nullArea
			areaSE = self.getAreaAtIndices(area.indexX + 1, area.indexY + 1)
			if areaSE:neighbours['SE'] = areaSE
			else:neighbours['SE'] = nullArea
			areaS = self.getAreaAtIndices(area.indexX, area.indexY + 1)
			if areaS:neighbours['S'] = areaS
			else:neighbours['S'] = nullArea
			areaSW = self.getAreaAtIndices(area.indexX - 1, area.indexY + 1)
			if areaSW:neighbours['SW'] = areaSW
			else:neighbours['SW'] = nullArea
			return neighbours
		if 'NE' in directions:
			areaNE = self.getAreaAtIndices(area.indexX + 1, area.indexY - 1)
			if areaNE:neighbours['NE'] = areaNE
			else:neighbours['NE'] = nullArea
		if 'N' in directions:
			areaN = self.getAreaAtIndices(area.indexX, area.indexY - 1)
			if areaN:neighbours['N'] = areaN
			else:neighbours['N'] = nullArea
		if 'NW' in directions:
			areaNW = self.getAreaAtIndices(area.indexX - 1, area.indexY - 1)
			if areaNW:neighbours['NW'] = areaNW
			else:neighbours['NW'] = nullArea
		if 'E' in directions:
			areaE = self.getAreaAtIndices(area.indexX + 1, area.indexY)
			if areaE:neighbours['E'] = areaE
			else:neighbours['E'] = nullArea
		if 'W' in directions:
			areaW = self.getAreaAtIndices(area.indexX - 1, area.indexY)
			if areaW:neighbours['W'] = areaW
			else:neighbours['W'] = nullArea
		if 'SE' in directions:
			areaSE = self.getAreaAtIndices(area.indexX + 1, area.indexY + 1)
			if areaSE:neighbours['SE'] = areaSE
			else:neighbours['SE'] = nullArea
		if 'S' in directions:
			areaS = self.getAreaAtIndices(area.indexX, area.indexY + 1)
			if areaS:neighbours['S'] = areaS
			else:neighbours['S'] = nullArea
		if 'SW' in directions:
			areaSW = self.getAreaAtIndices(area.indexX - 1, area.indexY + 1)
			if areaSW:neighbours['SW'] = areaSW
			else:neighbours['SW'] = nullArea
		return neighbours
		 

	def getModel(self):
		"""
		Returns a model of the terrain in its current state, suitable
		for serialisation and sending over network.
		"""
		model = {demo_const.AREAS_X_LABEL:self.areasX, demo_const.AREAS_Y_LABEL:self.areasY, demo_const.AREA_DATA_LABEL:[], demo_const.DIM_X_LABEL:self.bounds.size.dimX, demo_const.DIM_Y_LABEL:self.bounds.size.dimY}
		for x in range(self.areasX):
			for y in range(self.areasY):
				model[demo_const.AREA_DATA_LABEL].append(self.areas[x][y].getProfile())
		return model
	
	def containsLocation(self, x, y):
		"""
		Checks that specified coordinates are within boundaries of terrain.	
		"""
		return self.bounds.containsPoint(x, y)

	def start(self):
		"""
		Calls C{startPlay} method on areas.
		"""
		for area in self.areasById.values():area.startPlay()

	def stop(self):
		"""
		Calls C{stopPlay} method on areas.
		"""
		for area in self.areasById.values():area.stopPlay()

	def update(self, timeInterval):
		"""
		Updates areas within terrain.
		"""
		self.changes = {}
		for area in self.areasById.values():area.update(timeInterval)

	def reportChange(self, changeData, sourceEntity):
		"""
		Collects changes in condition of areas and agents.
		"""
		if not changeData:return
		if not self.changes:self.changes = {}
		if not self.changes.has_key(sourceEntity): self.changes[sourceEntity] = []
		self.changes[sourceEntity].append(changeData)

	def getChanges(self):
		"""
		Get reported changes in terrain.
		"""
		if not self.changes:return None
		if len(self.changes) == 0:return None
		return self.changes
		
		

class TerrainArea(ScriptableEntity):
	"""
	Area within terrain capable of being scripted.
	"""
	def __init__(self, terrain, indexX, indexY):
		ScriptableEntity.__init__(self)
		self.terrain = terrain
		self.density = 0.0
		self.bounds = Rect2D(0.0, 0.0, 0.0, 0.0)
		self.indexX = indexX
		self.indexY = indexY
		self.occupants = []
	
	def getId(self):
		"""
		Returns id for area.
		"""
		return self.idNum

	def setLocation(self, newPosX, newPosY):
		"""
		Sets location of terrain area, making sure that 
		it is within the bounds of the terrain.
		"""
		if self.terrain.containsLocation(newPosX, newPosY):self.bounds.setOrigin(newPosX, newPosY)

	def setDimensions(self, dimX, dimY):
		"""
		Sets new dimensions for area.
		"""
		self.bounds.setSize(dimX, dimY)

	def setDensity(self, newValue):
		"""
		Sets the density of the terrain area, making
		sure it falls within the range 0.0 - 1.0, and 
		updating game views.
		"""
		if newValue < 0.0: newValue = 0.0
		if newValue > 1.0 : newValue = 1.0
		self.density = newValue
		self.terrain.reportChange(self.getProfile(), demo_const.AREAS_LABEL)

	def getDensity(self):
		"""
		Returns the density value for the terrain area.
		"""
		return self.density

	def getProfile(self):
		"""
		Returns L{TerrainAreaProfile} object for sending over network.
		"""
		#return self.profile 
		return {demo_const.AREA_ID_LABEL: self.idNum, demo_const.DENSITY_LABEL:self.density}

	def getScriptIdentifier(self):
		"""
		Returns a string name for the object that can be 
		used by the L{ScriptHandler}.
		"""
		return "<area_%d>" % self.idNum

	def executeScriptOnAgent(self, agent):
		"""
		Executes the current script on the specified agent.
		"""
		pass

	def executeStartPlayScript(self):
		"""
		Executes C{startPlay} handler in script attached to area.
		"""
		self.scriptHandler.executeCurrentScriptMethod('startPlay')

	def executeStopPlayScript(self):
		"""
		Executes C{stopPlay} handler in script attached to area.
		"""
		self.scriptHandler.executeCurrentScriptMethod('stopPlay')

	def executeAgentEnteredScript(self, agent):
		"""
		Executes C{agentEntered} handler in script attached to area.
		"""
		self.scriptHandler.executeCurrentScriptMethod('agentEntered', agent)

	def executeAgentExitedScript(self, agent):
		"""
		Executes C{agentExited} handler in script attached to area.
		"""
		self.scriptHandler.executeCurrentScriptMethod('agentExited', agent)

	def startPlay(self):
		"""
		Called when play starts.
		"""
		self.scriptHandler.loadCurrentScript()
		self.executeStartPlayScript()

	def stopPlay(self):
		"""
		Called when play stops.
		"""
		self.scriptHandler.loadCurrentScript()
		self.executeStopPlayScript()

	def update(self, timeInterval):
		"""
		Updates area.
		"""
		self.scriptHandler.loadCurrentScript()
		self.executeScriptOnSelf()

	def getNeighbours(self, *directions, **kwargs):
		"""
		Returns a list of areas which are adjacent to this area.  

		If the directions are set these can be used to return
		only the neighbours in the specified directions. The directions
		are defined by compass names, 'N', 'S', 'E', 'W', 'NE', etc.

		@rtype: Array
		"""
		if kwargs.has_key('markBoundary'):
			return self.terrain.getAdjacentAreas(self, directions, markBoundary=kwargs['markBoundary'])
		else:
			return self.terrain.getAdjacentAreas(self, directions)

	def agentEntered(self, agent):
		"""
		Responds to agent entering area.
		"""
		self.addOccupant(agent)
		self.executeAgentEnteredScript(agent)

	def agentExited(self, agent):
		"""
		Responds to agent leaving area.
		"""
		self.removeOccupant(agent)
		self.executeAgentExitedScript(agent)

	def addOccupant(self, occupant):
		"""
		Adds an occupant to the list of occupants in area.
		"""
		if not occupant in self.occupants:self.occupants.append(occupant)

	def removeOccupant(self, occupant):
		"""
		Removes an occupant to the list of occupants in area.
		"""
		if occupant in self.occupants:self.occupants.remove(occupant)

	def hasOccupants(self):
		"""
		Returns C{True} if there are any occupants in area, otherwise C{False}.
		"""
		if len(self.occupants):return True
		else:return False

	def isAccessible(self):
		"""
		Checks if area has any occupants or if density is 1.0, in which case
		it cannot be entered by an agent.
		"""
		if self.density < 1.0 and len(self.occupants) == 0:return True
		else:return False

class BoundaryArea(TerrainArea):
	"""
	Area on boundary of terrain.
	"""
	def __init__(self, terrain, indexX, indexY):
		TerrainArea.__init__(self, terrain, indexX, indexY)

	def isAccessible(self):
		"""
		Checks if area has any occupants or if density is 1.0, in which case
		it cannot be entered by an agent.
		"""
		return False
