# svs_demogame.tracking_gui

#    Copyright (c) 2005 Simon Yuill.
#
#    This file is part of 'Social Versioning System' (SVS).
#
#    'Social Versioning System' is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation; either version 2 of the License, or
#    (at your option) any later version.
#
#    'Social Versioning System' is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with 'Social Versioning System'; if not, write to the Free Software
#    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

"""
GUI for tracking clients.

The gui is built using the Tkinter library:
U{http://www.pythonware.com/library/tkinter/introduction/}.

@author:	Simon Yuill
@copyright:	2005 Simon Yuill
@license:	GNU GPL version 2 or any later version
@contact:	simon@lipparosa.org
"""

# external imports
from Tkinter import *
from twisted.internet import tksupport
from time import asctime, localtime

# internal imports
from svs_demogame.utils import demo_const
from svs_demogame.gameviews import TrackerGameView
from svs_demogame.scripthistoryviews import ScriptHistoryView
from svs_demogame.messagehistoryviews import MessageHistoryView
from svs_demogame.networkviews import NetworkView
from svs_demogame.gamelogicviews import GameLogicView
from svs_demogame.timebasedviews import TimeCursorController
from svs_core.gui.clientgui import ClientGUI



class TrackingGUI(ClientGUI):
	"""
	Interface for tracking client.

	The tracking client interface is similar to the basic SVS client.
	It has a basci command console and a separate graphic window which
	can run either within a window frame or fullscreen.
	"""
	def __init__(self, client):
		ClientGUI.__init__(self, client)
		self.gameModel = None
		self.gameWorldSetup = False
		#self.openView()

	def openView(self, fullscreen=False):
		"""
		Open visualisation view.  If 'fullscreen' is 'True' open
		as fullscreen display, otherwise in window.
		"""
		if not self.viswin:
			self.viswin = TrackerVisWindow(self, fullscreen)
		else:
			if not fullscreen:self.viswin.show()
			else:self.viswin.goFullscreen()
		if not self.gameWorldSetup:
			self.viswin.setupGameWorld(self.gameModel)
			self.viswin.setupScriptHistory(self.client.scriptHistory)
			self.viswin.setupScriptNetwork(self.client.scriptNetwork)
			self.viswin.setupMessageHistory(self.client.messageHistory)
			self.viswin.setupMessageNetwork(self.client.messageNetwork)
			self.gameWorldSetup = True

	def destroyView(self, args=None):
		"""
		Destroys visualisation view.
		"""
		ClientGUI.destroyView(self, args)
		print "destroyView, self.viswin:", self.viswin
		self.gameWorldSetup = False
		self.viswin = None

	def setupGameWorld(self, gameModel):
		"""
		Creates gameworld views from specified model.

		The model is received as a dictionary in the following form:

			- areas_x: number of areas on x axis
			- areas_y: number of areas on y axis
			- area_data: array of information about specific areas
		"""
		if not gameModel:return # make error
		self.gameModel = gameModel
		if not self.viswin:return
		self.viswin.setupGameWorld(gameModel)
		self.gameWorldSetup = True
		
	def updateGameWorld(self, updateData):
		"""
		Updates gameworld views from specified data.
		"""
		if not self.viswin:return
		self.viswin.updateGameWorld(updateData)
		#self.viswin.updateAnalysis()

	def update(self, timeElapsed):
		"""
		Responds to update call from server.
		"""
		if not self.viswin:return
		self.viswin.updateAnalysis(timeElapsed)

	#########################
	# SELECTION METHODS
	#########################
	def areaSelected(self, areaIdNum):
		"""
		Called by area view components when selected, passes info onto client.
		"""
		self.client.areaSelected(areaIdNum)

	def nodeSelected(self, node):
		"""
		Called by node view components when selected, passes info onto client.
		"""
		self.client.nodeSelected(node)


class TrackerVisWindow:
	def __init__(self, parent, fullscreen=False, width=600, height=300):
		self.parent = parent
		self.win = Toplevel()
		self.win.title('tracking')
		self.win.geometry("%dx%d+0+0" % (width, height))
		self.win.bind("<Escape>", self.parent.destroyView)
		self.win.protocol("WM_DELETE_WINDOW", self.parent.destroyView)
		self.win.config(bg='#000000')
		self.fullscreen = False
		self.width = width
		self.height = height
		self.gameRunning = False
		self.build()
		if fullscreen:self.goFullscreen()

	def goFullscreen(self):
		"""
		Make window fullscreen.
		"""
		if self.fullscreen:return
		if self.win.state() == 'normal':self.hide()
		w, h = self.win.winfo_screenwidth(), self.win.winfo_screenheight()
		self.win.overrideredirect(1)
		self.win.geometry("%dx%d+0+0" % (w, h))
		self.win.focus_set()
		self.fullscreen = True
		self.win.deiconify()

	def goWindowed(self):
		"""
		Turns fullscreen window into framed window.
		"""
		if not self.fullscreen:return
		if self.win.state() == 'normal':self.hide()
		self.win.overrideredirect(0)
		self.win.geometry("%dx%d+0+0" % (self.width, self.height))
		self.win.focus_set() # <-- move focus to this widget
		self.fullscreen = False
		self.win.deiconify()
	
	def show(self):
		"""
		Make window visible if hidden.
		"""
		if self.fullscreen:self.goWindowed()
		self.win.deiconify()

	def hide(self):
		"""
		Hide window.
		"""
		self.win.withdraw()

	def destroy(self, args=None):
		"""
		Destroy window.
		"""
		# need to check everything is reset so it
		# will start again properly
		self.win.destroy()

	def build(self):
		"""
		Create view components.
		"""
		w, h = self.win.winfo_screenwidth(), self.win.winfo_screenheight()
		self.gameFrame = Frame(self.win, bg=demo_const.BG_COLOUR)
		self.gameFrame.pack(side=TOP, expand=YES, fill=BOTH)
		## game logic view
		self.gameLogicView = GameLogicView(self.gameFrame, self)
		self.gameLogicView.frame.place(x=0, y=0, relwidth=0.25, relheight=0.25)
		## game view
		self.gameView = TrackerGameView(self.gameFrame, self)
		self.gameView.canvas.place(relx=0.25, y=0, relwidth=0.25, relheight=0.25)
		## script network view
		self.scriptNetworkView = NetworkView(self.gameFrame, self)
		self.scriptNetworkView.canvas.place(relx=0.5, y=0, relwidth=0.25, relheight=0.25)
		## message network view
		self.messageNetworkView = NetworkView(self.gameFrame, self)
		self.messageNetworkView.canvas.place(relx=0.75, y=0, relwidth=0.25, relheight=0.25)
		## script history view
		self.scriptView = ScriptHistoryView(self.gameFrame, self)
		self.scriptView.canvas.place(x=0, rely=0.25, relwidth=1.0, relheight=0.5)
		## message history view
		self.messageView = MessageHistoryView(self.gameFrame, self)
		self.messageView.canvas.place(x=0, rely=0.75, relwidth=1.0, relheight=0.25)
		## layout
		self.gameFrame.rowconfigure(0, weight=0)
		self.gameFrame.rowconfigure(1, weight=1)
		self.gameFrame.rowconfigure(2, weight=1)
		self.gameFrame.columnconfigure(0, weight=0)
		self.gameFrame.columnconfigure(1, weight=0)
		## time display
		self.scriptView.scrollPartner = self.messageView # tmp!
		self.messageView.scrollPartner = self.scriptView # tmp!
		self.timeCursorController = TimeCursorController()
		self.timeCursorController.addView(self.scriptView)
		self.timeCursorController.addView(self.messageView)
		self.timeCursorController.start()
		

	
	def setupGameWorld(self, gameModel):
		"""
		Creates gameworld views from specified model.

		The model is received as a dictionary in the following form:

			- areas_x: number of areas on x axis
			- areas_y: number of areas on y axis
			- area_data: array of information about specific areas
		"""
		terrainModel = gameModel.get(demo_const.TERRAIN_LABEL)
		self.gameView.setWorldDimensions(terrainModel[demo_const.DIM_X_LABEL], terrainModel[demo_const.DIM_Y_LABEL])
		self.gameView.setupTerrainAreas(terrainModel[demo_const.AREAS_X_LABEL], terrainModel[demo_const.AREAS_Y_LABEL], terrainModel[demo_const.AREA_DATA_LABEL])
		self.gameView.setupAgents(gameModel.get(demo_const.AGENTS_LABEL))
		self.gameView.canvasAdjusted()
		self.gameLogicView.setNameLabel(gameModel.get(demo_const.WORLD_NAME))
		self.handleGameRunningStatus(gameModel.get(demo_const.WORLD_RUNNING, False))

	def updateGameWorld(self, updateData):
		"""
		Updates gameworld views from specified data.
		"""
		if not updateData:return # don't bail
		terrainData = updateData.get(demo_const.TERRAIN_LABEL, None)
		if not terrainData:return # don't bail
		self.gameView.updateTerrainAreas(terrainData.get(demo_const.AREAS_LABEL, None))
		self.gameView.updateAgents(terrainData.get(demo_const.AGENTS_LABEL, None))
		running = updateData.get(demo_const.WORLD_RUNNING, None)
		if running != None:self.handleGameRunningStatus(running)

	def handleGameRunningStatus(self, running):
		"""
		Responds when the game changes between running and not running.
		"""
		if running:
			self.scriptView.unpauseTimeCursor()
			self.messageView.unpauseTimeCursor()
			self.gameLogicView.startGame()
		else:
			self.scriptView.pauseTimeCursor()
			self.messageView.pauseTimeCursor()
			self.gameLogicView.stopGame()
		self.gameRunning = running

	def updateAnalysis(self, timeElapsed):
		"""
		Updates analysis components and time cursor.
		"""
		self.gameLogicView.updateDisplay(self.timeCursorController.setCursorTime())


	def setupScriptHistory(self, dataSource):
		"""
		Creates view component for displaying script history.
		"""
		self.scriptView.setDataSource(dataSource)

	def setupMessageHistory(self, dataSource):
		"""
		Creates view component for displaying message history.
		"""
		self.messageView.setDataSource(dataSource)

	def setupScriptNetwork(self, dataSource):
		"""
		Creates view component for displaying script network.
		"""
		self.scriptNetworkView.setDataSource(dataSource)

	def setupMessageNetwork(self, dataSource):
		"""
		Creates view component for displaying message network.
		"""
		self.messageNetworkView.setDataSource(dataSource)

	#########################
	# SELECTION METHODS
	#########################
	def areaSelected(self, areaIdNum):
		"""
		Called by area view components when selected, passes info onto client.
		"""
		self.parent.areaSelected(areaIdNum)

	def agentSelected(self, agentIdNum):
		"""
		Called by agent view compoents when selected, passes info onto client.

		@todo: provide responding action.
		"""
		pass

	def getScriptForAgent(self, agentIdNum):
		"""
		Passes on request to client for script for agant.

		@todo: provide responding action.
		"""
		pass

	def nodeSelected(self, node):
		"""
		Called by node view components when selected, passes info onto client.
		"""
		self.parent.nodeSelected(node)

	def displayPosition(self, viewX, viewY):
		"""
		Displays gameworld coordinates for specified view location.

		Does nothing, provided for compatability.
		"""
		pass

	def syncSelection(self, selectedItem):
		"""
		Synchronises seelcted items across all views.
		"""
		self.scriptNetworkView.selectItem(selectedItem)
		self.messageNetworkView.selectItem(selectedItem)
		self.gameView.selectItem(selectedItem)
		self.scriptView.selectItem(selectedItem)
		self.messageView.selectItem(selectedItem)
		

		
