# svs_simulation.agents.basic_agents

#    Copyright (c) 2005 Simon Yuill.
#
#    This file is part of 'Social Versioning System' (SVS).
#
#    'Social Versioning System' is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation; either version 2 of the License, or
#    (at your option) any later version.
#
#    'Social Versioning System' is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with 'Social Versioning System'; if not, write to the Free Software
#    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

"""
Agents within simulation.

@author:	Simon Yuill
@copyright:	2005 Simon Yuill
@license:	GNU GPL version 2 or any later version
@contact:	simon@lipparosa.org
"""
# internal imports
from svs_simulation.entities.base_entities import SpatialEntity
from svs_simulation.ai_lib.statemachine import StateManager
from svs_simulation.ai_lib.tasks import TaskManager
from svs_simulation.numdata.geomlib import SphereBounds2D, angleFromAtoB
from svs_simulation.numdata.vectors import Vector2D, vec2DNormalize
from svs_simulation.utilities.constants import sim_const

# states
from svs_simulation.agents.agentstates.movement import *


#################################################
# state layers for parallel state machine
#################################################
movementStates = {
    "move":move,
    "rest": rest,
    "stop": rest
    }

#############################
# AGENT STATE MANAGER
#############################
class AgentStateManager(StateManager):
    """
    StateManager providing standard states for agent.
    """
    def __init__(self, agent):
        StateManager.__init__(self, agent)
        self.layers["movement"] = movementStates


#############################
# AGENT BASE CLASS
#############################
class SimAgentBaseClass(SpatialEntity):
	"""
	Base class with common properties an dmethids used by SimAgent and SimAgentProxy.
	"""
	def __init__(self):
		SpatialEntity.__init__(self)

	#############################
	# EVENTS
	#############################
	def handleSimEvent(self, event):
		"""
		Handles event from simulation.
		"""
		print event

	#############################
	# BASIC INFO
	#############################
	def getName(self):
		"""
		Returns name for structure.  If the structure
		has no name returns its idtag.
		"""
		if not self.name:return self._uid
		return self.name

	def getBoundsMinX(self):
		"""
		Returns minimum x coordinate from bounds.
		"""
		return self.bounds.x - self.bounds.radius

	def getBoundsMinY(self):
		"""
		Returns minimum y coordinate from bounds.
		"""
		return self.bounds.y - self.bounds.radius

	def getBoundsMaxX(self):
		"""
		Returns maximum x coordinate from bounds.
		"""
		return self.bounds.x + self.bounds.radius

	def getBoundsMaxY(self):
		"""
		Returns maximum y coordinate from bounds.
		"""
		return self.bounds.y + self.bounds.radius

	def containsPoint(self, x, y):
		"""
		Tests if coordinates are contaiend within structure bounds.
		"""
		return self.bounds.containsPointInWorld(x, y)

	def __str__(self):
		"""
		Returns string representation of object.
		"""
		if not self.name:return "agent [%s]" % self._uid
		return "agent [%s]" % self.name

	def getWorldbounds(self):
		"""
		Returns the bounds of the structure as mapped to the world coordinates.
		"""
		return self.bounds.worldbounds

	#############################
	# ARCHIVING
	#############################
	def setup(self, name=None):
		SpatialEntity.setup(self)
		self.name = name

	def decode(self, data):
		"""
		Applies encoded data to self.
		"""
		if data.has_key(sim_const.LABEL_BOUNDS):
			self.bounds = SphereBounds2D()
			self.bounds.decode(data[sim_const.LABEL_BOUNDS])
			self.location.x = self.bounds.worldbounds.x
			self.location.y = self.bounds.worldbounds.y
		if data.has_key(sim_const.LABEL_LOCATION):self.setLocation(data[sim_const.LABEL_LOCATION][0],data[sim_const.LABEL_LOCATION][1])
		if data.has_key(sim_const.LABEL_FACING):self.facing = data[sim_const.LABEL_FACING]
		name = data.get(sim_const.LABEL_NAME, None)
		self.setup(name=name)

	#############################
	# PROCESS FUNCTIONS
	#############################
	def updateWorld(self, simTime):
		"""
		Forwards C{update} call from world.
		"""
		print "SimAgent updateWorld"

	def startWorld(self, simTime):
		"""
		Forwards C{start} call from world.
		"""
		#print "SimAgent startWorld"
		pass

	def stopWorld(self, simTime):
		"""
		Forwards C{stop} call from world.
		"""
		#print "SimAgent stopWorld"
		pass

#############################
# BASIC AGENT
#############################
class SimAgent(SimAgentBaseClass):
	"""
	Basic class for agents with statemachine and task handling.
	"""
	def __init__(self):
		SimAgentBaseClass.__init__(self)
		self.stateManager = AgentStateManager(self)
		self.stateManager.setInitialState("stop", "movement")
		self.taskManager = TaskManager(self)

	def setTask(self, taskClass):
		"""
		Sets new task for agent.  This receives an 
		uniniated class variable which it instantiates
		before passing to L{SimAgent.taskManager}.
		"""
		self.taskManager.addSubtask(taskClass(self))

	#############################
	# EVENTS
	#############################
	def handleSimEvent(self, event):
		"""
		Handles event from simulation.
		"""
		print event


	#############################
	# ARCHIVING
	#############################
	def encode(self):
		"""
		Returns encoded representation of self.
		
		@rtype:dict
		"""
		data = {sim_const.LABEL_BOUNDS:self.bounds.encode()}
		data[sim_const.LABEL_FACING] = self.facing
		if self.name:data[sim_const.LABEL_NAME] = self.name
		if self._uid:data[sim_const.LABEL_IDTAG] = self._uid 
		#NOTE: self._uid is only decoded by proxy
		return data


	#############################
	# PROCESS FUNCTIONS
	#############################
	def updateWorld(self, simTime):
		"""
		Forwards C{update} call from world.
		"""
		print "SimAgent updateWorld"
		# tmp for test

	def startWorld(self, simTime):
		"""
		Forwards C{start} call from world.
		"""
		#print "SimAgent startWorld"
		pass

	def stopWorld(self, simTime):
		"""
		Forwards C{stop} call from world.
		"""
		#print "SimAgent stopWorld"
		pass

	#######################
	# STATES
	#######################
	def changeState(self, newState):
		"""
		Changes current state in stateManager.
		
		@type 	newState: string
		@param 	newState: new state to change to
		"""
		if not self.stateManager:return
		resultState = self.stateManager.setInput(newState)
		if resultState: 
			data = {sim_const.LABEL_STATECHANGE:resultState}
			self.reportChange(data)


#############################
# MOVING AGENT
#############################
class MovingAgent(SimAgent):
	"""
	Agents moveemnt capabilities.
	"""
	def __init__(self):
		SimAgent.__init__(self)
		self.location = Vector2D(0.0, 0.0)
		self.velocity = Vector2D(0.0, 0.0)
		self.heading = Vector2D(0.0, 0.0)
		self.side = Vector2D(0.0, 0.0)
		self.mass = 0.005
		self.dftMaxSpeed = 100.0
		self.maxSpeed = 100.0 # tmp value
		self.maxForce = 0.0
		self.maxTurnRate = 0.0
		self.threatRange = 100.0
		self.targetPoint = Vector2D(0.0, 0.0)
		self.steering = None
		self.moving = False
		# paths
		self.pathFollower = None

	#############################
	# PROCESS FUNCTIONS
	#############################
	def updateWorld(self, simTime):
		"""
		Forwards C{update} call from world.
		"""
		self.updateLocation(simTime[1])

	############################
	# MOVEMENT
	############################
	def setTargetPoint(self, x, y):
		"""
		Sets a target point for agent to move towards.
		"""
		self.targetPoint.x = x
		self.targetPoint.y = y

	def setSteeringBehaviour(self, steeringClass):
		"""
		Receives a class providing steering behaviour
		which is instantiated and applied to agent.
		"""
		self.steering = steeringClass(self)

	def updateLocation(self, timeElapsed):
		"""
		Updates agent location.
		"""
		if not self.moving:return
		prevLoc = self.location
		steeringForce = self.steering.calculate()
		acceleration = steeringForce/self.mass
		self.velocity += acceleration * timeElapsed
		self.velocity.truncate(self.maxSpeed)
		self.location += self.velocity * timeElapsed
		self.bounds.setLocation(self.location.x, self.location.y)
		# update the heading if the vehicle has a non zero velocity
		if self.velocity.lengthSq() > 0.00000001:
			self.heading = vec2DNormalize(self.velocity)
			self.side = self.heading.perp()
			self.facing = angleFromAtoB(0,0, self.heading.x, self.heading.y)
		# if following path, update this
		if self.pathFollower:self.pathFollower.updateRoute()
		data = {sim_const.LABEL_LOCATION:(self.location.x, self.location.y),
			sim_const.LABEL_FACING:self.facing}
		self.reportChange(data)

	def move(self):
		"""
		Starts movement.
		"""
		self.changeState("move")
		self.moving = True

	def halt(self):
		"""
		Stops movement.
		"""
		self.changeState("rest")
		#print "halted"


	############################
	# PATHS
	############################
	def followPath(self, path=None, name=None, start=None, destination=None, reverse=False):
		"""
		Sets path for entity to follow.
		"""
		#print "followPath"
		from svs_simulation.terrain.paths import PathFollower
		#print "self.partition:", self.partition
		if name:
			if not self.partition:pathObj= path
			else:
				pathObj = self.partition.terrain.getPath(name)
				if not pathObj:pathObj= path
		else:pathObj= path
		#print "pathObj:", pathObj
		if not pathObj:return # should be error
		if reverse:self.pathFollower = PathFollower(self, pathObj.reverse())
		else:self.pathFollower = PathFollower(self, pathObj)
		self.pathFollower.beginPath()

	def moveToNextPathNode(self, node):
		"""
		Starts agent moving towards specifed node on path.
		"""
		#print "moveToNextPathNode"
		self.move()
		self.steering.seekToPoint(node.x, node.y)
		
	def endPath(self):
		"""
		Called when end of path is reached.
		"""
		#print "MovingEntity endPath"
		self.halt()
		self.pathFollower = None

	def enterPathNode(self, node):
		"""
		Called when the entity enters a new path node.
		"""
		pass
		#print "enterPathNode:", node

	def exitPathNode(self, node):
		"""
		Called when the entity leaves a path node.
		"""
		pass
		#print "exitPathNode:", node
		

#############################
# PROXY AGENT
#############################
class SimAgentProxy(SimAgentBaseClass):
	"""
	Lightweight client-side representation of agent.
	"""
	def __init__(self):
		SimAgentBaseClass.__init__(self) 
		self.stateManager = {}
		self.sourceClass = None

	def decode(self, data):
		"""
		Applies encoded data to self.
		"""
		SimAgentBaseClass.decode(self, data)
		self._uid = data.get(sim_const.LABEL_IDTAG, None)
		properties = self.__dict__.keys()
		for key, value in data.items():
			if not key in properties:setattr(self,key,value)
		

	def decodeChangeData(self, data):
		"""
		Applies data to self.
		"""
		newLoc = data.get(sim_const.LABEL_LOCATION, None)
		if newLoc:self.bounds.setLocation(newLoc[0], newLoc[1])
		newFacing = data.get(sim_const.LABEL_FACING, None)
		if newFacing:self.facing = newFacing
