# svs_simulation.ai_lib.bdt

#    Copyright (c) 2005 Simon Yuill.
#
#    This file is part of 'Social Versioning System' (SVS).
#
#    'Social Versioning System' is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation; either version 2 of the License, or
#    (at your option) any later version.
#
#    'Social Versioning System' is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with 'Social Versioning System'; if not, write to the Free Software
#    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

"""
Binary decision tree.

@author:	Simon Yuill
@copyright:	2005 Simon Yuill
@license:	GNU GPL version 2 or any later version
@contact:	simon@lipparosa.org
"""


class BDTNode:
	"""
	A generic node within a binary decision tree.
	"""
	def __init__(self):
		self.decision = False
		self.no_node = None
		self.yes_node = None
		self._externalRule = False

	def evaluate(self, inputData, performActions = True):
		"""
		Evaluates the input data.

		If the evaluation is positive, passes the data on
		to the C{yes_node}, otherwise to the C{no_node}.
		"""
		if not self._externalRule:result = self.rule(inputData)
		else:result = self._rule(self, inputData)
		if result[0]:self.yes_node.evaluate(result[1], performActions)
		else:self.no_node.evaluate(result[1], performActions)


	def rule(self, inputData):
		"""
		Rule used to perform evaluation.

		This should be overridden by implementing classes.
		
		@rtype: boolean
		"""
		return self.decision, inputData

	def setRule(self, ruleFunction):
		self._rule = ruleFunction
		self._externalRule = True
		

class BDTTerminal:
	"""
	A generic terminal node on a binary decision tree.

	This represents the end of a decision chain, which 
	should normally result in some action being performed.
	"""
	def __init__(self):
		self._externalAction = False

	def evaluate(self, inputData, performActions = True):
		"""
		Performs action on the input data.
		"""
		if not performActions:return
		if self._externalAction:self._action(self, inputData)
		else:self.action(inputData)

	def action(self, inputData):
		"""
		Defines action performed by node.
		"""
		pass

	def setAction(self, actionFunction):
		"""
		Defines external action performed by node.
		"""
		self._action = actionFunction
		self._externalAction = True

class BDT:
	"""
	Binary Decision Tree.
	"""
	def __init__(self):
		self.root = None

	def evaluate(self, inputData, performActions = True):
		"""
		Evaluates the input data, starting from the root node of tree.
		"""
		if not self.root:return
		self.root.evaluate(inputData, performActions)
