# svs_simulation.ai_lib.fsm

#    Copyright (c) 2005 Simon Yuill.
#
#    This file is part of 'Social Versioning System' (SVS).
#
#    'Social Versioning System' is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation; either version 2 of the License, or
#    (at your option) any later version.
#
#    'Social Versioning System' is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with 'Social Versioning System'; if not, write to the Free Software
#    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

"""
Generic classes for Finite State Machines.

@author:	Simon Yuill
@copyright:	2005 Simon Yuill
@license:	GNU GPL version 2 or any later version
@contact:	simon@lipparosa.org
"""
# internal imports
from svs_core.utilities.lib import Enumeration


#############################
# CONSTANTS
#############################
fsm_state_types = Enumeration('goal_state',['FSM_STATE_NONE', 'FSM_STATE_APPROACH', 'FSM_STATE_ATTACK', 'FSM_STATE_EVADE', 'FSM_STATE_GETPOWERUP', 'FSM_STATE_IDLE', 'FSM_STATE_COUNT'])

fsm_machine_types = Enumeration('goal_state',['FSM_MACH_NONE', 'FSM_MACH_MAINSHIP', 'FSM_MACH_COUNT'])



#############################
# CLASSES
#############################
class FSMState:
	"""
	Single state within a FSM.
	"""
	def __init__(self, parent, fsmType=fsm_state_types.FSM_STATE_NONE):
		self.parent = parent
		self.fsmType = fsmType

	def exit(self):
		pass

	def update(self, dt):
		pass

	def initState(self):
		pass

	def checkTransitions(self):
		return FSM_STATE_NONE
	

class FSMMachine(FSMState):
	def __init__(self, parent, fsmType=fsm_state_types.FSM_STATE_NONE):
		self.parent = parent
		self.fsmType = fsmType
		self.defaultState = None
		self.goalState = None
		self.currentState = None
		self.goalID = None
		self.states = []

	def updateMachine(self, dt):
		# don't do anything if you have no states
		if not self.states:return
		# don't do anything if there's no current 
		# state, and no default state
		if not self.currentState:self.currentState = self.defaultState
		if not self.currentState:return
		# check for transitions, and then update
		oldStateID = self.currentState.fsmType
		self.goalID = self.currentState.checkTransitions()
		# switch if there was a transition
		if self.goalID != oldStateID:
			if self.goalID:
				self.currentState.exit()
				self.currentState = self.goalState
				self.currentState.enter()
		self.currentState.update(dt)

	def addState(self, state):
		self.states.append(state)

	def setDefaultState(self, state):
		self.defaultState = state

	def setGoalID(self, goal):
		self.goalID = goal

	def transitionState(self, goal):
		# don't do anything if you have no states
		if not self.states:return false
		# determine if we have state of type 'goal'
		# in the list, and switch to it, otherwise, quit out
		for state in self.states:
			if state.fsmType == goal:
				self.goalState = state
				return True
		return False

	def reset(self):
		self.exit()
		if(self.currentState):
			self.currentState.exit()
			self.currentState = self.defaultState
		# init all the states
		for state in self.states:
			state.initState()
		# and now enter the self.defaultState, if any
		if(self.currentState):self.currentState.enter()

