# svs_simulation.ai_lib.fuzzy

#    Copyright (c) 2005 Simon Yuill.
#
#    This file is part of 'Social Versioning System' (SVS).
#
#    'Social Versioning System' is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation; either version 2 of the License, or
#    (at your option) any later version.
#
#    'Social Versioning System' is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with 'Social Versioning System'; if not, write to the Free Software
#    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

"""
Implementation of fuzzy logic system.

The evaluation method of the L{FuzzyVariable} is based on the Seguno
method.  This requires that each subset be represented as a 'fuzzy
singleton' (ie a single value rather than a range).  When subsets
are created, a C{segunoSingleton} property is created for it based
on the centroid of the fitness vector.

@author:	Simon Yuill
@copyright:	2005 Simon Yuill
@license:	GNU GPL version 2 or any later version
@contact:	simon@lipparosa.org
"""
# internal imports
from svs_core.utilities.constants import Constants


#######################
# CONSTANTS
#######################
logic_const = Constants()

# operands
logic_const.NO_OP = 200
logic_const.AND = 201
logic_const.OR = 202
logic_const.NOT = 202


#######################
# EXCEPTIONS
#######################
class FuzzyException(Exception):
	pass

#######################
# FUZZY EVALUATOR
#######################
class SimpleFuzzyEvaluator:
	"""
	The C{SimpleFuzzyEvaluator} patches two fuzzy sets together
	and evaluates their input.
	"""
	def __init__(self, setA, setB):
		self.setA = setA
		self.setB = setB
		self.patches = {}

	def makePatch(self, subsetA, subsetB):
		"""
		Creates a patch between subsets within the two sets.
		"""
		self.patches[subsetA] = self.setB[subsetB]

	def evaluate(self, inputvalue, threshold=0.0):
		results = self.setA.valuesFor(inputvalue)
		dividend = 0.0
		divisor = 0.0
		for res in results:
			outputSet = self.patches.get(res[0], None)
			if not outputSet:continue
			dividend += res[1] * outputSet.segunoSingleton
			divisor += res[1]
		if divisor == 0.0: return 0.0
		return dividend / divisor


class FuzzyEvaluator:
	"""
	Collects a set of rules and handles their evaluation of input.
	"""
	def __init__(self, name):
		self.name = name
		self.rules = []
		self.fuzzyVariables = {}

	def addRule(self, antecedent, consequent):
		"""
		Adds rule to evaluator.
		"""
		self.rules.append(FuzzyRule(antecedent, consequent))

	def addFuzzyVariable(self, fuzzyVariable):
		if not self.fuzzyVariables.has_key(fuzzyVariable.name):
			self.fuzzyVariables[fuzzyVariable.name] = fuzzyVariable

	def fuzzify(self, **inputValues):
		"""
		Applies input values to fuzzy variables held by evaluator.
		"""
		for key, value in inputValues:
			fuzzyVariable = self.fuzzyVariables.get(key, None)
			if fuzzyVariable:fuzzyVariable.setInputValue(value)

	def defuzzify(self, outputVariableName):
		"""
		Evaluates C{outputVariable} based on previous input.
		"""
		outputVariable =  self.fuzzyVariables.get(outputVariableName, None)
		if not outputVariable:raise FuzzyException("no variable with name <%s>", outputVariableName)
		for fuzzyVariable in self.fuzzyVariables:fuzzyVariable.reset()
		for rule in self.rules:rule.evaluate()
		return outputVariable.defuzzify()
		
		


#######################
# FUZZY RULE
#######################
class FuzzyRule:
	"""
	Respresents and evaluates a single fuzzy rule.
	"""
	def __init__(self, antecedent, consequent):
		self.antecedent = antecedent
		self.consequent = consequent
		self.outputValue = 0.0

	def evaluate(self):
		"""
		Evaluates rule.
		"""
		self.outputValue = self.antecedent.evaluateDOM()
		self.consequent.orDOM(self.outputValue)

	def getResult(self):
		"""
		Resturns a tuple combining the consequent with the 
		output value of the rule's evaluation.
		"""
		return (self.consequent, self.outputValue)

		

############################
# FUZZY LINGUISTIC VARIABLE
############################
class FuzzyVariable:
	"""
	Encapsulation of a fuzzy super set.
	"""
	def __init__(self, name, minValue=0, maxValue=100, unit=1):
		self.name = name
		self.minValue = minValue
		self.maxValue = maxValue
		self.inputValue = self.minValue
		self.unit = unit
		self.indices = range(self.minValue, self.maxValue + self.unit, self.unit)
		self.fuzzySets = {}
		
	def addSubSet(self, setName, fitVector):
		"""
		Adds new subset.
		
		@type	setName: string
		@param	setName: name for set
		@type	fitVector: list
		@param	fitVector: profile of values for set
		"""
		newSet = FuzzySet(setName, fitVector, self)
		buildLookupForSet(newSet)
		newSet.segunoSingleton = nearestIndex(centroid(newSet), self.indices)
		self.fuzzySets[setName] = newSet

	def __getitem__(self, item):
		"""
		Returns subset with name matching item.
		
		@type	item: string
		@param	item: name for set
		"""
		return self.fuzzySets.get(item, None)

	def setInputValue(self, inputValue):
		"""
		Sets new input value for variable, and sets all
		member sets DOM to 0.0.
		"""
		if inputValue < self.minValue:inputValue = self.minValue
		if inputValue > self.maxValue:inputValue = self.maxValue

	def reset(self):
		"""
		Sets set memenrs DOM to 0.0.
		"""
		self.inputValue = inputValue
		for set in self.fuzzySets:set.clearDOM()

	def defuzzify(self):
		"""
		Returns infered result of fuzzy evaluation.
		"""
		preferredSet = None
		maxValue = 0.0
		for set in self.fuzzySets:
			dividend += set.domvalue * set.segunoSingleton
			divisor += set.domvalue
			if set.domvalue > maxValue:preferredSet = set
		if divisor == 0.0: return 0.0
		crispValue = dividend / divisor
		return (preferredSet, crispValue)
		
	def valuesFor(self, index):
		"""
		Returns list of membership values for 
		each of the subsets within this set.
		
		@type	index: float or integer
		@param	index: value to test in sets
		"""
		result = []
		for name, set in self.fuzzySets.items():
			result.append((name, set.evaluateDOM(index)))
		return result
		
	def valueIs(self, index, threshold=0.0):
		"""
		Returns list of sets in which 
		the tested value has a member value
		greater then the threshold.
		
		@type	index: float or integer
		@param	index: value to test in sets
		@type	threshold: float
		@param	threshold: threshold for inclusion
		"""
		result = []
		for name, set in self.fuzzySets.items():
			value = set.evaluateDOM(index)
			if value >= threshold:
				result.append((name, value))
		return result
		
	def valueInSet(self, index, setName):
		"""
		Returns the membership value in a specific set.
		
		@type	index: float or integer
		@param	index: value to test in sets
		@type	setName: string
		@param	setName: name of set to inspect
		"""
		set = self.fuzzySets.get(setName, None)
		if not set:
			return 0.0
		return set.evaluateDOM(index)
		
	def getSegunoConstantForSubSet(self, setName):
		"""
		Returns the Seguno constant for a specific set.
		
		@type	setName: string
		@param	setName: name of set to inspect
		"""
		set = self.fuzzySets.get(setName, None)
		if not set:
			return 0
		return set.segunoSingleton

	def segunoValueIs(self, index, threshold=0.0):
		"""
		Returns result of L{valueIs} operation
		as the Seguno value.
		
		@type	index: float or integer
		@param	index: value to test in sets
		@type	threshold: float
		@param	threshold: threshold for inclusion
		"""
		result = self.valueIs(index, threshold)
		return self.getSegunoConstantForSubSet(result[0])
		

###########################
# FUZZY TERMS
###########################
class FuzzyTerm:
	"""
	Base class from which fuzzy sets and operators are derived.
	"""
	def __init__(self):
		self.domvalue = 0.0

	def evaluateDOM(self):
		"""
		Returns degree of membership (DOM) value.
		"""
		return self.domvalue

	def orDOM(self, inputValue):
		"""
		Sets DOM by ORing input value against current value.
		"""
		self.domvalue = max(self.domvalue, inputValue)

	def clearDOM(self):
		"""
		Clears degree of membership (DOM) value.
		"""
		self.domvalue = 0.0

	def addToEvalutor(self, evaluator):
		"""
		Called when term is added to a fuzzy evaluator.
		"""
		pass


class FuzzySet(FuzzyTerm):
	"""
	A set contained within a fuzzy linguistic variable.
	
	The fit vector for the set is defined as a list of tuples.
	The first item of the tuple relates to an index point
	within the range of the super set and the second item in
	the tuple defines the degree of membership for the set
	at that index point.
	"""
	def __init__(self, name, fitVector, fuzzyVariable):
		FuzzyTerm.__init__(self)
		self.name = name
		self.fitVector = fitVector
		self.fuzzyVariable = fuzzyVariable
		self.table = []
		self.segunoSingleton = 0
		
	def setValueAt(self, index, value):
		"""
		Sets index point in lookup table to specified value.
		
		@type	index: integer
		@param	index: index point in lookup table
		@type	value: float
		@param	value: new value to insert
		"""
		# clip value to range 0 - 1
		if value < 0.0:
			value = 0.0
		elif value > 1.0:
			value = 1.0
		
		luIdx = positionOf(index, self.fuzzyVariable.indices)
		if luIdx:
			self.table[luIdx] = value
			
	def evaluateDOM(self):
		"""
		Returns membership value.
		"""
		luIdx = positionOf(self.fuzzyVariable.inputValue, self.fuzzyVariable.indices)
		if luIdx:self.domvalue = self.table[luIdx]
		else:self.domvalue = 0.0
		return self.domvalue

	def addToEvalutor(self, evaluator):
		"""
		Called when term is added to a fuzzy evaluator.
		"""
		evaluator.addFuzzyVariable(self.fuzzyVariable)


class FuzzyAND(FuzzyTerm):
	"""
	Performs fuzzy AND operation on member terms.
	"""
	def __init__(self, *terms):
		FuzzyTerm.__init__(self)
		self.terms = terms

	def evaluateDOM(self):
		"""
		Returns degree of membership (DOM) value.
		"""
		self.domvalue = 1.0
		for term in self.terms:
			result = term.evaluateDOM()
			if result < self.domvalue:self.domvalue = result
		return self.domvalue

	def clearDOM(self):
		"""
		Clears degree of membership (DOM) value.
		"""
		self.domvalue = 0.0
		for term in self.terms:term.clearDOM()

	def addToEvalutor(self, evaluator):
		"""
		Called when term is added to a fuzzy evaluator.
		"""
		for term in self.terms:term.addToEvalutor(evaluator)


class FuzzyOR(FuzzyTerm):
	"""
	Performs fuzzy OR operation on member terms.
	"""
	def __init__(self, *terms):
		FuzzyTerm.__init__(self)
		self.terms = terms

	def evaluateDOM(self):
		"""
		Returns degree of membership (DOM) value.
		"""
		self.domvalue = 0.0
		for term in self.terms:
			result = term.evaluateDOM()
			if result > self.domvalue:self.domvalue = result
		return self.domvalue

	def clearDOM(self):
		"""
		Clears degree of membership (DOM) value.
		"""
		self.domvalue = 0.0
		for term in self.terms:term.clearDOM()

	def addToEvalutor(self, evaluator):
		"""
		Called when term is added to a fuzzy evaluator.
		"""
		for term in self.terms:term.addToEvalutor(evaluator)


class FuzzyNOT(FuzzyTerm):
	"""
	Performs fuzzy NOT operation on member terms.
	"""
	def __init__(self, term):
		FuzzyTerm.__init__(self)
		self.term = term

	def evaluateDOM(self):
		"""
		Returns degree of membership (DOM) value.
		"""
		self.domvalue = 1.0 - term.evaluateDOM()
		return self.domvalue

	def clearDOM(self):
		"""
		Clears degree of membership (DOM) value.
		"""
		self.domvalue = 0.0
		term.clearDOM()

	def addToEvalutor(self, evaluator):
		"""
		Called when term is added to a fuzzy evaluator.
		"""
		term.addToEvalutor(evaluator)

	
#######################
# UTILITY FUNCTIONS
#######################
def buildLookupForSet(fuzzySet):
	"""
	Creates a lookup table for a set.
	
	@type	fuzzySet: L{FuzzySet}
	@param	fuzzySet: set to build lookup table for
	"""
	# create empty table
	fuzzySet.table = []
	# flatten table to all zeros
	for index in fuzzySet.fuzzyVariable.indices:
		fuzzySet.table.append(0.0)
	# apply fit vector values
	itemA = (fuzzySet.fuzzyVariable.indices[0], 0.0)
	for item in fuzzySet.fitVector:
		mapIndices(itemA, item, fuzzySet)
		itemA = item
	itemB = (fuzzySet.fuzzyVariable.indices[-1], 0.0)
	mapIndices(itemA, itemB, fuzzySet)
	
def centroid(fuzzySet):
	"""
	Finds the centre ofgravity for the set.
	
	@type	fuzzySet: L{FuzzySet}
	@param	fuzzySet: set to build lookup table for
	"""
	table = fuzzySet.table
	indices = fuzzySet.fuzzyVariable.indices
	dividend = 0.0
	divisor = 0.0
	
	for value, index in zip(table, indices):
		dividend += value * index
		divisor += value
		
	return dividend / divisor
	
	
		
def mapIndices(itemA, itemB, fuzzySet):
	"""
	Maps the values between two items
	into the specified set according.
	
	@type	itemA: tuple
	@param	itemA: first item
	@type	itemB: tuple
	@param	itemB: second item
	@type	fuzzySet: L{FuzzySet}
	@param	fuzzySet: set to map to
	"""
	indexA = nearestIndex(itemA[0], fuzzySet.fuzzyVariable.indices)
	indexB = nearestIndex(itemB[0], fuzzySet.fuzzyVariable.indices)
	valA = itemA[1]
	valB = itemB[1]
	unit = fuzzySet.fuzzyVariable.unit
	
	# interpolate
	if indexA == indexB:
		fuzzySet.setValueAt(indexB, valB)
	elif valA == valB:
		for index in range(indexA, indexB + unit, unit):
			fuzzySet.setValueAt(index, valB)
	else:
		max = (indexB - indexA) * 1.0
		for index in range(indexA, indexB + unit, unit):
			t = (index - indexA) / max 
			n = valA + (t * (valB - valA))
			fuzzySet.setValueAt(index, n)
	
def nearestIndex(testIndex, indices):
	"""
	Finds nearest match for an index
	within a range of indices.
	
	@type	testIndex: integer
	@param	testIndex: index number to match
	@type	indices: list
	@param	indices: list to match to
	"""
	below = 0
	above = 0
	
	for index in indices:
		if index == testIndex:
			return testIndex
		elif index < testIndex:
			below = index
		elif index > testIndex:
			above = index
			break
			
	if (testIndex - below) < ((above - below) / 2):
		return below
	return above
			
def positionOf(index, indices):
	"""
	Returns position of index value in indices.
	"""
	index = nearestIndex(index, indices)
	for pos, value in enumerate(indices):
		if value == index:return pos
	return None
	
def parseFitVectorString(fvStr):
	"""
	Converts a formatted string into a fitvector.
	
	The string should be in the form:
		
		C{<index>:<membership>, <index>:<membership>, etc}
		
	where each entry is separated by a comma, and the 
	index point and membership value are separated by
	a colon.
	
	"""
	import string
	
	fitvector = []
	
	totalChars = len(fvStr)
	i = 0
	tmpStr =''
	index = 0
	membership = 0.0
	
	while(i < totalChars):
		#print "fvStr[i] %s" % fvStr[i]
		if fvStr[i] == ":":
			#print ": tmpStr %s" % tmpStr
			index = int(string.strip(tmpStr))
			tmpStr =''
		elif fvStr[i] == ",":
			#print ", tmpStr %s" % tmpStr
			membership = float(string.strip(tmpStr))
			tmpStr =''
			fitvector.append((index, membership))
		else:
			tmpStr += fvStr[i]
		i += 1
		
	return fitvector

